package ru.yandex.avia.booking.partners.gateways.aeroflot.converter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.avia.booking.enums.ClassOfService;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotAnonymousTraveller;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCategoryOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOriginDestination;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotPriceDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegmentNode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.model.booking.CodeContext;
import ru.yandex.avia.booking.partners.gateways.model.search.CategoryPrice;
import ru.yandex.avia.booking.partners.gateways.model.search.FareInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Flight;
import ru.yandex.avia.booking.partners.gateways.model.search.PriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Segment;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.avia.booking.partners.gateways.aeroflot.converter.AeroflotCodesMapping.classOfServiceEnum;

public class AeroflotVariantConverter {
    private static final String COUNTRY_FALLBACK_CODE = "RU";

    public static Variant convertVariant(AeroflotVariant variant) {
        String country = variant.getContext().getCountryCode();
        return Variant.builder()
                .codeContext(CodeContext.IATA)
                .segments(convertLegs(variant))
                .priceInfo(convertPriceInfo(variant.getOffer(), variant.getTravellers()))
                .allTariffs(variant.getAllTariffs().stream()
                        .map(altOffer -> convertPriceInfo(altOffer, variant.getTravellers()))
                        .collect(toList()))
                .searchData(variant.getSearchData())
                // backward compatibility
                .countryOfSale(!Strings.isNullOrEmpty(country) ? country : COUNTRY_FALLBACK_CODE)
                .build();
    }

    private static List<Segment> convertLegs(AeroflotVariant variant) {
        ClassOfService baseClass = classOfServiceEnum(variant.getContext().getCabinType());
        List<Segment> yaLegs = new ArrayList<>(); // => Aeroflot Flights
        for (AeroflotOriginDestination od : variant.getOriginDestinations()) {
            List<Flight> yaSegments = new ArrayList<>(); // => Aeroflot Segments
            for (AeroflotSegment segment : variant.getOriginDestinationSegments(od.getId())) {
                yaSegments.add(convertSegment(segment, baseClass));
            }
            Preconditions.checkArgument(!yaSegments.isEmpty(),
                    "No segments for the leg: od=%s, variant=%s", od, variant);
            yaLegs.add(Segment.builder()
                    .flights(yaSegments)
                    .build());
        }
        Preconditions.checkArgument(!yaLegs.isEmpty(), "No segments: variant=%s", variant);
        return yaLegs;
    }

    private static Flight convertSegment(AeroflotSegment segment, ClassOfService baseClass) {
        AeroflotSegmentNode dep = segment.getDeparture();
        AeroflotSegmentNode arr = segment.getArrival();
        return Flight.builder()
                .id(segment.getId())
                .depCode(dep.getAirportCode())
                .arrCode(arr.getAirportCode())
                .marketingAirlineCode(segment.getMarketingCarrier().getAirlineId())
                .operatingAirlineCode(segment.getOperatingCarrier().getAirlineId())
                .flightNumber(segment.getMarketingCarrier().getFlightNumber())
                .departureDateTime(dep.getDateTime())
                .arrivalDateTime(arr.getDateTime())
                .baseClass(baseClass)
                .flightStops(Collections.emptyList())
                .build();
    }

    public static PriceInfo convertPriceInfo(AeroflotTotalOffer offer, List<AeroflotAnonymousTraveller> travellers) {

        Map<String, AeroflotAnonymousTraveller> travellersMap = travellers.stream()
                .collect(toMap(AeroflotAnonymousTraveller::getId, t -> t));
        List<CategoryPrice> categoryPrices = offer.getCategoryOffers()
                .stream()
                .map(categoryOffer -> {
                    AeroflotAnonymousTraveller traveller = travellersMap.get(categoryOffer.getTravellerId());
                    return convertCategoryOffer(offer, categoryOffer, traveller);
                })
                .collect(toList());
        Money totalPrice = categoryPrices.stream()
                .map(cat -> cat.getTotal().multiply(cat.getQuantity()))
                .reduce(Money::add)
                .orElseThrow(() -> new RuntimeException("No category offers found: offer=" + offer));
        return PriceInfo.builder()
                .id(offer.getId())
                .total(totalPrice)
                .categoryPrices(categoryPrices)
                .build();
    }

    private static CategoryPrice convertCategoryOffer(AeroflotTotalOffer offer, AeroflotCategoryOffer categoryOffer,
                                                      AeroflotAnonymousTraveller traveller) {
        AeroflotPriceDetail offerPrice = categoryOffer.getTotalPrice();
        return CategoryPrice.builder()
                .passengerCategory(AeroflotCodesMapping.categoryEnum(traveller.getCategory()))
                .quantity(traveller.getQuantity())
                .total(offerPrice.getTotalPrice())
                .fare(offerPrice.getBasePrice())
                .taxes(offerPrice.getTaxes())
                .fareInfo(categoryOffer.getFareBasisCodes().stream()
                        .map(fbCode -> FareInfo.builder()
                                .fareBasis(fbCode.getFareCode())
                                .fareFamilyCode(offer.getSegmentRef(fbCode.getSegmentId()).getClassOfServiceCode())
                                .flightId(fbCode.getSegmentId())
                                .build())
                        .collect(toList()))
                .build();
    }
}
