package ru.yandex.avia.booking.partners.gateways.aeroflot.demo;

import java.io.IOException;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Dsl;
import org.asynchttpclient.Param;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;

import static java.util.Arrays.asList;

@Slf4j
public class AeroflotDemoCardTokenizer implements AutoCloseable {
    public static final String TOKENIZATION_ENDPOINT = "https://pay.test.aeroflot.ru/test-rc/aeropayment/api/paymentToken";

    public static final TokenizeCardForm demoCardFormWith3ds = TokenizeCardForm.builder()
            .card("4111111111111111")
            .holder("test testov")
            .expires("202412")
            .cvc("123")
            // another 3ds card, doesn't work at the moment
            //.card("639002000000000003")
            .build();
    public static final TokenizeCardForm demoCardFormNo3ds = TokenizeCardForm.builder()
            .card("4154810047474554")
            .holder("test testov")
            .expires("202412")
            .cvc("314")
            // another no-3ds card allowed in Trust testing:
            //.card("5555555555555599")
            //.holder("test testov")
            //.expires("202412")
            //.cvc("123")
            .build();

    private final ObjectMapper json;
    private final String endpoint;
    private final AsyncHttpClient httpClient;

    private AeroflotDemoCardTokenizer(ObjectMapper json, String endpoint) {
        this.json = json;
        this.endpoint = endpoint;
        this.httpClient = Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("tokenizerAhcPool")
                .setPooledConnectionIdleTimeout(100)
                .build());
    }

    public AeroflotDemoCardTokenizer() {
        this(new ObjectMapper(), TOKENIZATION_ENDPOINT);
    }

    public String sendTokenizeRequest(TokenizeCardForm form) {
        try {
            log.debug("Sending card tokenization request to {}", endpoint);
            Request req = Dsl.post(endpoint)
                    .setHeader("Content-Type", "application/x-www-form-urlencoded")
                    .setFormParams(asList(
                            new Param("pan", form.getCard()),
                            new Param("cardholderName", form.getHolder()),
                            new Param("expDate", form.getExpires()),
                            new Param("cvc", form.getCvc())))
                    .build();
            Response resp = httpClient.executeRequest(req).get();
            // OK : {
            //      "errorCode":"0",
            //      "token":"f1ba3788-2636-4d21-9c21-ef9aba4de964",
            //      "tokenExpDate":"2018-12-27T11:27:05.197+0000"}
            // FAILED : {
            //      "errorCode":"2",
            //      "errorMessage":"pan - wrong.value; pan - 12 to 19 digits expected"}
            JsonNode data = json.readTree(resp.getResponseBody());
            Preconditions.checkArgument("0".equals(data.get("errorCode").textValue()));
            return data.get("token").textValue();
        } catch (Exception e) {
            log.warn("Failed to tokenize the card: holder={}, e.msg={}", form.getHolder(), e.getMessage());
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws IOException {
        httpClient.close();
    }

    public String tokenizeDemoCardWith3ds() {
        return sendTokenizeRequest(demoCardFormWith3ds);
    }

    public String tokenizeDemoCardNo3ds() {
        return sendTokenizeRequest(demoCardFormNo3ds);
    }

    public static String tokenizeDemoCard(boolean with3ds) {
        try (AeroflotDemoCardTokenizer tokenizer = new AeroflotDemoCardTokenizer()) {
            return with3ds ?
                    tokenizer.tokenizeDemoCardWith3ds() :
                    tokenizer.tokenizeDemoCardNo3ds();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
