package ru.yandex.avia.booking.partners.gateways.aeroflot.model;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

import ru.yandex.travel.commons.enums.MultiEnumIndex;
import ru.yandex.travel.commons.enums.ValueBasedEnum;

@RequiredArgsConstructor
@Getter
public enum AeroflotOrderSubStatus implements ValueBasedEnum<String> {
    /**
     * Заказ оплачен. Билеты выпущены.
     * Если билеты в ответе сервиса получены, то дальнейшая обработка не требуется.
     * Иначе для получения билетов вызвать IATA_OrderRetrieveRQ не ранее чем через 5 минут после вызова сервиса,
     * в рамках которого происходил выпуск документов.
     */
    FULLY_PAID("100"),
    /**
     * Заказ оплачен, билеты не выпущены.
     * КС должна сообщить Клиенту о том, что выпуск билетов произойдет в течение
     * 3х часов. Билеты будут выпущены вручную сотрудниками контакт-центра ПАО «Аэрофлот».
     * Если КС для внутренних процессов необходимо получить итоговый статус билетов, то в этом случае рекомендуется
     * производить запрос IATA_OrderRetrieveRQ не ранее, чем через 3 часа после вызова сервиса, в рамках которого
     * происходил выпуск документов.
     */
    PAID_NOT_TICKETED("150"),
    /**
     * Заказ не может быть оплачен.
     * Дальнейшие действия не требуются (через NDC данный заказ оплатить не удастся).
     */
    CANT_BE_PAID("500"),
    /**
     * Заказ не может быть оплачен.
     * Время возможности оплаты истекло.
     * Дальнейшие действия не требуются (через NDC данный заказ оплатить не удастся).
     */
    PAYMENT_EXPIRED("510"),
    /**
     * Заказ не может быть оплачен. Заказ отменен.
     */
    PAYMENT_EXPIRED_2("590"),
    /**
     * Платёж отклонён, т.к. один из пассажиров внесен в Реестр лиц, воздушная перевозка которых ограничена ПАО
     * «Аэрофлот» (список физических лиц, которым ПАО «Аэрофлот» отказано в заключении договора воздушной перевозки
     * в соответствии со ст. 107.1 Воздушного кодекса РФ)
     * Дальнейшие действия не требуются.
     */
    PASSENGER_REJECTED("580"),
    /**
     * Платёж отклонён банком либо платёжной системой.
     */
    CARD_REJECTED_620("620"),
    /**
     * Заказ не оплачен.
     * Требуется авторизация по 3-D Secure / Protect Buy.
     * Необходимо авторизоваться, завершить оплату, затем для получения статуса оплаты и документов
     * вызывать IATA_OrderRetrieveRQ.
     */
    REQUIRES_3DS("630"),
    /**
     * Платёж отклонён из-за неверной даты окончания действия карты.
     */
    CARD_REJECTED_643("643"),
    /**
     * Платёж отклонён из-за неверной даты CVC карты.
     */
    CARD_REJECTED_644("644"),
    /**
     * Платёж отклонён из-за неверного номера карты (или оплата картой данной платёжной системой не поддерживается).
     */
    CARD_REJECTED_650("650"),
    /**
     * Внутренняя ошибка (для IATA_OrderCreateRQ, IATA_OrderChangeRQ, IATA_OrderCancelRQ, IATA_AirDocIssueRQ).
     * Осуществить вызов IATA_AirDocIssueRQ один раз не менее чем через 5 минут после вызова сервиса,
     * который вернул данный статус.
     */
    INTERNAL_ERROR("750");

    private final String value;

    public static AeroflotOrderSubStatus forValue(String value) {
        return MultiEnumIndex.getGlobal(AeroflotOrderSubStatus.class).getByValue(value);
    }

    public boolean isCardRejectedError() {
        return getValue() == CARD_REJECTED_620.getValue() ||
                getValue() == CARD_REJECTED_643.getValue() ||
                getValue() == CARD_REJECTED_644.getValue() ||
                getValue() == CARD_REJECTED_650.getValue();
    }
}
