package ru.yandex.avia.booking.partners.gateways.aeroflot.model;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.experimental.Accessors;
import org.javamoney.moneta.Money;

import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.ff.model.SegmentFare;
import ru.yandex.avia.booking.partners.gateways.aeroflot.converter.AeroflotCodesMapping;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingFailureReason;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingPriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.ClientInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.PassengerPricePart;
import ru.yandex.avia.booking.partners.gateways.model.booking.ServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.avia.booking.promo.AviaPromoCampaignsInfo;

import static java.util.stream.Collectors.toList;

@Data
@Builder(toBuilder = true)
@Accessors(chain = true)
@NoArgsConstructor // json support
@AllArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
public class AeroflotServicePayload implements ServicePayload {
    //@NonNull // backward compatibility, all new payloads must have the field set
    private String variantId;
    @NonNull
    private AeroflotVariant variant;
    @NonNull
    private String partnerId;
    @NonNull
    private List<TravellerInfo> travellers;
    @NonNull
    private Money preliminaryCost;
    @NonNull
    private ClientInfo clientInfo;

    private AeroflotOrderRef bookingRef;

    // travellerInfoId -> [ticket numbers]
    private Map<String, List<String>> tickets;
    // ticketId -> [couponId:couponState]
    private Map<String, List<AeroflotTicketCoupon>> ticketCoupons;

    //@NonNull // backward compatibility, all new payloads must have the field set
    private Map<String, SegmentFare> fareTerms;

    private BookingFailureReason bookingFailureReason;

    private AviaPromoCampaignsInfo promoCampaignsInfo;

    private AviaDicts aviaDicts;

    @JsonIgnore
    @Override
    public BookingPriceInfo getActualCosts() {
        Map<String, PassengerPricePart> passengerPrices = new HashMap<>();
        for (TravellerInfo travellerInfo : travellers) {
            AeroflotAnonymousTraveller traveller = variant.getTravellerForCategory(
                    AeroflotCodesMapping.categoryCode(travellerInfo.getCategory()));
            AeroflotCategoryOffer categoryOffer = variant.getOffer().getCategoryOfferFor(traveller.getId());
            AeroflotPriceDetail categoryPrice = categoryOffer.getTotalPrice();
            passengerPrices.put(travellerInfo.getTravellerInfoId(), PassengerPricePart.builder()
                    .total(categoryPrice.getTotalPrice())
                    .fare(categoryPrice.getBasePrice())
                    .tax(categoryPrice.getTaxes())
                    .build());
        }
        return BookingPriceInfo.builder()
                .totalAmount(variant.getOffer().getTotalPrice())
                .passengerPrices(passengerPrices)
                .build();
    }

    @JsonIgnore
    public List<TravellerInfo> getTravellersOfCategory(PassengerCategory category) {
        return travellers.stream()
                .filter(t -> t.getCategory() == category)
                .collect(toList());
    }

    @JsonIgnore
    public List<String> getTicketsFlatList() {
        if (tickets == null) {
            return null;
        }
        return tickets.values().stream()
                .flatMap(Collection::stream)
                .collect(toList());
    }
}
