package ru.yandex.avia.booking.partners.gateways.aeroflot.model;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.experimental.Accessors;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotNdcApiVersion;

import static java.util.stream.Collectors.toSet;

@Data
@Accessors(chain = true)
@Builder(toBuilder = true)
@NoArgsConstructor // json support
@AllArgsConstructor
public class AeroflotVariant {
    //@NonNull for all new variants
    private SearchData searchData;
    //@NonNull for all valid variants (non temporary)
    private AeroflotRequestContext context;
    //@NonNull
    private List<AeroflotAnonymousTraveller> travellers;
    //@NonNull
    private List<AeroflotOriginDestination> originDestinations;
    //@NonNull
    private List<AeroflotSegment> segments;
    //@NonNull
    private AeroflotTotalOffer offer;
    //@NonNull
    @Builder.Default
    private List<AeroflotTotalOffer> allTariffs = new ArrayList<>();
    @Builder.Default
    private AeroflotNdcApiVersion apiVersion = AeroflotNdcApiVersion.V1;

    @NonNull
    public AeroflotVariant validateReferences() {
        Map<String, AeroflotOriginDestination> odMap = toMap(originDestinations);
        Set<String> segmentOdRefs = new HashSet<>();
        for (AeroflotSegment segment : segments) {
            segmentOdRefs.add(segment.getOriginDestinationId());
            Preconditions.checkNotNull(odMap.get(segment.getOriginDestinationId()),
                    "Unknown originDestination: segment=%s, variant=%s", segment.getId(), this);
        }
        Preconditions.checkArgument(segmentOdRefs.equals(odMap.keySet()),
                "Unused originDestination found: ods=%s, segmentRefs=%s",
                odMap.keySet(), segmentOdRefs);

        validateOfferReferences(offer);
        for (AeroflotTotalOffer alternative : allTariffs) {
            validateOfferReferences(alternative);
        }
        return this;
    }

    private void validateOfferReferences(@NonNull AeroflotTotalOffer offer) {
        Map<String, AeroflotSegment> segmentsMap = toMap(segments);
        Set<String> segmentRefs = new HashSet<>();
        for (AeroflotApplicableSegmentRef segmentRef : offer.getSegmentRefs()) {
            segmentRefs.add(segmentRef.getSegmentId());
            Preconditions.checkNotNull(segmentsMap.get(segmentRef.getSegmentId()),
                    "Unknown offer segment: segment=%s, variant=%s", segmentRef.getSegmentId(), this);
        }
        Preconditions.checkArgument(segmentRefs.equals(segmentsMap.keySet()),
                "Priced segments/refs mismatch: segments=%s, pricedRefs=%s",
                segmentsMap.keySet(), segmentRefs);

        Map<String, AeroflotAnonymousTraveller> travellersMap = toMap(travellers);
        Set<String> travellerRefs = new HashSet<>();
        for (AeroflotCategoryOffer categoryOffer : offer.getCategoryOffers()) {
            travellerRefs.add(categoryOffer.getTravellerId());
            Preconditions.checkNotNull(travellersMap.get(categoryOffer.getTravellerId()),
                    "Unknown traveller: traveller_id=%s, variant=%s", categoryOffer.getTravellerId(), this);
        }
        Preconditions.checkArgument(travellerRefs.equals(travellersMap.keySet()),
                "Travellers without an offer found: travellers=%s, offerRefs=%s",
                travellersMap.keySet(), travellerRefs);

        for (AeroflotCategoryOffer categoryOffer : offer.getCategoryOffers()) {
            Set<String> offerSegmentRefs = categoryOffer.getFareBasisCodes().stream()
                    .map(AeroflotSegmentFareCode::getSegmentId).collect(toSet());
            Preconditions.checkArgument(segmentsMap.keySet().equals(offerSegmentRefs),
                    "The category offer doesn't match the full segments list: " +
                            "offer=%s, variant=%s", categoryOffer, this);
        }
    }

    @NonNull
    public AeroflotAnonymousTraveller getTraveller(@NonNull String travellerId) {
        for (AeroflotAnonymousTraveller traveller : travellers) {
            if (traveller.getId().equals(travellerId)) {
                return traveller;
            }
        }
        throw new NoSuchElementException("traveller: id=" + travellerId);
    }

    @NonNull
    public AeroflotAnonymousTraveller getTravellerForCategory(@NonNull String categoryCode) {
        for (AeroflotAnonymousTraveller traveller : travellers) {
            if (traveller.getCategory().equals(categoryCode)) {
                return traveller;
            }
        }
        throw new NoSuchElementException("traveller: category=" + categoryCode);
    }

    @NonNull
    private <T extends AeroflotEntity> Map<String, T> toMap(@NonNull List<T> entities) {
        return entities.stream().collect(Collectors.toMap(AeroflotEntity::getId, e -> e));
    }

    @NonNull
    public AeroflotSegment getSegment(@NonNull String segmentId) {
        for (AeroflotSegment segment : segments) {
            if (segment.getId().equals(segmentId)) {
                return segment;
            }
        }
        throw new NoSuchElementException("segment: id=" + segmentId);
    }

    @NonNull
    public AeroflotOriginDestination getOriginDestination(@NonNull String originDestinationId) {
        for (AeroflotOriginDestination od : originDestinations) {
            if (od.getId().equals(originDestinationId)) {
                return od;
            }
        }
        throw new NoSuchElementException("originDestination: id=" + originDestinationId);
    }

    public List<AeroflotSegment> getOriginDestinationSegments(String originDestinationId) {
        List<AeroflotSegment> segments = new ArrayList<>();
        for (AeroflotSegment segment : this.segments) {
            if (segment.getOriginDestinationId().equals(originDestinationId)) {
                segments.add(segment);
            }
        }
        return segments;
    }
}
