package ru.yandex.avia.booking.partners.gateways.aeroflot.parsing;

import java.io.InputStream;
import java.io.StringReader;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.xml.sax.SAXException;

@Slf4j
public class XmlUtils {
    private static SAXReader createParser() {
        try {
            SAXReader saxReader = new SAXReader();
            // https://wiki.yandex-team.ru/security/for/web-developers/xxe/
            saxReader.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
            saxReader.setFeature("http://xml.org/sax/features/external-general-entities", false);
            saxReader.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
            return saxReader;
        } catch (SAXException e) {
            throw new RuntimeException("Failed to create a SAXReader", e);
        }
    }

    public static Document parseXml(String content) {
        try {
            return createParser().read(new StringReader(content));
        } catch (DocumentException e) {
            log.debug("Failed to parse the document: {}", content);
            throw new RuntimeException("Failed to parse the document", e);
        }
    }

    public static Document parseXml(InputStream is) {
        try {
            return createParser().read(is);
        } catch (DocumentException e) {
            log.debug("Failed to parse the document: <input stream>");
            throw new RuntimeException("Failed to parse the document", e);
        }
    }

    // ********************
    // Extraction helpers
    // ********************
    @NonNull
    public static String extractText(@NonNull Node node, @NonNull String xpath) {
        return extractNode(node, xpath).getText();
    }

    @NonNull
    public static double extractDouble(@NonNull Node node, @NonNull String xpath) {
        String value = extractText(node, xpath);
        return Double.parseDouble(value);
    }

    @NonNull
    public static Node extractNode(@NonNull Node node, @NonNull String xpath) {
        return extractNode(node, xpath, null);
    }

    @NonNull
    public static Node extractNode(@NonNull Node node, @NonNull String xpath, Predicate<Node> filter) {
        List<Node> candidates = node.selectNodes(xpath);
        if (filter != null) {
            candidates = candidates.stream().filter(filter).collect(Collectors.toList());
        }
        Preconditions.checkArgument(candidates.size() == 1,
                "exactly one result is expected: xpath=%s, found=%s, with_filter=%s",
                xpath, candidates.size(), filter != null);
        return candidates.get(0);
    }
}
