package ru.yandex.avia.booking.partners.gateways.aeroflot.v3;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.TreeMap;

import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotNdcApiVersion;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotAnonymousTraveller;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotApplicableSegmentRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCarrier;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCategoryOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOriginDestination;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotPriceDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotRequestContext;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegmentFareCode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegmentNode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTotalOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.SearchData;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRsBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CabinType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CarrierAircraftType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CarrierOffers;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.DataLists;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.DatedOperatingLeg;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FareComponent;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FareDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FarePriceType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.MarketingCarrierInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.MoneyAmount;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Offer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferItem;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferPriceRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferService;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OffersGroup;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OperatingCarrierInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OriginDest;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PTC;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Pax;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaxJourney;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaxSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PriceClass;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PriceDetalization;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.RBD;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ScheduledLocation;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

@Slf4j
public class AeroflotNdcApiV3CompatibilityConverter {
    public AeroflotVariant convertToV1Variant(AirShoppingRs airShoppingRs, String offerId,
                                              AeroflotRequestContext context, SearchData searchData) {
        DataLists dataLists = airShoppingRs.getResponse().getDataLists();
        Offer offer = AeroflotNdcApiV3Helper.findOfferById(airShoppingRs, offerId);
        List<Offer> allOffers = airShoppingRs.getResponse().getOffersGroup().getCarrierOffers().getOffer();
        return convertToV1VariantImpl(dataLists, offer, allOffers, null).toBuilder()
                .context(context == null ? null : context
                        .setLanguage(context.getLanguage().toUpperCase())
                        .setCountryCode(context.getCountryCode().toUpperCase())
                        .setResponseId(airShoppingRs.getResponse().getShoppingResponse().getShoppingResponseID()))
                .searchData(searchData)
                .build();
    }

    public AeroflotVariant convertToV1Variant(OfferPriceRs response, AeroflotVariant sourceVariant) {
        DataLists dataLists = response.getResponse().getDataLists();
        Offer offer = response.getResponse().getPricedOffer();
        return convertToV1VariantImpl(dataLists, offer, List.of(), sourceVariant);
    }

    private AeroflotVariant convertToV1VariantImpl(DataLists dataLists, Offer offer, List<Offer> allOffers,
                                                   AeroflotVariant sourceVariant) {
        return AeroflotVariant.builder()
                .apiVersion(AeroflotNdcApiVersion.V3)
                .originDestinations(dataLists.getOriginDestList().stream()
                        .map(this::convertOriginDestination)
                        .collect(toList()))
                .segments(dataLists.getPaxSegmentList().stream()
                        .map(segment -> convertSegments(segment, dataLists, sourceVariant))
                        .collect(toList()))
                .travellers(dataLists.getPaxList().stream()
                        .map(this::convertPassenger)
                        .collect(toList()))
                .offer(convertOffer(offer, dataLists.getPriceClassList()))
                .allTariffs(allOffers.stream()
                        .filter(this::isNotABrokenOffer)
                        .map(anyOffer -> convertOffer(anyOffer, dataLists.getPriceClassList()))
                        .collect(toList()))
                .build()
                .validateReferences();
    }

    public AirShoppingRs convertToV3AirShoppingDataForPriceCheck(AeroflotVariant variant) {
        Preconditions.checkArgument(variant.getApiVersion() == AeroflotNdcApiVersion.V3,
                "Only NDC API V3 variants can be converted for price check but got %s", variant.getApiVersion());
        ListMultimap<String, String> odId2journeyIds = ArrayListMultimap.create();
        variant.getSegments().forEach(segment ->
                odId2journeyIds.put(segment.getOriginDestinationId(), segment.getJourneyId()));
        LinkedHashMap<String, List<String>> journeyId2segmentIds = new LinkedHashMap<>();
        variant.getSegments().forEach(segment ->
                journeyId2segmentIds.computeIfAbsent(segment.getJourneyId(), (key) -> new ArrayList<>())
                        .add(segment.getId()));
        return AirShoppingRs.builder()
                .response(AirShoppingRsBody.builder()
                        .dataLists(DataLists.builder()
                                .originDestList(variant.getOriginDestinations().stream()
                                        .map(od -> convertOriginDestination(od,
                                                // preserving the order but without duplicates
                                                List.copyOf(new LinkedHashSet<>(odId2journeyIds.get(od.getId())))))
                                        .collect(toList()))
                                .paxJourneyList(journeyId2segmentIds.keySet().stream()
                                        .map(journeyId -> convertJourney(journeyId,
                                                List.copyOf(journeyId2segmentIds.get(journeyId))))
                                        .collect(toList()))
                                .paxSegmentList(variant.getSegments().stream()
                                        .map(this::convertSegment)
                                        .collect(toList()))
                                .paxList(variant.getTravellers().stream()
                                        .map(this::convertPassenger)
                                        .collect(toList()))
                                .priceClassList(convertPriceClasses(variant.getAllTariffs()))
                                .build())
                        .offersGroup(OffersGroup.builder()
                                .carrierOffers(CarrierOffers.builder()
                                        .offer(variant.getAllTariffs().stream()
                                                .map(this::convertOffer)
                                                .collect(toList()))
                                        .build())
                                .build())
                        .build())
                .build();
    }

    private AeroflotOriginDestination convertOriginDestination(OriginDest originDestination) {
        return AeroflotOriginDestination.builder()
                .id(originDestination.getOriginDestID())
                .departureCode(originDestination.getOriginCode())
                .arrivalCode(originDestination.getDestCode())
                .build();
    }

    private OriginDest convertOriginDestination(AeroflotOriginDestination originDestination, List<String> journeyIds) {
        return OriginDest.builder()
                .originDestID(originDestination.getId())
                .originCode(originDestination.getDepartureCode())
                .destCode(originDestination.getArrivalCode())
                .paxJourneyRefID(journeyIds)
                .build();
    }

    private PaxJourney convertJourney(String journeyId, List<String> segmentIds) {
        return PaxJourney.builder()
                .paxJourneyID(journeyId)
                .paxSegmentRefID(segmentIds)
                .build();
    }

    private AeroflotSegment convertSegments(PaxSegment segment, DataLists dataLists, AeroflotVariant sourceVariant) {
        String segmentId = segment.getPaxSegmentID();
        String journeyId = dataLists.getPaxJourneyList().stream()
                .filter(journey -> journey.getPaxSegmentRefID().contains(segmentId))
                .map(PaxJourney::getPaxJourneyID)
                .findFirst().orElseThrow(() -> new NoSuchElementException(
                        "No journey for segment id " + segmentId));
        String originDestinationId = dataLists.getOriginDestList().stream()
                .filter(originDest -> originDest.getPaxJourneyRefID().contains(journeyId))
                .map(OriginDest::getOriginDestID)
                .findFirst().orElseThrow(() -> new NoSuchElementException(
                        "No origin-destination for journey id " + journeyId));
        return AeroflotSegment.builder()
                .id(segmentId)
                .originDestinationId(originDestinationId)
                .journeyId(journeyId)
                .segmentTypeCode(segment.getSegmentTypeCode())
                .departure(convertScheduledLocation(segment.getDep()))
                .arrival(convertScheduledLocation(segment.getArrival()))
                .marketingCarrier(convertCarrier(segment.getMarketingCarrierInfo()))
                .operatingCarrier(segment.getOperatingCarrierInfo() != null ?
                        convertCarrier(segment.getOperatingCarrierInfo()) :
                        sourceVariant.getSegment(segmentId).getOperatingCarrier())
                .aircraftCode(segment.getDatedOperatingLeg() != null ?
                        segment.getDatedOperatingLeg().getCarrierAircraftType().getCarrierAircraftTypeCode() :
                        sourceVariant.getSegment(segmentId).getAircraftCode())
                .flightDuration(segment.getDuration().toString())
                .build();
    }

    private PaxSegment convertSegment(AeroflotSegment segment) {
        return PaxSegment.builder()
                .paxSegmentID(segment.getId())
                .segmentTypeCode(segment.getSegmentTypeCode())
                .dep(convertScheduledLocation(segment.getDeparture()))
                .arrival(convertScheduledLocation(segment.getArrival()))
                .marketingCarrierInfo(convertMarketingCarrier(segment.getMarketingCarrier()))
                .operatingCarrierInfo(convertOperatingCarrier(segment.getOperatingCarrier()))
                .datedOperatingLeg(DatedOperatingLeg.builder()
                        .carrierAircraftType(CarrierAircraftType.builder()
                                .carrierAircraftTypeCode(segment.getAircraftCode())
                                .build())
                        .build())
                .duration(Duration.parse(segment.getFlightDuration()))
                .build();
    }

    private AeroflotSegmentNode convertScheduledLocation(ScheduledLocation location) {
        LocalDateTime scheduledAt = location.getAircraftScheduledDateTime();
        return AeroflotSegmentNode.builder()
                .airportCode(location.getIataLocationCode())
                .date(scheduledAt.toLocalDate().toString())
                .time(scheduledAt.toLocalTime().toString())
                .build();
    }

    private ScheduledLocation convertScheduledLocation(AeroflotSegmentNode segmentNode) {
        return ScheduledLocation.builder()
                .aircraftScheduledDateTime(segmentNode.getDateTime())
                .iataLocationCode(segmentNode.getAirportCode())
                .build();
    }

    private AeroflotCarrier convertCarrier(MarketingCarrierInfo carrierInfo) {
        return AeroflotCarrier.builder()
                .airlineId(carrierInfo.getCarrierDesigCode())
                .flightNumber(carrierInfo.getMarketingCarrierFlightNumberText())
                .build();
    }

    private MarketingCarrierInfo convertMarketingCarrier(AeroflotCarrier carrier) {
        return MarketingCarrierInfo.builder()
                .carrierDesigCode(carrier.getAirlineId())
                .marketingCarrierFlightNumberText(carrier.getFlightNumber())
                .build();
    }

    private OperatingCarrierInfo convertOperatingCarrier(AeroflotCarrier carrier) {
        return OperatingCarrierInfo.builder()
                .carrierDesigCode(carrier.getAirlineId())
                .operatingCarrierFlightNumberText(carrier.getFlightNumber())
                .build();
    }

    private AeroflotCarrier convertCarrier(OperatingCarrierInfo carrierInfo) {
        return AeroflotCarrier.builder()
                .airlineId(carrierInfo.getCarrierDesigCode())
                .flightNumber(carrierInfo.getOperatingCarrierFlightNumberText())
                .build();
    }

    private AeroflotAnonymousTraveller convertPassenger(Pax pax) {
        return AeroflotAnonymousTraveller.builder()
                .id(pax.getPaxID())
                .category(pax.getPtc().getValue())
                // also, should remove the quantity field (... or shouldn't for backward compatibility)
                .quantity(1)
                .build();
    }

    private Pax convertPassenger(AeroflotAnonymousTraveller traveller) {
        Preconditions.checkArgument(traveller.getQuantity() == 1,
                "Unexpected traveller quantity; expected exactly 1 but got %s", traveller.getQuantity());
        return Pax.builder()
                .paxID(traveller.getId())
                .ptc(PTC.forValue(traveller.getCategory()))
                .build();
    }

    private List<PriceClass> convertPriceClasses(List<AeroflotTotalOffer> offers) {
        // reconstructing price classes from the expanded denormalized form
        Map<String, PriceClass> priceClasses = new TreeMap<>();
        for (AeroflotTotalOffer offer : offers) {
            for (AeroflotCategoryOffer travellerOffer : offer.getCategoryOffers()) {
                for (AeroflotSegmentFareCode fb : travellerOffer.getFareBasisCodes()) {
                    // removing the potential child discount part "/CH25"
                    String fareCode = fb.getFareCode().split("/")[0];
                    PriceClass pc = PriceClass.builder()
                            .priceClassID(fb.getPriceClassRefId())
                            .code(fb.getPriceClassCode())
                            .fareBasisCode(fareCode)
                            .build();
                    if (!priceClasses.containsKey(pc.getPriceClassID())) {
                        priceClasses.put(pc.getPriceClassID(), pc);
                    } else {
                        Preconditions.checkArgument(pc.equals(priceClasses.get(pc.getPriceClassID())),
                                "Prices class denormalized descriptions mismatch: [%s, %s]",
                                pc, priceClasses.get(pc.getPriceClassID()));
                    }
                }
            }
        }
        return List.copyOf(priceClasses.values());
    }

    private boolean isNotABrokenOffer(Offer offer) {
        boolean isBroken = offer.getOfferItem().stream()
                .anyMatch(oi -> oi.getFareDetail().getFareComponent().stream()
                        .anyMatch(fc -> fc.getPriceClassRefID() == null));
        if (isBroken) {
            log.warn("Broken offer detected, no PriceClassRefID for FareComponent: id {}, url {}; skipping it",
                    offer.getOfferID(), offer.getWebAddressURL());
        }
        return !isBroken;
    }

    private AeroflotTotalOffer convertOffer(Offer offer, List<PriceClass> priceClasses) {
        Map<String, AeroflotApplicableSegmentRef> segmentRefs = new LinkedHashMap<>();
        for (OfferItem offerItem : offer.getOfferItem()) {
            for (FareComponent fareComponent : offerItem.getFareDetail().getFareComponent()) {
                String segmentId = fareComponent.getPaxSegmentRefID();
                AeroflotApplicableSegmentRef segmentRef = AeroflotApplicableSegmentRef.builder()
                        .segmentId(segmentId)
                        .marriedSegmentGroup("0")
                        .classOfServiceCode(fareComponent.getRbd().getRbdCode())
                        .build();
                if (!segmentRefs.containsKey(segmentId)) {
                    segmentRefs.put(segmentId, segmentRef);
                } else {
                    Preconditions.checkArgument(segmentRef.equals(segmentRefs.get(segmentId)),
                            "Denormalized segment refs mismatch: [%s, %s]", segmentRefs.get(segmentId), segmentRef);
                }
            }
        }
        return AeroflotTotalOffer.builder()
                .id(offer.getOfferID())
                .ownerCode(offer.getOwnerCode())
                .totalPrice(convertMoney(offer.getTotalPrice().getTotalAmount()))
                .categoryOffers(offer.getOfferItem().stream()
                        .flatMap(oi -> convertOfferItem(oi, priceClasses).stream())
                        .collect(toList()))
                .segmentRefs(List.copyOf(segmentRefs.values()))
                .disclosureUrl(offer.getWebAddressURL())
                .build();
    }

    private Offer convertOffer(AeroflotTotalOffer offer) {
        return Offer.builder()
                .offerID(offer.getId())
                .ownerCode(offer.getOwnerCode())
                .totalPrice(PriceDetalization.builder()
                        .totalAmount(convertMoney(offer.getTotalPrice()))
                        .build())
                .offerItem(convertOfferItem(offer.getCategoryOffers()))
                .webAddressURL(offer.getDisclosureUrl())
                .build();
    }

    private List<AeroflotCategoryOffer> convertOfferItem(OfferItem offerItem, List<PriceClass> priceClasses) {
        PriceDetalization priceInfo = offerItem.getFareDetail().getFarePriceType().getPrice();
        Money totalPrice = convertMoney(priceInfo.getTotalAmount());
        Money basePrice = convertMoney(priceInfo.getEquivAmount());
        Map<String, String> pcId2PcCode = priceClasses.stream()
                .collect(toMap(PriceClass::getPriceClassID, PriceClass::getCode));
        return offerItem.getService().getPaxRefID().stream()
                .map(paxRefId -> AeroflotCategoryOffer.builder()
                        .id(offerItem.getOfferItemID())
                        .travellerId(paxRefId)
                        .totalPrice(AeroflotPriceDetail.builder()
                                .basePrice(basePrice)
                                .taxes(totalPrice.subtract(basePrice))
                                .totalPrice(totalPrice)
                                .build())
                        .fareBasisCodes(offerItem.getFareDetail().getFareComponent().stream()
                                .map(fc -> convertFaceComponent(fc, pcId2PcCode.get(fc.getPriceClassRefID())))
                                .collect(toList()))
                        .build())
                .collect(toList());
    }

    private List<OfferItem> convertOfferItem(List<AeroflotCategoryOffer> travellerOffers) {
        // the offers were split per each user, now we're grouping them back
        Map<String, OfferItem> offerItems = new TreeMap<>();
        ArrayListMultimap<String, String> offerItemId2paxId = ArrayListMultimap.create();
        for (AeroflotCategoryOffer travellerOffer : travellerOffers) {
            AeroflotPriceDetail totalPrice = travellerOffer.getTotalPrice();
            OfferItem offerItem = OfferItem.builder()
                    .offerItemID(travellerOffer.getId())
                    .fareDetail(FareDetail.builder()
                            .fareComponent(travellerOffer.getFareBasisCodes().stream()
                                    .map(this::convertFaceComponent)
                                    .collect(toList()))
                            .farePriceType(FarePriceType.builder()
                                    .farePriceTypeCode(FarePriceType.DEFAULT_PRICE_TYPE_CODE)
                                    .price(PriceDetalization.builder()
                                            .equivAmount(convertMoney(totalPrice.getBasePrice()))
                                            .totalAmount(convertMoney(totalPrice.getTotalPrice()))
                                            .build())
                                    .build())
                            .build())
                    .build();
            if (!offerItems.containsKey(offerItem.getOfferItemID())) {
                offerItems.put(offerItem.getOfferItemID(), offerItem);
            } else {
                Preconditions.checkArgument(offerItem.equals(offerItems.get(offerItem.getOfferItemID())),
                        "Denormalized offer items mismatch: [%s, %s]",
                        offerItem, offerItems.get(offerItem.getOfferItemID()));
            }
            offerItemId2paxId.put(travellerOffer.getId(), travellerOffer.getTravellerId());
        }
        return offerItems.values().stream()
                .map(oi -> oi.toBuilder()
                        .service(OfferService.builder()
                                .paxRefID(offerItemId2paxId.get(oi.getOfferItemID()))
                                .build())
                        .build())
                .collect(toList());
    }

    private Money convertMoney(MoneyAmount money) {
        return Money.of(money.getValue(), money.getCurCode());
    }

    static MoneyAmount convertMoney(Money money) {
        return MoneyAmount.builder()
                // not using getNumberStripped to preserve the format
                .value(new BigDecimal(money.getNumber().toString()))
                .curCode(money.getCurrency().getCurrencyCode())
                .build();
    }

    private AeroflotSegmentFareCode convertFaceComponent(FareComponent fareComponent, String priceClassCode) {
        return AeroflotSegmentFareCode.builder()
                .fareCode(fareComponent.getFareBasisCode())
                .segmentId(fareComponent.getPaxSegmentRefID())
                .priceClassRefId(fareComponent.getPriceClassRefID())
                .priceClassCode(priceClassCode)
                .cabinTypeCode(fareComponent.getCabinType().getCabinTypeCode())
                .rbdCode(fareComponent.getRbd().getRbdCode())
                .build();
    }

    private FareComponent convertFaceComponent(AeroflotSegmentFareCode fareComponent) {
        return FareComponent.builder()
                .fareBasisCode(fareComponent.getFareCode())
                .paxSegmentRefID(fareComponent.getSegmentId())
                .priceClassRefID(fareComponent.getPriceClassRefId())
                .cabinType(CabinType.builder()
                        .cabinTypeCode(fareComponent.getCabinTypeCode())
                        .build())
                .rbd(RBD.builder().rbdCode(fareComponent.getRbdCode()).build())
                .build();
    }
}
