package ru.yandex.avia.booking.partners.gateways.aeroflot.v3;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.partners.gateways.aeroflot.converter.AeroflotCodesMapping;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FareDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FarePriceType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.MoneyAmount;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Offer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferItem;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OffersGroup;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PTC;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Pax;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PriceDetalization;
import ru.yandex.avia.booking.partners.gateways.model.search.CategoryPrice;
import ru.yandex.avia.booking.partners.gateways.model.search.PriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.search.Variant;
import ru.yandex.travel.commons.streams.CustomCollectors;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.avia.booking.partners.gateways.aeroflot.v3.AeroflotNdcApiV3CompatibilityConverter.convertMoney;

@Slf4j
public class AeroflotNdcApiV3VariantsSynchronizer {
    public static AirShoppingRs synchronizeVariants(AirShoppingRs airShoppingRs, Variant variantInfo) {
        Map<String, PriceInfo> newOffers = variantInfo.getAllTariffs().stream()
                .collect(toMap(PriceInfo::getId, e -> e));
        List<Pax> passengers = airShoppingRs.getResponse().getDataLists().getPaxList();
        OffersGroup offersGroup = airShoppingRs.getResponse().getOffersGroup();
        List<Offer> offers = offersGroup.getCarrierOffers().getOffer();
        List<Offer> updatedOffers = synchronizeOffers(offers, newOffers, passengers);
        if (updatedOffers == null) {
            // no changes detected
            return null;
        } else {
            return airShoppingRs.toBuilder()
                    .response(airShoppingRs.getResponse().toBuilder()
                            .offersGroup(offersGroup.toBuilder()
                                    .carrierOffers(offersGroup.getCarrierOffers().toBuilder()
                                            .offer(updatedOffers)
                                            .build())
                                    .build())
                            .build())
                    .build();
        }
    }

    private static List<Offer> synchronizeOffers(List<Offer> offers, Map<String, PriceInfo> newOffers,
                                                 List<Pax> passengers) {
        List<Offer> updatedOffers = new ArrayList<>(offers);
        boolean pricesChanged = false;
        for (int i = 0; i < updatedOffers.size(); i++) {
            Offer offer = updatedOffers.get(i);
            PriceInfo newOffer = newOffers.get(offer.getOfferID());
            if (newOffer == null) {
                log.warn("No new offer for AirShopping offer {}", offer.getOfferID());
                continue;
            }
            PriceDetalization total = offer.getTotalPrice();
            MoneyAmount newTotal = convertMoney(newOffer.getTotal());
            if (newTotal.equals(total.getTotalAmount())) {
                continue;
            }
            log.info("The offer price has changed: {} -> {}, offer id {}",
                    total.getTotalAmount(), newTotal, offer.getOfferID());
            pricesChanged = true;
            updatedOffers.set(i, offer.toBuilder()
                    .totalPrice(total.toBuilder()
                            .totalAmount(newTotal)
                            // can try to calculate it but the total equiv value isn't used at the moment
                            //.equivAmount()
                            .build())
                    .offerItem(offer.getOfferItem().stream()
                            .map(item -> synchronizeOfferItem(item, newOffer, passengers))
                            .collect(toList()))
                    .build());
        }
        return pricesChanged ? updatedOffers : null;
    }

    private static OfferItem synchronizeOfferItem(OfferItem item, PriceInfo newOffer, List<Pax> passengers) {
        Map<String, PTC> paxId2pax = passengers.stream().collect(toMap(Pax::getPaxID, Pax::getPtc));
        List<PassengerCategory> uniqueCategories = item.getService().getPaxRefID().stream()
                .map(paxRef -> AeroflotCodesMapping.categoryEnum(paxId2pax.get(paxRef).getValue()))
                .distinct()
                .collect(toList());
        Preconditions.checkState(uniqueCategories.size() == 1,
                "Exactly 1 unique category is expected for offer item %s but got %s",
                item.getOfferItemID(), uniqueCategories);
        CategoryPrice categoryPrice = newOffer.getCategoryPrices().stream()
                .filter(price -> price.getPassengerCategory() == uniqueCategories.get(0))
                .collect(CustomCollectors.exactlyOne());
        FareDetail fareDetail = item.getFareDetail();
        FarePriceType farePrice = fareDetail.getFarePriceType();
        PriceDetalization priceDetalization = farePrice.getPrice();
        return item.toBuilder()
                .fareDetail(fareDetail.toBuilder()
                        .farePriceType(farePrice.toBuilder()
                                .price(priceDetalization.toBuilder()
                                        .totalAmount(convertMoney(categoryPrice.getTotal()))
                                        .equivAmount(convertMoney(categoryPrice.getFare()))
                                        .build())
                                .build())
                        .build())
                .build();
    }
}
