package ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests;

import java.io.IOException;

import javax.xml.stream.XMLInputFactory;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.deser.std.StdScalarDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.dataformat.xml.XmlFactory;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import com.fasterxml.jackson.dataformat.xml.util.DefaultXmlPrettyPrinter;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationModule;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AeroflotNdcApiV3ModelXmlConverter {
    private final ObjectMapper xmlMapper;

    public AeroflotNdcApiV3ModelXmlConverter(AeroflotNdcApiV3ModelXmlConverterConfig config) {
        ObjectMapper xmlMapper = new XmlMapper(createXmlFactory())
                // do not wrap Collection fields into an extra element by default
                .setDefaultUseWrapper(false)
                .enable(ToXmlGenerator.Feature.WRITE_XML_DECLARATION)
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, !config.isUnknownPropertiesAllowed())
                .registerModule(new JavaTimeModule())
                .registerModule(new JaxbAnnotationModule())
                .registerModule(textTrimmerModule())
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_NULL)
                .setPropertyNamingStrategy(PropertyNamingStrategy.UPPER_CAMEL_CASE);
        if (config.isPrettyPrinterEnabled()) {
            DefaultXmlPrettyPrinter.Indenter indenter = new LfNSpacesIndenter(4);
            DefaultXmlPrettyPrinter printer = new DefaultXmlPrettyPrinter();
            printer.indentObjectsWith(indenter);
            printer.indentArraysWith(indenter);
            xmlMapper.enable(SerializationFeature.INDENT_OUTPUT);
            xmlMapper.setDefaultPrettyPrinter(printer);
        }
        this.xmlMapper = xmlMapper;
    }

    private static XmlFactory createXmlFactory() {
        // to get the "xmlns" value as an attribute we need to disable its special treatment:
        // https://github.com/FasterXML/jackson-dataformat-xml/issues/272
        XMLInputFactory xif = XMLInputFactory.newInstance();
        xif.setProperty(XMLInputFactory.IS_NAMESPACE_AWARE, false);
        // https://wiki.yandex-team.ru/security/for/web-developers/xxe/
        // also implemented in the constructor of XmlFactory
        xif.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
        xif.setProperty(XMLInputFactory.SUPPORT_DTD, false);
        return new XmlFactory(xif);
    }

    private static Module textTrimmerModule() {
        // this deserializer is needed to trim element content like this: <ID>\n ID1 \n</ID>
        return new SimpleModule() {
            {
                // We have to explicitly specify the <String> template type parameter because of the following jdk bug:
                // https://stackoverflow.com/questions/54775253/jdk-11-0-2-compilation-fails-with-javac-npe-on-
                // It can be remove we the move to any 11.0.4+ version
                //noinspection Convert2Diamond
                addDeserializer(String.class, new StdScalarDeserializer<String>(String.class) {
                    @Override
                    public String deserialize(JsonParser jp, DeserializationContext context) throws IOException {
                        String value = jp.getValueAsString();
                        return value != null ? value.trim() : null;
                    }
                });
            }
        };
    }

    public String convertToXml(Object object) {
        try {
            return xmlMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            log.warn("Failed to serialize the object: " + object);
            throw new RuntimeException(e);
        }
    }

    public <T> T convertFromXml(String xml, Class<T> responseClass) {
        try {
            return xmlMapper.readValue(xml, responseClass);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
