package ru.yandex.avia.booking.partners.gateways.aeroflot.v3.requests;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableBiMap;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.avia.booking.enums.ClassOfService;
import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.enums.Sex;
import ru.yandex.avia.booking.model.Passengers;
import ru.yandex.avia.booking.model.SearchRequest;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.AeroflotNdcApiV3Helper;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Aggregator;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRqBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRs;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.AirShoppingRsBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CabinTypeCode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CabinTypeCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Carrier;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CarrierCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ContactInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Country;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.CreateOrderItem;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.DataLists;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.DestArrivalCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.EmailAddress;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FareComponent;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FareDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FarePriceType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FlightCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FlightItem;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.FlightItemFareDetail;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.GenderCode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.IdentityDoc;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.IdentityDocType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Individual;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.LoyaltyProgram;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.LoyaltyProgramAccount;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.MarketingCarrierInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Offer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferItemType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferPriceRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OfferPriceRqBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderCreateRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderCreateRqBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderFilterCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderRetrieveRq;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OrderRetrieveRqBody;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OriginDepCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OriginDest;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.OriginDestCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PTC;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Party;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Pax;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaxJourney;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaxSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PayloadAttributes;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentInfo;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentMethod;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentTrx;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PaymentType;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Phone;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PointOfSale;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PriceClass;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PriceDetalization;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.PricedOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ResponseParameters;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ScheduledLocation;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.SelectedOffer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.SelectedOfferItem;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.Sender;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.ShoppingCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.SpecialNeedsCriteria;
import ru.yandex.avia.booking.partners.gateways.aeroflot.v3.model.SuffixName;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.travel.commons.map.MappingUtils;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.travel.commons.map.MappingUtils.getMappingOrThrow;

@AllArgsConstructor
@Slf4j
public class AeroflotNdcApiV3RequestFactory {
    public static final Map<ClassOfService, CabinTypeCode> classMapping =
            ImmutableBiMap.<ClassOfService, CabinTypeCode>builder()
                    .put(ClassOfService.ECONOMY, CabinTypeCode.ECONOMY)
                    .put(ClassOfService.PREMIUM_ECONOMY, CabinTypeCode.COMFORT)
                    .put(ClassOfService.BUSINESS, CabinTypeCode.BUSINESS)
                    .build();
    private static final Map<PassengerCategory, PTC> categoryMapping =
            ImmutableBiMap.<PassengerCategory, PTC>builder()
                    .put(PassengerCategory.ADULT, PTC.ADULT)
                    .put(PassengerCategory.CHILD, PTC.CHILD)
                    .put(PassengerCategory.INFANT, PTC.INFANT)
                    .build();
    private static final Map<Sex, GenderCode> sexMapping =
            ImmutableBiMap.<Sex, GenderCode>builder()
                    .put(Sex.MALE, GenderCode.MALE)
                    .put(Sex.FEMALE, GenderCode.FEMALE)
                    .build();

    private final AeroflotNdcApiV3RequestFactoryConfig config;

    public AirShoppingRq createAirShoppingRq(SearchRequest searchRequest, boolean allTariffs) {
        return AirShoppingRq.builder()
                .party(createParty())
                .payloadAttributes(createPayloadAttributes(searchRequest.getLanguage().toUpperCase()))
                .pointOfSale(createPointOfSale(searchRequest.getCountry().toUpperCase()))
                .request(AirShoppingRqBody.builder()
                        .flightCriteria(FlightCriteria.builder()
                                .originDestCriteria(searchRequest.getRoute().stream()
                                        .map(od -> OriginDestCriteria.builder()
                                                .originDepCriteria(OriginDepCriteria.builder()
                                                        .date(od.getDepartureDate())
                                                        .iataLocationCode(od.getDepartureLocationCode())
                                                        .build())
                                                .destArrivalCriteria(DestArrivalCriteria.builder()
                                                        .iataLocationCode(od.getArrivalLocationCode())
                                                        .build())
                                                .build())
                                        .collect(toList()))
                                .build())
                        .paxList(convertPassengers(searchRequest.getPassengers()))
                        .responseParameters(allTariffs ?
                                ResponseParameters.ALL_TARIFFS_SEARCH : ResponseParameters.INITIAL_SEARCH)
                        .shoppingCriteria(ShoppingCriteria.builder()
                                .cabinTypeCriteria(CabinTypeCriteria.builder()
                                        .cabinTypeCode(getMappingOrThrow(
                                                classMapping, searchRequest.getClassOfService()))
                                        .build())
                                .carrierCriteria(CarrierCriteria.builder()
                                        .carrier(Carrier.AEROFLOT)
                                        .build())
                                .flightCriteria(FlightCriteria.builder().build())
                                .specialNeedsCriteria(SpecialNeedsCriteria.builder()
                                        .freeText(SpecialNeedsCriteria.NUMBER_OF_RESULTS)
                                        .qty(searchRequest.getMaxSearchResults())
                                        .build())
                                .build())
                        .build())
                .build();
    }

    private List<Pax> convertPassengers(Passengers passengers) {
        List<Pax> converted = new ArrayList<>();
        Preconditions.checkArgument(passengers.getAdults() != null && passengers.getAdults() > 0,
                "Adult passengers parameter should be a positive integer but got %s", passengers.getAdults());
        Map<PTC, Integer> categoryCounters = new LinkedHashMap<>();
        categoryCounters.put(PTC.ADULT, passengers.getAdults());
        categoryCounters.put(PTC.CHILD, passengers.getChildren());
        categoryCounters.put(PTC.INFANT, passengers.getInfants());
        for (PTC ptc : categoryCounters.keySet()) {
            Integer amount = categoryCounters.get(ptc);
            if (amount == null) {
                continue;
            }
            for (int i = 0; i < amount; i++) {
                converted.add(Pax.builder()
                        .ptc(ptc)
                        .paxID(String.valueOf(converted.size() + 1))
                        .build());
            }
        }
        return converted;
    }

    public AirShoppingRs createTrimmedAirShoppingRsData(AirShoppingRs airShoppingRs, String selectedOfferId) {
        AirShoppingRsBody responseBody = airShoppingRs.getResponse();
        DataLists dataLists = responseBody.getDataLists();
        Offer offer = AeroflotNdcApiV3Helper.findOfferById(airShoppingRs, selectedOfferId);

        // we need this order to keep other references sorted, see getPaxSegmentList processing
        List<String> journeyIds = getSegments(offer);
        // only the same variant (route + schedule) offers with various tariffs
        List<Offer> allTariffs = responseBody.getOffersGroup().getCarrierOffers().getOffer().stream()
                .filter(o -> getSegments(o).equals(journeyIds))
                .collect(toList());

        List<OriginDest> origDestinations = dataLists.getOriginDestList().stream()
                // only relevant ODs
                .filter(od -> od.getPaxJourneyRefID().stream().anyMatch(journeyIds::contains))
                .map(od -> od.toBuilder()
                        .paxJourneyRefID(od.getPaxJourneyRefID().stream()
                                // with only relevant journey refs
                                .filter(journeyIds::contains)
                                .collect(toList()))
                        .build())
                .collect(toList());

        List<PaxJourney> journeys = dataLists.getPaxJourneyList().stream()
                // only relevant journeys
                .filter(j -> journeyIds.contains(j.getPaxJourneyID()))
                // restoring the proper data order after it gets shuffled by the Ticket Daemon
                .sorted(Comparator.comparing(j -> journeyIds.indexOf(j.getPaxJourneyID())))
                .collect(toList());

        Set<String> segmentIds = journeys.stream()
                .flatMap(j -> j.getPaxSegmentRefID().stream())
                .collect(toSet());
        List<PaxSegment> segments = dataLists.getPaxSegmentList().stream()
                // only relevant segments
                .filter(od -> segmentIds.contains(od.getPaxSegmentID()))
                .collect(toList());

        Set<String> priceClassIds = allTariffs.stream()
                .flatMap(o -> o.getOfferItem().stream())
                .flatMap(oi -> oi.getFareDetail().getFareComponent().stream())
                .map(FareComponent::getPriceClassRefID)
                .collect(toSet());
        List<PriceClass> priceClasses = dataLists.getPriceClassList().stream()
                // only relevant price classes
                .filter(priceClass -> priceClassIds.contains(priceClass.getPriceClassID()))
                .collect(toList());

        // keeping all data as is except for the filtered lists
        return airShoppingRs.toBuilder()
                .response(responseBody.toBuilder()
                        .dataLists(dataLists.toBuilder()
                                .originDestList(origDestinations)
                                .paxJourneyList(journeys)
                                .paxSegmentList(segments)
                                .priceClassList(priceClasses)
                                .build())
                        .offersGroup(responseBody.getOffersGroup().toBuilder()
                                .carrierOffers(responseBody.getOffersGroup().getCarrierOffers().toBuilder()
                                        .offer(allTariffs)
                                        .build())
                                .build())
                        .build())
                .build();
    }

    private List<String> getSegments(Offer offer) {
        return offer.getOfferItem().stream()
                .flatMap(oi -> oi.getService().getServiceAssociations().getPaxJourneyRefID().stream())
                .collect(toList());
    }

    public OfferPriceRq createOfferPriceRq(OfferPriceRequestParams params) {
        DataLists dataLists = params.getDataLists();
        Offer offer = params.getOffer();
        PricedOffer pricedOffer = createPricedOffer(params.getOffer(), dataLists.getOriginDestList());
        List<PriceClass> filteredPriceClasses = collectRequiredPriceClasses(dataLists.getPriceClassList(), offer);
        return OfferPriceRq.builder()
                .party(createParty())
                .payloadAttributes(createPayloadAttributes(params.getLanguage()))
                .pointOfSale(createPointOfSale(params.getCountryOfSale()))
                .request(OfferPriceRqBody.builder()
                        .dataLists(DataLists.builder()
                                .originDestList(dataLists.getOriginDestList())
                                .paxJourneyList(toShortJourneys(dataLists.getPaxJourneyList()))
                                .paxList(dataLists.getPaxList())
                                .paxSegmentList(toShortSegments(dataLists.getPaxSegmentList()))
                                .priceClassList(toShortPriceClasses(filteredPriceClasses))
                                .build())
                        .pricedOffer(pricedOffer.toBuilder()
                                .selectedOffer(SelectedOffer.builder()
                                        .offerRefID(offer.getOfferID())
                                        .ownerCode(offer.getOwnerCode())
                                        .selectedOfferItem(offer.getOfferItem().stream()
                                                .map(oi -> SelectedOfferItem.builder()
                                                        .offerItemRefID(oi.getOfferItemID())
                                                        .paxRefID(oi.getService().getPaxRefID())
                                                        .build())
                                                .collect(toList()))
                                        .shoppingResponseRefID(params.getShoppingResponseId())
                                        .build())
                                .build())
                        .build())
                .build();
    }

    private PricedOffer createPricedOffer(Offer offer, List<OriginDest> originDestList) {
        return PricedOffer.builder()
                .createOrderItem(offer.getOfferItem().stream()
                        .map(oi -> CreateOrderItem.builder()
                                .offerItemID(oi.getOfferItemID())
                                .offerItemType(OfferItemType.builder()
                                        .flightItem(createOfferFlightItem(
                                                oi.getFareDetail(), originDestList, oi.getService().getPaxRefID()))
                                        .build())
                                .ownerCode(offer.getOwnerCode())
                                .build())
                        .collect(toList()))
                .build();
    }

    private FlightItem createOfferFlightItem(FareDetail fareDetail, List<OriginDest> originDestList,
                                             List<String> paxRefID) {
        return FlightItem.builder()
                .fareDetail(FlightItemFareDetail.builder()
                        .fareComponent(fareDetail.getFareComponent().stream()
                                .map(fc -> FareComponent.builder()
                                        .paxSegmentRefID(fc.getPaxSegmentRefID())
                                        .priceClassRefID(fc.getPriceClassRefID())
                                        .rbd(fc.getRbd())
                                        .build())
                                .collect(toList()))
                        .farePriceType(FarePriceType.builder()
                                .farePriceTypeCode(fareDetail.getFarePriceType().getFarePriceTypeCode())
                                .price(PriceDetalization.builder()
                                        .totalAmount(fareDetail.getFarePriceType().getPrice().getTotalAmount())
                                        .build())
                                .build())
                        .paxRefID(paxRefID)
                        .build())
                .originDestRefID(originDestList.stream()
                        .map(OriginDest::getOriginDestID)
                        .collect(toList()))
                .build();
    }

    private List<PaxJourney> toShortJourneys(List<PaxJourney> journeys) {
        return journeys.stream()
                .map(pj -> PaxJourney.builder()
                        .paxJourneyID(pj.getPaxJourneyID())
                        .paxSegmentRefID(pj.getPaxSegmentRefID())
                        .build())
                .collect(toList());
    }

    private List<PaxSegment> toShortSegments(List<PaxSegment> segments) {
        return segments.stream()
                .map(s -> PaxSegment.builder()
                        .arrival(ScheduledLocation.builder()
                                .iataLocationCode(s.getArrival().getIataLocationCode())
                                .build())
                        .dep(ScheduledLocation.builder()
                                .iataLocationCode(s.getDep().getIataLocationCode())
                                .aircraftScheduledDateTime(s.getDep().getAircraftScheduledDateTime())
                                .build())
                        .marketingCarrierInfo(MarketingCarrierInfo.builder()
                                .carrierDesigCode(s.getMarketingCarrierInfo().getCarrierDesigCode())
                                .marketingCarrierFlightNumberText(s.getMarketingCarrierInfo().getMarketingCarrierFlightNumberText())
                                .build())
                        .paxSegmentID(s.getPaxSegmentID())
                        .build())
                .collect(toList());
    }

    private List<PriceClass> toShortPriceClasses(List<PriceClass> priceClassList) {
        return priceClassList.stream()
                .map(pc -> PriceClass.builder()
                        .priceClassID(pc.getPriceClassID())
                        .code(pc.getCode())
                        // the element is mandatory but without any useful meaning, we pass the code here as
                        // the name isn't always available after all the compatibility conversions
                        .name(pc.getFareBasisCode())
                        .build())
                .collect(toList());
    }

    private List<PriceClass> collectRequiredPriceClasses(List<PriceClass> priceClassList, Offer offer) {
        Set<String> priceClassRefs = offer.getOfferItem().stream()
                .flatMap(oi -> oi.getFareDetail().getFareComponent().stream())
                .map(FareComponent::getPriceClassRefID)
                .collect(toSet());
        return priceClassList.stream()
                .filter(pc -> priceClassRefs.contains(pc.getPriceClassID()))
                .collect(toList());
    }

    public OrderCreateRq createOrderCreateRq(OrderCreateRequestParams params) {
        DataLists dataLists = params.getDataLists();
        Offer offer = params.getOffer();
        ContactInfo contactInfo = ContactInfo.builder()
                .contactInfoID("CONTACT1")
                .emailAddress(new EmailAddress(params.getContactEmail()))
                .phone(new Phone(params.getContactPhoneCountryCode(), params.getContactPhoneNumber()))
                .build();
        List<PriceClass> filteredPriceClasses = collectRequiredPriceClasses(dataLists.getPriceClassList(), offer);
        return OrderCreateRq.builder()
                .party(createParty())
                .payloadAttributes(createPayloadAttributes(params.getLanguage()))
                .pointOfSale(createPointOfSale(params.getCountryOfSale()))
                .request(OrderCreateRqBody.builder()
                        .createOrder(createPricedOffer(offer, dataLists.getOriginDestList()))
                        .dataLists(DataLists.builder()
                                .contactInfoList(List.of(contactInfo))
                                .originDestList(dataLists.getOriginDestList())
                                .paxJourneyList(toShortJourneys(dataLists.getPaxJourneyList()))
                                .paxList(enrichPaxListWithBookingData(dataLists.getPaxList(), params.getTravellers()))
                                .paxSegmentList(toShortSegments(dataLists.getPaxSegmentList()))
                                .priceClassList(toShortPriceClasses(filteredPriceClasses))
                                .build())
                        .paymentInfo(PaymentInfo.builder()
                                .amount(offer.getTotalPrice().getTotalAmount())
                                .paymentInfoID("PAY1")
                                .paymentMethod(PaymentMethod.createRedirectInstructions(params.getRedirectUrl()))
                                .paymentTrx(PaymentTrx.builder().trxID(params.getTokenizedCard()).build())
                                .typeCode(PaymentType.CREDIT_CARD)
                                .build())
                        .build())
                .build();
    }

    private List<Pax> enrichPaxListWithBookingData(List<Pax> sourcePaxList, List<TravellerInfo> travellerInfoList) {
        List<Pax> paxList = new ArrayList<>();
        Preconditions.checkArgument(sourcePaxList.size() == travellerInfoList.size(),
                "Pax and traveller lists mismatch; %s pax, %s travellers",
                sourcePaxList.size(), travellerInfoList.size());
        for (int i = 0; i < travellerInfoList.size(); i++) {
            TravellerInfo travellerInfo = travellerInfoList.get(i);
            PTC travellerPtc = MappingUtils.getMappingOrThrow(categoryMapping, travellerInfo.getCategory());
            Pax sourcePax = sourcePaxList.get(i);
            Preconditions.checkArgument(travellerPtc == sourcePax.getPtc(),
                    "Passenger type mismatch: expected %s but got %s", sourcePax.getPtc(), travellerPtc);
            Pax.PaxBuilder builder = Pax.builder()
                    .identityDoc(IdentityDoc.builder()
                            .birthdate(travellerInfo.getDateOfBirth())
                            .citizenshipCountryCode(reMapCountryCodes(travellerInfo.getNationalityCode().toUpperCase()))
                            .expiryDate(travellerInfo.getDocumentValidTill())
                            .identityDocID(travellerInfo.getDocumentNumber())
                            // that's the only support document type, birth certificates go with it too
                            .identityDocTypeCode(IdentityDocType.PASSPORT)
                            // at the moment we consider nationality and document issuer country equal
                            .issuingCountryCode(reMapCountryCodes(travellerInfo.getNationalityCode().toUpperCase()))
                            .build())
                    .individual(Individual.builder()
                            .individualID(String.valueOf(i + 1))
                            .givenName(travellerInfo.getFirstName())
                            .middleName(travellerInfo.getMiddleName())
                            .suffixName(!Strings.isNullOrEmpty(travellerInfo.getLastNameSuffix()) ?
                                    SuffixName.forValue(travellerInfo.getLastNameSuffix()) : null)
                            .surname(travellerInfo.getLastName())
                            .genderCode(getMappingOrThrow(sexMapping, travellerInfo.getSex()))
                            .build())
                    .paxID(sourcePax.getPaxID())
                    .ptc(sourcePax.getPtc());
            if (!Strings.isNullOrEmpty(travellerInfo.getLoyaltyProgramCode()) &&
                    !Strings.isNullOrEmpty(travellerInfo.getLoyaltyProgramAccountNumber())) {
                builder.loyaltyProgramAccount(
                        LoyaltyProgramAccount.builder()
                                .loyaltyProgram(
                                        LoyaltyProgram.builder().programCode(travellerInfo.getLoyaltyProgramCode())
                                                .build()
                                )
                                .accountNumber(travellerInfo.getLoyaltyProgramAccountNumber())
                                .build()
                );
            }
            paxList.add(builder.build());
        }
        return paxList;
    }

    public OrderRetrieveRq createOrderRetrieveRq(OrderRetrieveRequestParams params) {
        return OrderRetrieveRq.builder()
                .party(createParty())
                .payloadAttributes(createPayloadAttributes(params.getLanguage()))
                .request(OrderRetrieveRqBody.builder()
                        .orderFilterCriteria(OrderFilterCriteria.builder()
                                .order(new OrderRef(params.getOrderId(), params.getOwnerCode()))
                                .build())
                        .build())
                .build();
    }

    private Party createParty() {
        return Party.builder()
                .sender(Sender.builder()
                        .aggregator(new Aggregator(config.getAggregatorId().get()))
                        .build())
                .build();
    }

    private PayloadAttributes createPayloadAttributes(String language) {
        return PayloadAttributes.builder()
                .primaryLangID(language)
                .versionNumber(config.getApiVersion())
                .build();
    }

    private PointOfSale createPointOfSale(String countryCode) {
        return PointOfSale.builder()
                .country(new Country(countryCode))
                .build();
    }

    private static String reMapCountryCodes(String countryCode) {
        if (Strings.isNullOrEmpty(countryCode)) {
            return countryCode;
        }
        switch (countryCode.toUpperCase()) {
            case "AB":
                // Абхазия -> Грузия
                return "GE";
            case "OS":
                // Южная Осетия -> Россия
                return "RU";
            default:
                return countryCode;
        }
    }
}
