package ru.yandex.avia.booking.partners.gateways.utils;

import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Properties;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;

import static java.util.stream.Collectors.toMap;

@Slf4j
public class TransliterationUtils {
    private static final String ICAO_9303_MAPPING_FILE = "utils/icao_9303_transliteration_mapping.properties";
    private static final Map<Character, String> ICAO_9303_MAPPING = ImmutableMap.copyOf(loadMapping(ICAO_9303_MAPPING_FILE));

    private static Map<Character, String> loadMapping(String mappingFile) {
        var input = TransliterationUtils.class.getClassLoader().getResourceAsStream(mappingFile);
        Preconditions.checkNotNull(input, "No mapping file; path={}", mappingFile);
        try (var ignored = input) {
            Properties mapping = new Properties();
            mapping.load(new InputStreamReader(input, StandardCharsets.UTF_8));
            return mapping.keySet().stream()
                    .map(key -> (String) key)
                    .peek(key -> Preconditions.checkArgument(key.length() == 1,
                            "Illegal mapping key length: key='%s', expected size is 1", key))
                    .collect(toMap(
                            key -> (Character) key.charAt(0),
                            mapping::getProperty
                    ));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String transliterateToLatinSafeIcao9303(String input) {
        if (Strings.isNullOrEmpty(input)) {
            return input;
        }
        StringBuilder converted = new StringBuilder(input.length());
        StringBuilder notMappedLetters = new StringBuilder(0);
        for (char ch : input.toCharArray()) {
            String replacement = ICAO_9303_MAPPING.get(ch);
            if (replacement != null) {
                converted.append(replacement);
            } else {
                if (Character.isLetter(ch) && !isLatin(ch)) {
                    notMappedLetters.append(ch);
                }
                converted.append(ch);
            }
        }
        Preconditions.checkArgument(notMappedLetters.length() == 0,
                "Not mapped non-latin characters detected: '%s' -> '%s'", input, notMappedLetters);
        return converted.toString();
    }

    private static boolean isLatin(char ch) {
        return (ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z');
    }
}
