package ru.yandex.avia.booking.remote.http;

import java.io.IOException;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import lombok.Builder;
import lombok.Data;
import lombok.NonNull;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Dsl;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.avia.booking.remote.RpcContext;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;

@Slf4j
public class HttpClient {

    @Data
    @Builder
    @Accessors(fluent = true)
    static public class ResponseWithStatus {
        private String response;
        private int status;
    }

    private final AsyncHttpClientWrapper httpClient;
    private final HttpClientDefaultHeadersProvider defaultHeadersProvider;

    public HttpClient(@NonNull HttpClientDefaultHeadersProvider defaultHeaders, @NonNull AsyncHttpClientWrapper httpClient) {
        this.httpClient = httpClient;
        this.defaultHeadersProvider = defaultHeaders;
    }

    @NonNull
    public ResponseWithStatus sendRequest(
            @NonNull String endpointUrl, @NonNull String message, Duration socketTimeout,
            @NonNull RpcContext context) throws IOException, TimeoutException, InterruptedException {
        try {
            return sendAsyncRequest(endpointUrl, message, socketTimeout, context)
                    .get(socketTimeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw ExceptionUtils.throwException(cause != null ? cause : e);
        }
    }

    @NonNull
    public CompletableFuture<ResponseWithStatus> sendAsyncRequest(
            @NonNull String endpointUrl,
            @NonNull String message,
            Duration socketTimeout,
            @NonNull RpcContext context) {
        RequestBuilder rb = Dsl.post(endpointUrl)
                .setRequestTimeout((int) socketTimeout.toMillis())
                .setReadTimeout((int) socketTimeout.toMillis())
                .setBody(message);

        for (Map.Entry<String, String> header : defaultHeadersProvider.getHeaders().entrySet()) {
            rb.setHeader(header.getKey(), header.getValue());
        }
        if (context.getExtraHeaders() != null) {
            for (Map.Entry<String, String> headerEntry : context.getExtraHeaders().entrySet()) {
                rb.setHeader(headerEntry.getKey(), headerEntry.getValue());
            }
        }

        return httpClient.executeRequest(rb)
                .thenApply(r -> {
                    String response = r.getResponseBody();
                    // todo(tlg-13): TRAVELBACK-1603: temporary w/a for successful responses with http code 500
                    /*
                    if (r.getStatusCode() >= 300) {
                        // temporary debug, the whole class will be reworked and removed before the final release
                        //System.out.printf("Req:\n%s\nResp:\n%s%n", message, response);
                        throw new RuntimeException("Failed call: code=" + r.getStatusCode() + ", body=" + response);
                    }
                    */
                    return ResponseWithStatus.builder().response(response).status(r.getStatusCode()).build();
                });

    }
}
