package ru.yandex.avia.booking.services.tdapi;

import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.common.base.Preconditions;
import lombok.Builder;
import lombok.NonNull;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
public class AviaTicketDaemonApiClient extends HttpApiClientBase {
    private static final DateTimeFormatter FLIGHT_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter FLIGHT_DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm");

    private final TvmWrapper tvm;
    private final String tvmAlias;
    private final boolean tvmEnabled;

    public AviaTicketDaemonApiClient(
            AsyncHttpClientWrapper aviaApiAhcClient,
            TvmWrapper tvm,
            AviaTicketDaemonApiProperties config
    ) {
        super(aviaApiAhcClient, config, new ObjectMapper()
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE));
        this.tvm = tvm;
        this.tvmAlias = config.getTvmAlias();
        this.tvmEnabled = config.getTvmEnabled();
        if (tvmEnabled && tvm != null) {
            tvm.validateAlias(tvmAlias);
        }
    }

    public void invalidateVariant(InvalidateVariantParams params, InvalidationReason reason) {
        log.info("Invalidating td variant cache for: {}", params);
        Preconditions.checkArgument(params.getForward().size() > 0,
                "at least one segment is expected: %s", params.getForward());
        String forwardDate = params.getForward().get(0).getDeparture().format(FLIGHT_DATE_FORMAT);
        String forwardPath = params.getForward().stream()
                .map(this::formatSegment)
                .collect(Collectors.joining(","));
        URIBuilder pathBuilder = new URIBuilder()
                .setPath("/jsendapi/v3/invalidate_variant")
                .setParameter("national", params.getNationalVersion())
                .setParameter("lang", params.getLang())
                .setParameter("klass", params.getKlass())
                .setParameter("point_from", params.getPointFrom())
                .setParameter("point_to", params.getPointTo())
                .setParameter("adults", String.valueOf(params.getAdults()))
                .setParameter("children", String.valueOf(params.getChildren()))
                .setParameter("infants", String.valueOf(params.getInfants()))
                .setParameter("partners", params.getPartner())
                .setParameter("service", "boy")
                .setParameter("date_forward", forwardDate)
                .setParameter("forward", forwardPath);
        if (InvalidationReason.VARIANT_NOT_AVAILABLE == reason && params.getVariantTag() != null) {
            pathBuilder.setParameter("variant_tag", params.getVariantTag());
        }
        if (params.getBackward() != null && params.getBackward().size() > 0) {
            String backwardDate = params.getBackward().get(0).getDeparture().format(FLIGHT_DATE_FORMAT);
            String backwardPath = params.getBackward().stream()
                    .map(this::formatSegment)
                    .collect(Collectors.joining(","));
            pathBuilder.setParameter("date_backward", backwardDate)
                    .setParameter("backward", backwardPath);
        }
        JsonNode rsp = sync(sendRequest(
                "POST", pathBuilder.toString(), null, JsonNode.class, "invalidate_variant"
        ));
        log.info("invalidate result: {}", rsp);
    }

    private String formatSegment(AviaTicketDaemonSegment segment) {
        // e.g. "SU 46.2019-05-02T00:40"
        return segment.getAirline()
                + " " + removeLeadingZeroes(segment.getFlightNumber())
                + "." + segment.getDeparture().format(FLIGHT_DATE_TIME_FORMAT);
    }

    private String removeLeadingZeroes(String flightNum) {
        return flightNum.replaceAll("^0+", "");
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (tvmEnabled && tvm != null) {
            rb.addHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), tvm.getServiceTicket(tvmAlias));
        }
        return rb;
    }

    @Value
    @Builder
    public static class InvalidateVariantParams {
        @NonNull
        private final String nationalVersion;
        @NonNull
        private final String lang;
        @NonNull
        private final String klass;
        @NonNull
        private final String pointFrom;
        @NonNull
        private final String pointTo;
        private final int adults;
        private final int children;
        private final int infants;
        @NonNull
        private final String partner;
        @NonNull
        private final List<AviaTicketDaemonSegment> forward;
        private final List<AviaTicketDaemonSegment> backward;
        private final String variantTag;
    }
}
