# -*- coding: UTF-8 -*-

try:
    from cStringIO import StringIO
except:
    from io import StringIO

import logging
import urllib
import uuid

import requests

from django.conf import settings
from django.core.files.storage import Storage
from django.utils.deconstruct import deconstructible
from scour import scour


log = logging.getLogger(__name__)
AVATAR_READ_URL = getattr(settings, 'AVATAR_READ_URL', 'https://avatars.mdst.yandex.net')
AVATAR_WRITE_URL = getattr(settings, 'AVATAR_WRITE_URL', 'http://avatars-int.mdst.yandex.net:13000')
AVATAR_NAMESPACE = getattr(settings, 'AVATAR_NAMESPACE', 'avia')


@deconstructible
class AvatarsClient(object):
    def __init__(self, namespace, read_url, write_url, logger):
        self._namespace = namespace
        self._read_url = read_url
        self._write_url = write_url
        self._logger = logger

    def put(self, name, content):
        # Из-за особенностей используемого web-фреймворка, заголовок Content-Disposition
        # обязан содержать атрибут filename (с любым значением).
        headers = {
            'Content-Disposition': 'filename=""',
        }

        files = {
            'file': content,
        }

        url = '{}/put-{}/{}'.format(self._write_url, self._namespace, uuid.uuid4())
        self._logger.debug('Send request to %s', url)

        try:
            r = requests.post(url, headers=headers, files=files)
        except Exception:
            self._logger.exception('Avatar API error')
            raise

        if not r.ok:
            self._logger.error(
                'Cannot upload image to Avatar: [%s] %r %r filename: %r',
                r.status_code, r.request.url, r.content, name
            )
            r.raise_for_status()

        self._logger.debug('MDS response: %r', r.content)
        path = self.extract_path(r.json())

        return self._read_url + path

    @staticmethod
    def extract_path(response):
        return response['sizes']['orig']['path']


@deconstructible
class AvatarsStorage(Storage):
    _url_postfix = None  # type: str

    '''
    @link https://wiki.yandex-team.ru/mds/avatars/
    '''
    def __init__(self):
        self._client = AvatarsClient(AVATAR_NAMESPACE, AVATAR_READ_URL, AVATAR_WRITE_URL, log)

    def _open(self, name, mode='rb'):
        # type: (str, str) -> StringIO
        return StringIO(urllib.urlopen(self.url(name)).read())

    def _save(self, name, content):
        # reopening file (seeking to 0) in case someone already read it
        # see django.core.files.File.open() for more info
        content.open()
        extracted = self._extract_content(content)
        return self._client.put(name, extracted)

    def delete(self, name):
        pass

    def exists(self, name):
        return False

    def listdir(self, path):
        pass

    def size(self, name):
        return None

    def url(self, path):
        # type: (str) -> str
        if self._url_postfix and not path.endswith(self._url_postfix):
            return path + self._url_postfix

        return path


@deconstructible
class AvatarsSvgStorage(AvatarsStorage):
    _url_postfix = '/orig'

    def _extract_content(self, content):
        svg = content.read()
        return self._minimize(svg)

    def _minimize(self, svg):
        # scour has no public API yet
        # https://github.com/scour-project/scour/issues/119
        return scour.scourString(svg, scour.parse_args([
            # scour does not support doctype removal yet
            # https://github.com/scour-project/scour/issues/58
            '--strip-xml-prolog', '--remove-descriptive-elements', '--enable-comment-stripping',
            '--no-line-breaks', '--enable-id-stripping', '--strip-xml-space']))

    # Compability layer
    # See https://st.yandex-team.ru/RASPTICKETS-15971 for details
    def url(self, path):
        # type: (str) -> str
        if self._url_postfix and not path.endswith((self._url_postfix, '/svg')):
            return path + self._url_postfix

        return path


@deconstructible
class AvatarsPngStorage(AvatarsStorage):
    _url_postfix = '/orig'

    def _extract_content(self, content):
        return content.read()

    # Compability layer
    # See https://st.yandex-team.ru/RASPTICKETS-15971 for details
    def url(self, path):
        # type: (str) -> str
        if self._url_postfix and not path.endswith((self._url_postfix, '/png')):
            return path + self._url_postfix

        return path
