# -*- encoding: utf-8 -*-
from logging import getLogger
from retrying import retry

from django.conf import settings
from django.core.management.base import BaseCommand

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache

log = getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        for national_version in settings.AVIA_NATIONAL_VERSIONS:
            if options['only_not_exists'] and CurrencyRatesCache.load(national_version):
                continue
            try:
                cache_rates_for_national_version(national_version)
            except Exception:
                log.exception('Couldn\'t cache rates for "%r"', national_version)

    def add_arguments(self, parser):
        parser.add_argument(
            '--only-not-exists',
            action='store_true',
            help=u'обновляем только если ключа в кеше ещё нет',
        )


@retry(
    wait_exponential_multiplier=200,
    wait_exponential_max=5000,
    stop_max_attempt_number=5,
)
def cache_rates_for_national_version(national_version):
    cache = CurrencyRatesCache.build(national_version=national_version)
    cache.save()

    converter_cache = CurrencyRatesCache.load(
        national_version=national_version,
    )

    if not converter_cache:
        raise RuntimeError('Error loading currency rates from cache after saving')

    log.info(
        'Fetched currency rates for "%s" [%s]: %s [%s]',
        national_version,
        converter_cache.src,
        converter_cache.rates,
        converter_cache.actual_datetime.strftime('%Y-%m-%dT%H:%M:%S'),
    )
