# -*- coding: utf-8 -*-
from __future__ import absolute_import

from datetime import datetime

from django.conf import settings
from django.core.cache import cache as django_cache
from retrying import retry

from travel.avia.library.python.avia_data.libs.currency.rates.converter import fetch_rates
from travel.avia.library.python.avia_data.libs.currency.rates.dohop_converter import fetch_rates as dohop_fetch_rates


class CurrencyRatesCache(object):
    VERSION = 2

    def __init__(self, rates, actual_datetime, national_version, src):
        self.rates = rates
        self.actual_datetime = actual_datetime
        self.national_version = national_version
        self.src = src

    @classmethod
    @retry(
        wait_exponential_multiplier=100,
        wait_exponential_max=500,
        stop_max_attempt_number=3,
    )
    def load(cls, national_version='ru', cache=None):
        if cache is None:
            cache = django_cache
        data = cache.get(cls._get_key(national_version), version=cls.VERSION)
        if data:
            return CurrencyRatesCache(
                national_version=national_version,
                rates=data['rates'],
                actual_datetime=data['actual_datetime'],
                src=data['src'],
            )

        return None

    @classmethod
    def _fetch(cls, currencies_codes, national_version):
        src, rates = fetch_rates(
            currencies_codes,
            settings.NATIONAL_CURRENCY_RATES_GEOID[national_version],
            base_currency_code=(
                settings.AVIA_NATIONAL_CURRENCIES[national_version])
        )

        if src:
            return src, rates

        if national_version == 'com':
            src, rates = dohop_fetch_rates(
                currencies_codes, settings.AVIA_NATIONAL_CURRENCIES[national_version]
            )

        return src, rates

    @classmethod
    def build(cls, national_version='ru', currencies_codes=None):
        if currencies_codes is None:
            from travel.avia.library.python.avia_data.models.currency import Currency
            currencies_codes = (
                Currency.objects.values_list('code', flat=True).distinct())

        src, rates = cls._fetch(
            currencies_codes,
            national_version
        )

        if src is None:
            raise RuntimeError('no data from currency converter')
        return cls(rates, datetime.now(), national_version, src)

    def save(self, cache=None):
        if cache is None:
            cache = django_cache
        # Кэшируем на 7 дней
        cache.set(
            key=self.key,
            value={
                'rates': self.rates,
                'actual_datetime': self.actual_datetime,
                'src': self.src,
            },
            timeout=60 * 60 * 24 * 7,
            version=self.VERSION,
        )

    @property
    def key(self):
        return self._get_key(self.national_version)

    @classmethod
    def _get_key(cls, national_version):
        return settings.CACHEROOT + 'currency_rates/%s' % national_version
