# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import logging
import os

import typing  # noqa
from django.db import migrations

AMADEUS_MERCHANTS_FILE = os.path.join(os.path.dirname(__file__), 'data', 'amadeus_merchants.json')


def _create_amadeus_merchants(apps, schema_editor):
    from travel.avia.admin.lib.logs import add_stdout_handler

    logger = logging.getLogger(b'_create_amadeus_merchants')
    logger.setLevel(logging.DEBUG)
    add_stdout_handler(logger)
    logger.info('Logging configured.')

    Currency = apps.get_model('avia_data', 'Currency')
    AmadeusMerchant = apps.get_model('avia_data', 'AmadeusMerchant')

    def _convert_language(lang):
        return {
            'EN': 'en-EN',
            'RU': 'ru-RU',
        }[lang]

    def _find_currency(currency_code):
        try:
            logger.info('Trying to get currency with iso_code "%s"', currency_code)
            return Currency.objects.get(iso_code=currency_code)
        except Exception:
            logger.exception('Failed to get currency with iso_code "%s"', currency_code)
            return None

    def _parse_merchants():
        # type: () -> typing.Iterable[AmadeusMerchant]
        logger.info('Loading AMADEUS_MERCHANTS_FILE at %s', AMADEUS_MERCHANTS_FILE)
        with open(AMADEUS_MERCHANTS_FILE, 'rb') as amadeus_merchants:
            for merchant_dict in json.load(amadeus_merchants):
                _amadeus_merchant, created = AmadeusMerchant.objects.get_or_create(merchant_id=merchant_dict['merchant_id'])

                _amadeus_merchant.code = merchant_dict['code']
                _amadeus_merchant.title = merchant_dict['title']

                _amadeus_merchant.language_ru = _convert_language(merchant_dict['language_ru'])
                _amadeus_merchant.currency_ru = _find_currency(merchant_dict['currency_ru'])

                _amadeus_merchant.language_com = _convert_language(merchant_dict['language_com'])
                _amadeus_merchant.currency_com = _find_currency(merchant_dict['currency_com'])

                _amadeus_merchant.language_kz = _convert_language(merchant_dict['language_kz'])
                _amadeus_merchant.currency_kz = _find_currency(merchant_dict['currency_kz'])

                logger.info('%s merchant %s from %s', 'Created' if created else 'Updated', _amadeus_merchant.merchant_id, merchant_dict)

                yield _amadeus_merchant

    for amadeus_merchant in _parse_merchants():
        amadeus_merchant.save()


class Migration(migrations.Migration):

    dependencies = [
        ('avia_data', '0060_auto_20190708_1541'),
    ]

    operations = [
        migrations.RunPython(_create_amadeus_merchants, reverse_code=lambda apps, schema_editor: None),
    ]
