# -*- coding: utf-8 -*-
from django.conf import settings
from django.core.validators import MinValueValidator, MaxValueValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Settlement, Station, Country
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.fields import TrimmedCharField

from travel.avia.library.python.avia_data.models.air_traffic_recovery import *  # noqa
from travel.avia.library.python.avia_data.models.amadeus_merchant import *  # noqa
from travel.avia.library.python.avia_data.models.balance import *  # noqa
from travel.avia.library.python.avia_data.models.company import *  # noqa
from travel.avia.library.python.avia_data.models.currency import *  # noqa
from travel.avia.library.python.avia_data.models.directions import *  # noqa
from travel.avia.library.python.avia_data.models.images import *  # noqa
from travel.avia.library.python.avia_data.models.limits import *  # noqa
from travel.avia.library.python.avia_data.models.national_version import *  # noqa
from travel.avia.library.python.avia_data.models.prices import *  # noqa
from travel.avia.library.python.avia_data.models.razladki import *  # noqa
from travel.avia.library.python.avia_data.models.registry import *  # noqa
from travel.avia.library.python.avia_data.models.review import *  # noqa
from travel.avia.library.python.avia_data.models.selfbook import *  # noqa
from travel.avia.library.python.avia_data.models.seo_top import *  # noqa
from travel.avia.library.python.avia_data.models.tablo import *  # noqa
from travel.avia.library.python.avia_data.models.tours import *  # noqa
from travel.avia.library.python.avia_data.models.popular_partners import *  # noqa

RECIPE_TYPES = (
    ('popular', u'Популярные направления'),
    ('prices', u'Минимальные цены'),
)


class AviaRecipe(models.Model):
    L_title = L_field(_(u'Название'), max_length=200)
    slug = models.SlugField(_(u'Адрес'), max_length=32, blank=True, help_text=_(
        u'Латинские буквы и цифры, не более 32, например happy-new-year-2016'
    ))

    L_h1 = L_field(u'H1', max_length=250, blank=True)
    L_meta_title = L_field(_(u'Meta title'), max_length=250, blank=True)
    L_meta_description = L_field(_(u'Meta description'), field_cls=models.TextField, blank=True)

    meta_title_key = models.CharField(_(u'Название ключа в танкере'), max_length=250, blank=True)

    enabled_ru = models.BooleanField(_(u'включить (ru)'), blank=False, null=False, default=False)
    enabled_ua = models.BooleanField(_(u'включить (ua)'), blank=False, null=False, default=False)
    enabled_tr = models.BooleanField(_(u'включить (tr)'), blank=False, null=False, default=False)
    enabled_com = models.BooleanField(_(u'включить (com)'), blank=False, null=False, default=False)
    enabled_kz = models.BooleanField(_(u'включить (kz)'), blank=False, null=False, default=False)

    recipe_type = models.CharField(
        _(u'Тип'), choices=RECIPE_TYPES, max_length=150,
        help_text=u'''
            <strong>Популярные направления</strong> – только известные нам поп. направления из города пользователя<br>
            <strong>Минимальные цены</strong> – по всем минимальным ценам из города пользователя
        ''',
    )
    countries = models.ManyToManyField('www.Country', verbose_name=_(u'Страны'), blank=True)
    regions = models.ManyToManyField('www.Region', verbose_name=_(u'Регионы'), blank=True)
    settlements = models.ManyToManyField('www.Settlement', verbose_name=_(u'Города'), blank=True)
    week_days = models.CharField(_(u'Дни недели'), max_length=50, blank=True, help_text=_(
        u'список через запятую (1 - пн, 2 - вт, ..., 7 - вс)'))
    date_start = models.DateField(_(u'Варианты с'), null=True, blank=True)
    date_end = models.DateField(_(u'Варианты по'), null=True, blank=True)
    backward_date_start = models.DateField(_(u'Варианты обратно с'), null=True, blank=True)
    backward_date_end = models.DateField(_(u'Варианты обратно по'), null=True, blank=True)
    from_regions = models.ManyToManyField(
        'www.Region', blank=True, default=None,
        verbose_name=_(u'Отображать только для регионов'),
        related_name='avia_recipe_from_regions'
    )
    from_settlements = models.ManyToManyField(
        'www.Settlement', blank=True, default=None,
        verbose_name=_(u'отображать только для городов'),
        related_name='avia_recipe_from_settlements'
    )
    show_start = models.DateField(_(u'Показывать с'), null=True, blank=True)
    show_end = models.DateField(_(u'Показывать по'), null=True, blank=True)
    show_year = models.BooleanField(_(u'Учитывать год'), default=True)

    order = models.IntegerField(_(u'Порядок'), default=0)

    def __unicode__(self):
        return self.title_ru or ''

    class Meta:
        verbose_name = _(u'авиа рецепт')
        verbose_name_plural = _(u'авиа рецепты')
        ordering = ['order']
        app_label = 'avia_data'
        db_table = 'avia_aviarecipe'

    @classmethod
    def get_active(cls, city_from, today, national_version):
        national_filter = {}
        national_filter['enabled_%s' % national_version] = True

        qs = cls.objects.filter(
            **national_filter
        ).filter(
            models.Q(from_settlements__isnull=True) | models.Q(from_settlements__in=[city_from]),
        ).extra(where=[
            # Я не смог придумать как здесь обойтись без raw sql
            '''
(show_start IS NULL OR (
    (show_year = True AND show_start <= %s) OR
    (show_year = False AND CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(show_start, '%%m-%%d')) <= %s)
   )
)
            ''',
            '''
(show_end IS NULL OR (
    (show_year = True AND show_end >= %s) OR
    (show_year = False AND CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(show_end, '%%m-%%d')) >= %s)
   )
)
            '''
        ], params=[today.isoformat(), today.isoformat(), today.isoformat(), today.isoformat()])

        if city_from.region_id:
            qs = qs.filter(models.Q(from_regions__isnull=True) |
                           models.Q(from_regions__in=[city_from.region_id]))

        return qs


class NearAirports(models.Model):
    """ FIXME После 3.9 авиабилетов удалить эту таблицу """

    settlement = models.OneToOneField(
        'www.Settlement', verbose_name=_(u'Город'), related_name='near_options'
    )

    min_distance = models.IntegerField(_(u'Минимальная'), null=True, blank=True)
    default_distance = models.IntegerField(_(u'По-умолчанию'), null=True, blank=True, help_text=_(
        u'Чтобы принудительно скрыть блок аэропортов - нужно убрать это значение'
    ))
    max_distance = models.IntegerField(_(u'Максимальная'), null=True, blank=True)

    def __unicode__(self):
        return self.settlement.L_title()

    class Meta:
        verbose_name = _(u'ближайший аэропорт')
        verbose_name_plural = _(u'ближайшие аэропорты')
        app_label = 'avia_data'
        db_table = 'avia_nearairports'


class Rating(models.Model):
    scores = models.IntegerField(verbose_name=_(u'Баллы'))
    good_count = models.IntegerField(verbose_name=_(u'Кол-во хороших'))
    bad_count = models.IntegerField(verbose_name=_(u'Кол-во плохих'))
    bad_percent = models.IntegerField(
        verbose_name=_(u'Процент плохих'),
        db_index=True
    )
    avg_scores = models.FloatField(
        verbose_name=_(u'Средний балл'),
        db_index=True
    )
    outrunning = models.IntegerField(
        verbose_name=_(u'Опережение'),
        null=False, blank=False, default=0
    )
    delayed_less_30 = models.IntegerField(
        verbose_name=_(u'Опоздание < 30'),
        null=False, blank=False, default=0
    )
    delayed_30_60 = models.IntegerField(
        verbose_name=_(u'Опоздание 30-60'),
        null=False, blank=False, default=0
    )
    delayed_60_90 = models.IntegerField(
        verbose_name=_(u'Опоздание 60-90'),
        null=False, blank=False, default=0
    )
    delayed_more_90 = models.IntegerField(
        verbose_name=_(u'Опоздание > 90'),
        null=False, blank=False, default=0
    )
    canceled = models.IntegerField(
        verbose_name=_(u'Отменен'),
        null=False, blank=False, default=0
    )

    class Meta:
        abstract = True


class FlightRating(Rating):
    number = models.CharField(max_length=10, verbose_name=_(u'Рейс'))
    scores_description = models.TextField(
        verbose_name=_(u'Описание'), null=True, blank=True
    )

    class Meta:
        verbose_name = _(u'рейтинг рейса')
        verbose_name_plural = _(u'рейтинги рейсов')
        unique_together = ('number', 'avg_scores')
        ordering = ['avg_scores', 'number']
        app_label = 'avia_data'
        db_table = 'avia_flightrating'


class CompanyRating(Rating):
    company = models.ForeignKey(Company, verbose_name=_(u'Компания'), related_name='+')
    flight_count = models.IntegerField(verbose_name=_(u'Количество рейсов'))

    class Meta:
        verbose_name = _(u'рейтинг авиакомпании')
        verbose_name_plural = _(u'рейтинги авиакомпаний')
        unique_together = ('company',)
        ordering = ['avg_scores']
        app_label = 'avia_data'
        db_table = 'avia_companyrating'


class NearCountries(models.Model):
    country = models.ForeignKey(Country, verbose_name=_(u'Страна'), related_name='near_countries')
    neighbours = models.ManyToManyField(Country, verbose_name=_(u'Соседи'), related_name='+')

    class Meta:
        verbose_name = _(u'соседи страны')
        verbose_name_plural = _(u'соседи стран')
        ordering = ('country__title',)
        app_label = 'avia_data'
        db_table = 'avia_nearcountries'


class BestOffers(models.Model):
    direction = models.CharField(
        max_length=15,
        verbose_name=_(u'Направление'),
        db_index=True
    )

    forward_date = models.DateField(
        verbose_name=_(u'Дата вылета'),
    )

    backward_date = models.DateField(
        verbose_name=_(u'Дата прилета'),
        null=True,

    )

    forward_route = models.CharField(
        max_length=128,
        verbose_name=_(u'Ресы туда'),
        db_index=True
    )

    backward_route = models.CharField(
        max_length=128,
        verbose_name=_(u'Ресы туда'),
        null=True,
        db_index=True
    )

    price = models.IntegerField(
        _(u'Цена'),
        null=True,
        blank=True
    )

    currency = models.ForeignKey(
        'currency.Currency',
        verbose_name=_(u'Валюта'),
        related_name='+',
        null=True
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    class Meta:
        verbose_name = _(u'интересные предложения')
        verbose_name_plural = _(u'интресные предложения')
        app_label = 'avia_data'
        db_table = 'avia_bestoffers'


class SeoDirection(models.Model):
    point_from = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город откуда'),
        related_name='+',
    )
    point_to = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город куда'),
        related_name='+',
    )

    publish = models.BooleanField(_(u'Опубликовать'), blank=False, null=False, default=False)
    text = models.TextField(_(u'Текст RU_ru'), blank=True)
    text_ua_ru = models.TextField(_(u'Текст UA_ru'), blank=True)
    text_ua_uk = models.TextField(_(u'Текст UA_uk'), blank=True)

    class Meta:
        verbose_name = _(u'Сео направление')
        verbose_name_plural = _(u'Сео направления')
        app_label = 'avia_data'
        db_table = 'avia_seodirection'


class CustomCacheTimeDirection(models.Model):
    point_from = models.ForeignKey(Settlement, verbose_name=_(u'Город откуда'),
                                   related_name='+', null=True, blank=True)
    point_to = models.ForeignKey(Settlement, verbose_name=_(u'Город куда'),
                                 related_name='+', null=True, blank=True)
    roundtrip = models.BooleanField(_(u'И обратно'), blank=False, null=False, default=True)

    time = models.IntegerField(_(u'Время кеша в минутах'), default=30, validators=[
        MaxValueValidator(30),
        MinValueValidator(3)
    ])

    enabled = models.BooleanField(_(u'Включен'), blank=False, null=False, default=False)

    def __unicode__(self):
        return u"CustomCacheTimeDirection: %s" % u' - '.join([
            p.L_title() for p in filter(None, [self.point_from, self.point_to])
        ])

    class Meta:
        verbose_name = _(u'направление с кастомным временем жизни')
        verbose_name_plural = _(u'направления с кастомным временем жизни')
        app_label = 'avia_data'
        db_table = 'avia_customcachetimedirection'


class TopFlight(models.Model):
    DAYS_OF_WEEK_CHOICES = tuple(
        [(v, v) for v in range(0, 7)]
    )

    from_point_key = models.CharField(
        blank=False,
        null=False,
        max_length=21,
    )

    to_point_key = models.CharField(
        blank=False,
        null=False,
        max_length=21,
    )

    day_of_week = models.SmallIntegerField(
        null=False,
        blank=False,
        choices=DAYS_OF_WEEK_CHOICES,
    )

    national_version = models.CharField(
        _(u'Тип'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    redirects = models.IntegerField(
        null=False,
        blank=False,
        db_index=True
    )

    flights = models.CharField(
        _(u'Рейсы'),
        max_length=128,
        blank=False,
        null=False,
    )

    class Meta:
        unique_together = (
            'national_version', 'from_point_key', 'to_point_key',
            'day_of_week', 'redirects', 'flights'
        )

        index_together = [
            ('flights', 'day_of_week', 'redirects'),
            ('flights', 'to_point_key', 'from_point_key', 'day_of_week', 'redirects'),
        ]

        ordering = (
            'national_version', 'from_point_key', 'to_point_key',
            'day_of_week', 'redirects'
        )
        verbose_name = _(u'Топ рейс')
        verbose_name_plural = _(u'Топ популярных рейсов')
        app_label = 'avia_data'
        db_table = 'tickets_topflight'


PARTNER_REVIEW_RESULT_CHOICES = (
    ('pass', _(u'Успешно')),
    ('problem', _(u'Проблема')),
    ('error', _(u'Ошибка'))
)


class PartnerReview(models.Model):
    result = models.CharField(
        verbose_name=_(u'Результат проверки'),
        choices=PARTNER_REVIEW_RESULT_CHOICES,
        max_length=12, null=False, blank=False,
    )
    description = TrimmedCharField(_(u'Описание'), max_length=255,
                                   null=False, blank=True, default='')

    partner = models.ForeignKey('order.Partner', null=False, blank=False,
                                verbose_name=_(u'Партнер'))

    hit_time = models.DateTimeField(_(u'Время перехода пользователя'),
                                    blank=False, null=False)

    review_time = models.DateTimeField(_(u'Время проверки'),
                                       blank=False, null=False)

    price = models.FloatField(_(u'Цена на выдаче'), blank=False, null=False)
    price_mbo = models.FloatField(_(u'Цена на сайте партнёра'), blank=True, null=True)
    price_diff_abs = models.FloatField(_(u'Абсолютная разница в цене'), blank=True, null=True)
    price_diff_rel = models.FloatField(_(u'Относительная разница в цене, %'),
                                       blank=True, null=True)

    currency = models.ForeignKey('currency.Currency', null=True, blank=True,
                                 verbose_name=_(u'Валюта'))

    settlement_from = models.ForeignKey(
        Settlement, null=True, blank=True,
        related_name='partner_review_from',
        verbose_name=_(u'Город отправления')
    )

    station_from = models.ForeignKey(
        Station, null=True, blank=True,
        related_name='partner_review_from',
        verbose_name=_(u'Станция отправления')
    )

    settlement_to = models.ForeignKey(
        Settlement, null=True, blank=True,
        related_name='partner_review_to',
        verbose_name=_(u'Город прибытия')
    )

    station_to = models.ForeignKey(
        Station, null=True, blank=True,
        related_name='partner_review_to',
        verbose_name=_(u'Станция прибытия')
    )

    date_forward = models.DateField(_(u'Дата туда'), null=True, blank=True)
    date_backward = models.DateField(_(u'Дата обратно'), null=True, blank=True)
    adults = models.IntegerField(_(u'Взрослых'), null=True, blank=True)
    children = models.IntegerField(_(u'Детей'), null=True, blank=True)
    infants = models.IntegerField(_(u'Младенцев'), null=True, blank=True)

    order_content = models.TextField(_(u'Параметры билета'))
    mbo_data = models.TextField(_(u'Ответ MBO'))
    redirect_params = models.TextField(_(u'Параметры перехода'))
    user_info = models.TextField(_(u'Данные пользователя'))

    class Meta:
        verbose_name = _(u'Проверка страницы партнёра')
        verbose_name_plural = _(u'Проверки страниц партнёров')
        app_label = 'avia_data'
        db_table = 'order_partnerreview'

    def fill_point(self, direction, point):
        # Чтобы в файле моделей не импортировать других моделей,
        # посмотрим на название класса
        if point.__class__.__name__ == 'Settlement':
            setattr(self, 'settlement_%s' % direction, point)

        elif point.__class__.__name__ == 'Station':
            setattr(self, 'station_%s' % direction, point)
