# -*- coding: utf-8 -*-
from django.db import models
from django.utils.translation import ugettext_lazy as _
from enum import Enum

from travel.avia.library.python.common.models.geo import Settlement


class TransportType(Enum):
    plane = 1
    train = 2

    @staticmethod
    def choices():
        return tuple((member.value, name) for name, member in TransportType.__members__.items())


class AirTrafficRecoveryStat(models.Model):
    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='+',
    )

    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='+',
    )

    date_forward = models.DateField(
        verbose_name=_(u'Дата вылета'),
        null=False,
    )

    last_search_date = models.DateField(
        verbose_name=_(u'Дата последнего события о поиске из лога'),
        null=False,
    )

    direct_flights = models.PositiveIntegerField(
        verbose_name=_(u'Актуальное число прямых рейсов'),
        blank=False,
        null=False,
        default=0,
        db_index=False,
    )

    connecting_flights = models.PositiveIntegerField(
        verbose_name=_(u'Актуальное число рейсов с пересадками'),
        blank=False,
        null=False,
        default=0,
        db_index=False,
    )

    last_min_price = models.PositiveIntegerField(
        verbose_name=_(u'Последняя минимальная цена в валюте нац. версии'),
        blank=False,
        null=True,
        default=None,
        db_index=False,
    )

    historical_max_direct_flights = models.PositiveIntegerField(
        verbose_name=_(u'Исторически максимальное число прямых рейсов за последние N дней'),
        blank=False,
        null=False,
        default=0,
        db_index=False,
    )

    historical_max_connecting_flights = models.PositiveIntegerField(
        verbose_name=_(u'Исторически максимальное число рейсов с пересадками за последние N дней'),
        blank=False,
        null=False,
        default=0,
        db_index=False,
    )

    historical_min_price = models.PositiveIntegerField(
        verbose_name=_(u'Исторически минимальная цена в валюте нац. версии за последние N дней'),
        blank=False,
        null=True,
        default=None,
        db_index=False,
    )

    next_direct_flight_day = models.DateField(
        verbose_name=_(u'Дата вылета, в которую есть прямые рейсы, начиная с текущей date_forward'),
        null=True,
    )

    air_traffic_dynamic = models.TextField(
        verbose_name=_(u'JSON-поле для хранения динамики изменения открытости направления за период поисков'),
        null=True,
        blank=True,
    )

    coefficient = models.FloatField(
        verbose_name=_(u'Индекс восстановления авиасообщения по направлению'),
        default=0.
    )

    city_coefficient = models.FloatField(
        verbose_name=_(u'Индекс восстановления авиасообщения для города вылета'),
        default=0.
    )

    transport = models.PositiveSmallIntegerField(
        verbose_name=_(u'Тип транспорта'),
        choices=TransportType.choices(),
        default=TransportType.plane.value,
        blank=False,
        null=False,
    )

    @property
    def min_price_changes(self):
        return round(float(self.last_min_price) / self.historical_min_price, 1)

    class Meta:
        verbose_name = _(u'статистика по восстановлению авиасообщения')
        verbose_name_plural = _(u'статистика по восстановлению авиасообщения')

        app_label = 'avia_data'
        db_table = 'avia_air_traffic_recovery_stat'
        index_together = [
            ('departure_settlement', 'date_forward'),
            ('departure_settlement', 'date_forward', 'transport'),
        ]
