# -*- coding: utf-8 -*-
from __future__ import division

import contextlib
import logging

from django.core.validators import MinValueValidator, MaxValueValidator
from lxml import etree

from django import forms
from django.core.files.base import ContentFile
from django.db import models
from django.utils.deconstruct import deconstructible
from django.utils.translation import ugettext_lazy as _
from simple_history.models import HistoricalRecords

from travel.avia.library.python.common.models.base_partner_model import BasePartner, DEFAULT_CACHE_TTL
from travel.avia.library.python.common.models.partner import ClickPrice, Partner
from travel.avia.library.python.common.models_utils import ConstField
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.image_processing import get_pilimage_content, svg2image

log = logging.getLogger(__name__)


AIRPLANE_T_TYPE_ID = 2

BYTES_IN_MEGABYTE = 2**20

BCP47_MAX_LENGTH = 35  # https://tools.ietf.org/html/rfc5646#section-4.4.1
AVAILABLE_LANGUAGES = [('ru-RU', 'ru-RU'), ('en-EN', 'en-EN'), ('tr-TR', 'tr-TR')]


@deconstructible
class UploadToGetter(object):
    def __init__(self, path, prefix, lang, ext):
        self.path = path
        self.prefix = prefix
        self.lang = lang
        self.ext = ext

    def __call__(self, instance, filename):
        return 'data/{}/{}_{}_{}.{}'.format(
            self.path, self.prefix, instance.id, self.lang, self.ext
        )


class CleanSizeCheckerMixin(object):
    MAX_SIZE = BYTES_IN_MEGABYTE

    def clean(self, *args, **kwargs):
        data = super(CleanSizeCheckerMixin, self).clean(*args, **kwargs)
        if not data.file:
            return data
        if self.MAX_SIZE is not None and data.file.size > self.MAX_SIZE:
            raise forms.ValidationError(
                _(u'Размер файла %s превышает допустимый лимит %.2f МБ') %
                (data.name, self.MAX_SIZE / BYTES_IN_MEGABYTE)
            )
        return data


class CleanSvgMixin(CleanSizeCheckerMixin):
    def clean(self, *args, **kwargs):
        data = super(CleanSvgMixin, self).clean(*args, **kwargs)
        try:
            data.file
        except IOError:
            return data
        if not data.file:
            return data
        try:
            etree.parse(data.file)
        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)
        except Exception as e:
            raise forms.ValidationError(_(u'Формат файла не поддерживается: %s' % e))
        return data


IMAGES_FORMATS_EXTENSIONS = {
    'gif': 'gif',
    'png': 'png',
    'jpeg': 'jpg',
}


class CleanImageMixin(CleanSizeCheckerMixin):
    def clean(self, *args, **kwargs):
        data = super(CleanImageMixin, self).clean(*args, **kwargs)
        if not data.file:
            return data
        # Try to import PIL in either of the two ways it can end up installed.
        try:
            from PIL import Image
        except ImportError:
            import Image
        try:
            image = Image.open(data.file)
        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)
        image_format = image.format.lower()
        try:
            self.file_ext = IMAGES_FORMATS_EXTENSIONS[image_format]
        except KeyError:
            raise forms.ValidationError(_('File type %s not supported.' % image_format))
        return data


class CustomImageField(CleanImageMixin, models.ImageField):
    pass


class SvgImageField(CleanSvgMixin, models.FileField):
    pass


class AmadeusMerchant(BasePartner):

    history = HistoricalRecords(use_base_model_db=True)

    @staticmethod
    def get_LogoSvgUploadToGetter(national_version):
        return UploadToGetter('partners/svg', 'amadeus', national_version, 'svg')

    @staticmethod
    def get_LogoSvg2PngUploadToGetter(national_version):
        return UploadToGetter('partners/svg2png', 'amadeus', national_version, 'png')

    @staticmethod
    def get_IconSvgUploadToGetter():
        return UploadToGetter('partners/icon', 'amadeus', '', 'svg')

    variant_cache_ttl = models.PositiveSmallIntegerField(
        verbose_name=_(u'Время жизни кеша предложений, в минутах'),
        blank=False,
        null=False,
        default=DEFAULT_CACHE_TTL,
        validators=[MinValueValidator(3), MaxValueValidator(180)],
    )

    description_ru_ru = models.TextField(
        verbose_name=_(u'Описание в русской версии'), null=True, blank=True
    )

    description_ua_uk = models.TextField(
        verbose_name=_(u'Описание в украинской версии (uk)'), null=True, blank=True
    )

    description_ua_ru = models.TextField(
        verbose_name=_(u'Описание в украинской версии (ru)'), null=True, blank=True
    )

    description_tr_tr = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (tr)'), null=True, blank=True
    )

    description_tr_en = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (en)'), null=True, blank=True
    )

    description_kz_ru = models.TextField(
        verbose_name=_(u'Описание в казахской версии (ru)'), null=True, blank=True
    )

    merchant_id = TrimmedCharField(_(u'ID мерчанта'), max_length=2, null=False, blank=False, unique=True)

    def save(self, *args, **kwargs):
        self.code = self.get_code_from_merchant_id(self.merchant_id)
        super(AmadeusMerchant, self).save(*args, **kwargs)
        self.process_svg2png_images()

    enabled = models.BooleanField(verbose_name=_(u'Может быть запрошен через демона'), blank=False, null=False, default=False)

    currency_ru = models.ForeignKey(
        'avia_data.Currency', verbose_name=_(u'Валюта в версии ru'),
        null=True, default=None, related_name='amadeus_merchant_ru',
    )
    currency_com = models.ForeignKey(
        'avia_data.Currency', verbose_name=_(u'Валюта в версии com'),
        null=True, default=None, related_name='amadeus_merchant_com',
    )
    currency_kz = models.ForeignKey(
        'avia_data.Currency', verbose_name=_(u'Валюта в версии kz'),
        null=True, default=None, related_name='amadeus_merchant_kz',
    )
    currency_ua = models.ForeignKey(
        'avia_data.Currency', verbose_name=_(u'Валюта в версии ua'),
        null=True, default=None, related_name='amadeus_merchant_ua'
    )
    currency_tr = models.ForeignKey(
        'avia_data.Currency', verbose_name=_(u'Валюта в версии tr'),
        null=True, default=None, related_name='amadeus_merchant_tr'
    )

    def get_currency_by_national_version(self, national_version, default):
        currency = {
            'ru': self.currency_ru.iso_code if self.currency_ru is not None else default,
            'com': self.currency_com.iso_code if self.currency_com is not None else default,
            'kz': self.currency_kz.iso_code if self.currency_kz is not None else default,
            'ua': self.currency_ua.iso_code if self.currency_ua is not None else default,
            'tr': self.currency_tr.iso_code if self.currency_tr is not None else default,
        }.get(national_version, default)
        if currency is not None:
            return currency
        else:
            return default

    language_ru = models.CharField(
        verbose_name=_(u'Язык сайта в версии ru'), blank=True, null=False, default='', max_length=BCP47_MAX_LENGTH,
        choices=AVAILABLE_LANGUAGES,
    )
    language_com = models.CharField(
        verbose_name=_(u'Язык сайта в версии com'), blank=True, null=False, default='', max_length=BCP47_MAX_LENGTH,
        choices=AVAILABLE_LANGUAGES,
    )
    language_kz = models.CharField(
        verbose_name=_(u'Язык сайта в версии kz'), blank=True, null=False, default='', max_length=BCP47_MAX_LENGTH,
        choices=AVAILABLE_LANGUAGES,
    )
    language_ua = models.CharField(
        verbose_name=_(u'Язык сайта в версии ua'), blank=True, null=False, default='', max_length=BCP47_MAX_LENGTH,
        choices=AVAILABLE_LANGUAGES,
    )
    language_tr = models.CharField(
        verbose_name=_(u'Язык сайта в версии tr'), blank=True, null=False, default='', max_length=BCP47_MAX_LENGTH,
        choices=AVAILABLE_LANGUAGES,
    )

    def get_language_by_national_version(self, national_version, default):
        locale = {
            'ru': self.language_ru,
            'com': self.language_com,
            'kz': self.language_kz,
            'ua': self.language_ua,
            'tr': self.language_tr,
        }.get(national_version, default)
        if locale:
            return locale
        else:
            return default

    t_type = models.ForeignKey(
        'www.TransportType', verbose_name=_(u'Тип транспорта'),
        default=AIRPLANE_T_TYPE_ID, related_name='+'
    )

    marker = None
    query_module_name = 'amadeus'

    @classmethod
    def get_code_from_merchant_id(cls, merchant_id):
        return 'amadeus_%s' % merchant_id

    def enabled_in(self, service, national_version, mobile=False):
        if mobile:
            return getattr(self, 'enabled_in_mobile_%s_%s' % (service, national_version), False)
        return getattr(self, 'enabled_in_%s_%s' % (service, national_version), False)

    def get_national_logo_svg(self, national_version):
        field_name = 'logo2_svg_{}'.format(national_version)
        logo = getattr(self, field_name, None)
        return logo.url if logo else None

    def get_national_logo_svg2png(self, national_version):
        field_name = 'logo2_svg2png_{}'.format(national_version)
        logo = getattr(self, field_name, None)
        return logo.url if logo else None

    def get_national_price(self, national_version):
        cache_field = 'clk_price_obj_%s' % national_version
        if not hasattr(self, cache_field):
            setattr(
                self, cache_field,
                ClickPrice(
                    Partner.objects.get(code='amadeus'),
                    national_version
                )
            )
        return getattr(self, cache_field)

    # Заглушка
    def get_staticpages_partner(self, *args, **kwargs):
        return None

    # Заглушка
    def get_national_logo(self, national_version):
        return None

    pricing_model = ConstField('cpa')

    def __unicode__(self):
        return self.code

    @classmethod
    def get_actual(cls, national_version='ru', from_rasp=False, mobile=False):
        merchants = AmadeusMerchant.objects.filter(enabled=True)
        service = 'rasp' if from_rasp else 'ticket'
        return [
            p for p in merchants
            if p.enabled and p.enabled_in(service, national_version, mobile)
        ]

    def process_svg2png_images(self):
        for pngfield, svgfield in [
            (self.logo2_svg2png_ru, self.logo2_svg_ru),
            (self.logo2_svg2png_ua, self.logo2_svg_ua),
            (self.logo2_svg2png_tr, self.logo2_svg_tr),
            (self.logo2_svg2png_com, self.logo2_svg_com),
            (self.logo2_svg2png_kz, self.logo2_svg_kz),
            (self.logo_svg2png_ru, self.logo_svg_ru),
            (self.logo_svg2png_ua, self.logo_svg_ua),
            (self.logo_svg2png_tr, self.logo_svg_tr),
            (self.logo_svg2png_com, self.logo_svg_com),
            (self.logo_svg2png_kz, self.logo_svg_kz),
        ]:
            if svgfield and not pngfield:
                try:
                    svgfield.open()
                    with contextlib.closing(svgfield) as f:
                        svg = f.read()
                    img = svg2image(svg, size=(420, 132))
                    img_content = get_pilimage_content(img)
                    pngfield.save('unused_name', ContentFile(img_content))
                except Exception:
                    log.exception('Error creating svg2png logo')

    class Meta(BasePartner.Meta):
        verbose_name = _(u'Мерчант Amadeus')
        verbose_name_plural = _(u'Мерчанты Amadeus')
        ordering = ('code',)
        app_label = 'avia_data'
        db_table = 'avia_amadeusmerchant'
