# -*- coding: utf-8 -*-
import logging
import re

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.fields import TrimmedCharField

log = logging.getLogger(__name__)


class AviaCompany(models.Model):
    rasp_company = models.OneToOneField(
        'www.Company', on_delete=models.CASCADE, primary_key=True,
    )

    NORMAL_COST = 'normal'
    LOW_COST = 'lowcost'
    HYBRID_COST = 'hybrid'
    COST_TYPES = (NORMAL_COST, LOW_COST, HYBRID_COST)
    cost_type = TrimmedCharField(
        verbose_name=_(u'Бюджет'), max_length=max(len(t) for t in COST_TYPES),
        choices=tuple((code, code.capitalize()) for code in COST_TYPES),
        default='normal', null=False, blank=False,
    )

    baggage_rules = models.TextField(
        _(u'Правила провоза багажа'), null=False, blank=True, default=''
    )
    baggage_rules_are_valid = models.BooleanField(_(u'Правила провоза багажа актуальны'), null=False, default=False)
    baggage_rules_url = TrimmedCharField(
        verbose_name=_(u'Ссылка на правила перевозки багажа на сайте АК'),
        max_length=255, null=False, blank=True, default=''
    )

    baggage_length = models.IntegerField(_(u'Багаж, длина, см'), null=True, blank=True)
    baggage_width = models.IntegerField(_(u'Багаж, ширина, см'), null=True, blank=True)
    baggage_height = models.IntegerField(_(u'Багаж, высота, см'), null=True, blank=True)
    baggage_dimensions_sum = models.IntegerField(_(u'Багаж, сумма измерений, см'), null=True, blank=True)

    carryon_length = models.IntegerField(_(u'Ручная кладь, длина, см'), null=True, blank=True)
    carryon_width = models.IntegerField(_(u'Ручная кладь, ширина, см'), null=True, blank=True)
    carryon_height = models.IntegerField(_(u'Ручная кладь, высота, см'), null=True, blank=True)
    carryon_dimensions_sum = models.IntegerField(_(u'Ручная кладь, сумма измерений, см'), null=True, blank=True)

    baggage_min_price = models.IntegerField(_(u'Стоимость багажа от'), null=True, blank=True)
    baggage_max_price = models.IntegerField(_(u'Стоимость багажа до'), null=True, blank=True)
    baggage_description_url = models.CharField(_(u'ссылка на стоимость провоза багажа'), null=True, blank=True, max_length=255)
    baggage_price_currency = models.ForeignKey('currency.Currency', verbose_name=_(u'Валюта'), related_name='+', null=True, blank=True)

    iata = models.CharField(verbose_name=_(u'код IATA'), max_length=3, null=True, unique=True, blank=True, default=None, db_index=True)

    def __unicode__(self):
        return u'{} {}'.format(self.rasp_company.id, self.rasp_company.title)

    def save(self, *args, **kwargs):
        if not self.iata:
            self.iata = None
        result = super(AviaCompany, self).save(*args, **kwargs)
        self.ensure_has_default_tariff()
        return result

    def ensure_has_default_tariff(self):
        self.tariffs.get_or_create(
            mask='',
            defaults={
                'baggage_allowed': self.cost_type == self.NORMAL_COST,
                'description': _(u'По умолчанию')
            }
        )

    class Meta:
        verbose_name = _(u'Авиакомпании')
        verbose_name_plural = _(u'Авиакомпании')
        app_label = 'avia_data'
        db_table = 'avia_aviacompany'


class CompanyTariff(models.Model):
    avia_company = models.ForeignKey(
        AviaCompany, verbose_name=_(u'Авиакомпания'),
        related_name='tariffs',
        on_delete=models.CASCADE
    )
    description = TrimmedCharField(
        verbose_name=_(u'Описание кода тарифа'), max_length=255, null=False, blank=True, default=''
    )
    mask = TrimmedCharField(verbose_name=_(u'Маска кода тарифа'), max_length=100, null=False, blank=True, default='')
    baggage_allowed = models.BooleanField(verbose_name=_(u'Наличие багажа'), null=False, blank=True, default=True)
    baggage_norm = models.FloatField(_(u'Норма багажа, кг'), null=True, blank=True)
    baggage_pieces = models.IntegerField(_(u'Количество мест'), default=1, null=True, blank=True)
    carryon = models.BooleanField(_(u'Ручная кладь'), null=False, blank=True, default=True)
    carryon_norm = models.FloatField(_(u'Норма ручной клади, кг'), null=True, blank=True)
    published = models.BooleanField(_(u'Опубликован'), null=False, blank=True, default=False)

    @property
    def is_default(self):
        return self.mask == ''

    def mask_match_code(self, code_tariff):
        if not hasattr(self, '_mask_re'):
            try:
                self._mask_re = re.compile(self.mask, re.I + re.U)
            except Exception as err:
                self._mask_re = None
                log.critical(u'CompanyTariff[%s][%s] compile mask %r. %r',
                             self.id, self.description, self.mask, err)
        return bool(self._mask_re.match(code_tariff)) if self._mask_re else False

    def __unicode__(self):
        return u'{} [{}] {}'.format(
            self.avia_company, self.mask, self.description
        ).strip()

    class Meta:
        verbose_name = _(u'Тариф авиакомпании')
        verbose_name_plural = _(u'Тарифы авиакомпании')
        ordering = ('avia_company',)
        app_label = 'avia_data'
        db_table = 'avia_companytariff'
        unique_together = (
            ('avia_company', 'mask'),
        )


class SimilarAirlines(models.Model):
    airline = models.ForeignKey('www.Company', related_name='+')
    similar_airline = models.ForeignKey('www.Company', related_name='+')
    score = models.FloatField(_(u'Мера близости для авиакомпаний'))

    def __unicode__(self):
        return u'SimilarAirlines {}-{} score={}'.format(
            self.airline_id, self.similar_airline_id, self.score
        )

    class Meta:
        verbose_name = _(u'Похожие авиакомпании')
        verbose_name_plural = _(u'Похожие авиакомпании')
        app_label = 'avia_data'
        db_table = 'avia_similar_airlines'
