# -*- coding: utf-8 -*-
from cachetools.func import ttl_cache

from django.db import models
from django.utils import translation
from django.utils.translation import ugettext_lazy as _


@ttl_cache(maxsize=None, ttl=60 * 30)
def _all_currency_translations():
    keyfields = ['currency__code', 'lang__code']
    infofields = ['title', 'title_in', 'template_whole']
    qs = CurrencyTranslation.objects.values(*(keyfields + infofields))
    return {
        tuple([item[k] for k in keyfields]): {k: item[k] for k in infofields}
        for item in qs
    }


def _currency_translation(currency_code, attr, lang=None):
    if not lang:
        lang = translation.get_language()
    translations = _all_currency_translations()
    try:
        return translations[(currency_code, lang)][attr]
    except KeyError:
        return None


class CurrencyLang(models.Model):
    title = models.CharField(
        verbose_name=_(u'Название языка'),
        max_length=100, null=False, blank=False)

    code = models.CharField(
        verbose_name=_(u'Код'),
        max_length=2, null=False, blank=False, db_index=True)

    enable = models.BooleanField(
        verbose_name=_(u'Показать'),
        default=False, null=False, blank=False)

    def __unicode__(self):
        return u'{title} ({code})'.format(
            title=self.title,
            code=self.code
        )

    class Meta:
        ordering = ('code',)
        verbose_name = _(u'Язык валют')
        verbose_name_plural = _(u'Языки валют')
        app_label = 'avia_data'
        db_table = 'avia_currencylang'


class Currency(models.Model):
    title = models.CharField(
        verbose_name=_(u'Название валюты'),
        max_length=100)

    code = models.CharField(
        verbose_name=_(u'Код'),
        max_length=3, null=False, blank=False, db_index=True)

    iso_code = models.CharField(
        verbose_name=_(u'Код ISO 4217'),
        max_length=3, null=False, blank=False, db_index=True)

    priority = models.PositiveSmallIntegerField(
        verbose_name=u'Приоритет сортировки',
        default=0, null=False, blank=False)

    enable = models.BooleanField(
        verbose_name=u'Показать',
        default=False, null=False, blank=False)

    def L_name(self):
        return _currency_translation(self.code, 'title') or self.title

    def L_name_in(self):
        return _currency_translation(self.code, 'title_in')

    def L_template_whole(self):
        return _currency_translation(self.code, 'template_whole')

    def json(self):
        return {
            'code': self.code,
            'iso_code': self.iso_code,
            'name': self.L_name(),
            'template_whole': self.L_template_whole(),
        }

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)
        verbose_name = _(u'Валюта')
        verbose_name_plural = _(u'Валюты')
        app_label = 'avia_data'
        db_table = 'avia_currency'


class CurrencyTranslation(models.Model):
    currency = models.ForeignKey(
        Currency, verbose_name=_(u'Валюта'),
        null=False, blank=False,
        on_delete=models.CASCADE
    )

    lang = models.ForeignKey(
        CurrencyLang, verbose_name=_(u'Язык валюты'),
        null=False, blank=False,
        on_delete=models.PROTECT
    )

    title = models.CharField(
        _(u'Название валюты'), help_text=_(u'например: Рубли'), max_length=100)

    title_in = models.CharField(
        _(u'Название "в"'), help_text=_(u'например: в&nbsp;рублях'),
        max_length=100, blank=True)

    template = models.CharField(
        verbose_name=_(u'Шаблон'),
        max_length=100, null=True, blank=True,
        help_text=u"например: %d&nbsp;р. %d&nbsp;к.")

    template_whole = models.CharField(
        verbose_name=_(u'Шаблон без центов'),
        max_length=100, null=True, blank=True,
        help_text=u"например: %d&nbsp;р. %d&nbsp;к.")

    template_cents = models.CharField(
        verbose_name=_(u'Шаблон центов'),
        max_length=100, null=True, blank=True,
        help_text=u"например: %d&nbsp;р. %d&nbsp;к.")

    class Meta:
        ordering = ('currency',)
        verbose_name = _(u'Перевод для валюты')
        verbose_name_plural = _(u'Переводы для валют')
        app_label = 'avia_data'
        db_table = 'avia_currencytranslation'

    def __unicode__(self):
        return u'{} {}'.format(self.currency, self.lang)
