# -*- coding: utf-8 -*-

from django.db import models
from django.conf import settings
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Settlement


class TopDirection(models.Model):
    u"""Топ популярных направлений"""
    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='top_directions'
    )
    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='arrival_top_directions'
    )
    national_version = models.CharField(
        _(u'Тип'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )
    count = models.PositiveIntegerField(
        blank=False,
        null=False,
        default=0,
        db_index=False
    )

    def __unicode__(self):
        return u'%s - %s' % (
            self.departure_settlement.title,
            self.arrival_settlement.title,
        )

    class Meta:
        unique_together = ('national_version', 'departure_settlement', 'arrival_settlement')
        ordering = ('national_version', 'departure_settlement', 'arrival_settlement')
        verbose_name = _(u'Топ направлений')
        verbose_name_plural = _(u'Топ популярных направлений')
        app_label = 'avia_data'
        db_table = 'www_topdirection'


class NearDirection(models.Model):
    departure_settlement = models.ForeignKey(
        'www.Settlement', verbose_name=_(u'Город отправления'), related_name='nears_departure'
    )

    arrival_settlement = models.ForeignKey(
        'www.Settlement', verbose_name=_(u'Город прибытия'), related_name='near_arrival'
    )

    min_distance = models.IntegerField(_(u'Минимальная'), null=True, blank=True)
    default_distance = models.IntegerField(_(u'По-умолчанию'), null=True, blank=True, help_text=_(
        u'Чтобы принудительно скрыть блок аэропортов - нужно убрать это значение'
    ))
    max_distance = models.IntegerField(_(u'Максимальная'), null=True, blank=True)

    def __unicode__(self):
        return u"Distances %s - %s" % (
            self.departure_settlement.L_title(), self.arrival_settlement.L_title())

    class Meta:
        verbose_name = _(u'ближайший аэропорт')
        verbose_name_plural = _(u'ближайшие аэропорты')
        app_label = 'avia_data'
        db_table = 'avia_neardirection'


class AviaDirection(models.Model):
    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='air_traffic_departure_settlement',
    )

    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='air_traffic_arrival_settlement',
    )

    direct_flights = models.IntegerField(
        verbose_name=_(u'Кол-во прямых рейсов'),
        default=0,
        null=False,
    )

    connecting_flights = models.IntegerField(
        verbose_name=_(u'Кол-во рейсов c пересадками'),
        default=0,
        null=False,
    )

    popularity = models.IntegerField(
        verbose_name=_(u'Популярность'),
        default=0,
        null=False,
    )

    class Meta:
        unique_together = ('departure_settlement', 'arrival_settlement')
        ordering = ['departure_settlement', '-popularity']
        app_label = 'avia_data'
        db_table = 'avia_aviadirection'


class AviaDirectionNational(models.Model):
    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='air_traffic_departure_settlement_national',
    )

    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='air_traffic_arrival_settlement_national',
    )

    direct_flights = models.IntegerField(
        verbose_name=_(u'Кол-во прямых рейсов'),
        default=0,
        null=False,
    )

    connecting_flights = models.IntegerField(
        verbose_name=_(u'Кол-во рейсов c пересадками'),
        default=0,
        null=False,
    )

    popularity = models.IntegerField(
        verbose_name=_(u'Популярность'),
        default=0,
        null=False,
    )

    national_version = models.CharField(
        _(u'Тип'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    class Meta:
        unique_together = ('national_version', 'departure_settlement', 'arrival_settlement')
        ordering = ['national_version', 'departure_settlement', '-popularity']
        app_label = 'avia_data'
        db_table = 'avia_aviadirectionnational'


class AviaSettlement(models.Model):
    settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город'),
        related_name='avia_settlement',
    )

    arrival = models.BooleanField(u'Прибытие', blank=False, null=False, default=False)

    popularity = models.IntegerField(
        verbose_name=_(u'Популярность'),
        default=0,
        null=False,
    )

    class Meta:
        unique_together = ('settlement', 'arrival')
        ordering = ['settlement', '-popularity']
        app_label = 'avia_data'
        db_table = 'avia_aviasettlement'


class AviaSettlementNational(models.Model):
    settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город'),
        related_name='avia_settlement_national',
    )

    arrival = models.BooleanField(u'Прибытие', blank=False, null=False, default=False)

    popularity = models.IntegerField(
        verbose_name=_(u'Популярность'),
        default=0,
        null=False,
    )

    national_version = models.CharField(
        _(u'Тип'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    class Meta:
        unique_together = ('national_version', 'settlement', 'arrival')
        ordering = ['national_version', 'settlement', '-popularity']
        app_label = 'avia_data'
        db_table = 'avia_aviasettlementnational'


class AviaWizardDirection(models.Model):
    settlement_1 = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления или прибытия'),
        related_name='+'
    )
    settlement_2 = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия или отправления'),
        related_name='+'
    )

    class Meta:
        verbose_name = _(u'авианаправление колдунщика')
        verbose_name_plural = _(u'авианаправления колдунщика')

        unique_together = ('settlement_1', 'settlement_2')
        ordering = ['settlement_1__title_ru', 'settlement_1__title',
                    'settlement_2__title_ru', 'settlement_2__title']

        app_label = 'avia_data'
        db_table = 'avia_aviawizarddirection'


class SimilarDirection(models.Model):
    direction = models.CharField(
        max_length=15,
        verbose_name=_(u'Направление'),
        db_index=True
    )

    similar_from = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Похожий город отправления'),
        related_name='+'
    )

    similar_to = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Похожий город прибытия'),
        related_name='+'
    )

    count = models.PositiveIntegerField(
        db_index=True
    )

    class Meta:
        verbose_name = _(u'похожее направление')
        verbose_name_plural = _(u'похожие направления')

        ordering = ['-count']

        app_label = 'avia_data'
        db_table = 'avia_similardirection'
