# -*- coding: utf-8 -*-
import random
from copy import copy

from django.db import models
from django.core.files.uploadedfile import TemporaryUploadedFile
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.caching import CachingManager

from travel.avia.library.python.avia_data.storage import AvatarsPngStorage


class BaseSettlementBigImage(models.Model):
    gradient_css = models.TextField(default='', blank=True, editable=False)

    # Остальные поля объявляются в наследниках, т.к. они либо ведут себя по разному
    # либо это related_field и тогда нельзя использовать один related_name

    @classmethod
    def get_default_image(cls, request):
        """ Задача: я хочу, чтобы они выбирались случайно,
            а те что уже были выбраны не участвовали в выборе повторно,
            до тех пор, пока не закончатся все заглушки. """

        if not hasattr(request, '_default_images'):
            request._default_images = list(cls.objects.filter(settlement__isnull=True))

        if not len(request._default_images):
            return None

        if not hasattr(request, '_default_iter') or not len(request._default_iter):
            request._default_iter = copy(request._default_images)
            random.shuffle(request._default_iter)

        return request._default_iter.pop()

    def __unicode__(self):
        return u'Изображение для %s' % unicode(self.settlement)

    class Meta:
        abstract = True

        verbose_name = _(u'Изображение города')
        verbose_name_plural = _(u'Изображения городов')
        app_label = 'avia_data'
        db_table = 'avia_settlementbigimage'
        ordering = ['settlement']


class SettlementBigImage(BaseSettlementBigImage):
    settlement = models.OneToOneField(
        'www.Settlement', verbose_name=_(u'Город'), related_name='big_image', null=True
    )

    url2 = models.CharField(_(u'Ссылка на изображение в Аватарнице'), max_length=250)

    objects = CachingManager(keys=['settlement_id'], use_get_fallback=False)

    class Meta(BaseSettlementBigImage.Meta):
        # Миграции будут происходить на AdminSettlementBigImage
        managed = False


class AdminSettlementBigImage(BaseSettlementBigImage):
    settlement = models.OneToOneField(
        'www.Settlement', verbose_name=_(u'Город'), related_name='+', null=True
    )

    url2 = models.ImageField(_(u'Ссылка на изображение в Аватарнице'), storage=AvatarsPngStorage(
    ), upload_to='default', max_length=250)

    def postprocess(self, *args, **kwargs):
        if self.url2 and isinstance(self.url2.file, TemporaryUploadedFile):
            from travel.avia.library.python.avia_data.utils.gradify import Gradify  # импорт здесь, чтобы не зависеть от PIL вне админки
            self.gradient_css = Gradify().get_css(self.url2.file.temporary_file_path())

    def save(self, *args, **kwargs):
        self.postprocess()

        return super(AdminSettlementBigImage, self).save(*args, **kwargs)
