# -*- coding: utf-8 -*-
from datetime import timedelta
from urllib import urlencode

from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Settlement, Country
from travel.avia.library.python.common.utils.date import uni_strftime


class MinPrice(models.Model):
    DAYS_OF_WEEK_CHOICES = tuple(
        [(v, v) for v in range(0, 7)]
    )

    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='min_price_departure_settlement'
    )

    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='min_price_arrival_settlement'
    )

    price = models.IntegerField(
        _(u'Минимальная цена'),
        null=True,
        blank=True
    )

    currency = models.ForeignKey(
        'currency.Currency',
        verbose_name=_(u'Валюта'),
        related_name='min_price_currency',
        null=True
    )

    eventtime = models.DateTimeField(
        verbose_name=_(u'Время события'),
        null=True
    )

    date_forward = models.DateField(
        verbose_name=_(u'Дата вылета'),
        null=True
    )

    date_backward = models.DateField(
        verbose_name=_(u'Дата прилета'),
        null=True
    )

    passengers = models.CharField(
        _(u'Пассажиры'),
        null=True,
        max_length=5
    )

    routes = models.TextField(
        _(u'Номера рейсов'),
        null=True
    )

    airlines = models.ManyToManyField(
        'www.Company',
        verbose_name=_(u'Авиакомпании'),
        blank=True)

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    direct_flight = models.NullBooleanField(
        default=None
    )

    day_of_week = models.PositiveSmallIntegerField(
        null=True,
        choices=DAYS_OF_WEEK_CHOICES,
        default=None
    )

    def __unicode__(self):
        return u'%s - %s (%s%s.)' % (
            self.departure_settlement.title,
            self.arrival_settlement.title,
            self.price, u'р.'
        )

    @classmethod
    def passengers_key_from_dict(cls, passengers):
        return '%s_%s_%s' % (
            passengers.get('adults', 0),
            passengers.get('children', 0),
            passengers.get('infants', 0),
        )

    def get_link(self, now=None, oneway=False, when=None, return_date=None):

        params = {
            'fromName': self.departure_settlement.title,
            'fromId': self.departure_settlement.point_key,
            'toName': self.arrival_settlement.title,
            'toId': self.arrival_settlement.point_key,
        }

        if when:
            params['when'] = uni_strftime(u'%d %B', when)
        elif now:
            params['when'] = uni_strftime(u'%d %B', now + timedelta(1))

        if return_date:
            params['return_date'] = uni_strftime(u'%d %B', return_date)

        if oneway:
            params['oneway'] = int(oneway)

        return '/search/?' + urlencode(params)

    class Meta:
        verbose_name = _(u'минимальная цена')
        verbose_name_plural = _(u'минимальные цены')
        unique_together = (
            'arrival_settlement',
            'departure_settlement',
            'national_version',
            'date_forward',
            'date_backward',
            'passengers',
            'direct_flight',
        )
        app_label = 'avia_data'
        db_table = 'www_minprice'


class AvgPrice(models.Model):
    departure_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город отправления'),
        related_name='avg_price_departure_settlement',
        db_index=False
    )

    arrival_settlement = models.ForeignKey(
        Settlement,
        verbose_name=_(u'Город прибытия'),
        related_name='avg_price_arrival_settlement',
        db_index=False
    )

    price = models.IntegerField(
        _(u'Средняя цена'),
        null=True,
        blank=True,
        db_index=False
    )

    currency = models.ForeignKey(
        'currency.Currency',
        verbose_name=_(u'Валюта'),
        related_name='avg_price_currency',
        null=False,
        db_index=False
    )

    month_forward = models.PositiveSmallIntegerField(
        verbose_name=_(u'Месяц вылета'),
        db_index=False
    )

    year_forward = models.PositiveSmallIntegerField(
        verbose_name=_(u'Год вылета'),
        db_index=False
    )

    month_backward = models.PositiveSmallIntegerField(
        verbose_name=_(u'Месяц возвращения'),
        null=True,
        db_index=False
    )

    year_backward = models.PositiveSmallIntegerField(
        verbose_name=_(u'Год возвращения'),
        null=True,
        db_index=False
    )

    passengers = models.CharField(
        _(u'Пассажиры'),
        max_length=5,
        db_index=False
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN,
        db_index=False
    )

    direct_flight = models.BooleanField(
        default=None,
        db_index=False
    )

    def __unicode__(self):
        return u'%s - %s (%s%s.)' % (
            self.departure_settlement.title,
            self.arrival_settlement.title,
            self.price,
            self.currency.code
        )

    class Meta:
        verbose_name = _(u'Средняя цена')
        verbose_name_plural = _(u'Средние цены в гибком поиске')
        unique_together = (
            "national_version", "departure_settlement", "arrival_settlement",
            "month_forward", "year_forward", "month_backward", "year_backward",
            "direct_flight", "passengers"
        )

        app_label = 'avia_data'
        db_table = 'avia_avgprice'


class MinCountryPrice(models.Model):
    DAYS_OF_WEEK_CHOICES = tuple(
        [(v, v) for v in range(0, 7)]
    )

    departure_country = models.ForeignKey(
        Country,
        verbose_name=_(u'Страна отправления'),
        related_name='+'
    )

    arrival_country = models.ForeignKey(
        Country,
        verbose_name=_(u'Страна прибытия'),
        related_name='+'
    )

    price = models.IntegerField(
        _(u'Минимальная цена'),
        null=True,
        blank=True
    )

    currency = models.ForeignKey(
        'currency.Currency',
        verbose_name=_(u'Валюта'),
        related_name='+',
        null=True
    )

    eventtime = models.DateTimeField(
        verbose_name=_(u'Время события'),
        null=True
    )

    date_forward = models.DateField(
        verbose_name=_(u'Дата вылета'),
        null=True
    )

    date_backward = models.DateField(
        verbose_name=_(u'Дата прилета'),
        null=True
    )

    passengers = models.CharField(
        _(u'Пассажиры'),
        null=True,
        max_length=5
    )

    routes = models.CharField(
        _(u'Номера рейсов'),
        null=True,
        max_length=128
    )

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    direct_flight = models.NullBooleanField(
        default=None
    )

    day_of_week = models.PositiveSmallIntegerField(
        null=True,
        choices=DAYS_OF_WEEK_CHOICES,
        default=None
    )

    def __unicode__(self):
        return u'%s - %s (%s%s.)' % (
            self.departure_country.title,
            self.arrival_country.title,
            self.price, u'р.'
        )

    class Meta:
        verbose_name = _(u'Минимальная цена для страны')
        verbose_name_plural = _(u'Минимальные цены для страны на морде города')
        unique_together = (
            "national_version", "departure_country", "arrival_country",
            "date_forward", "date_backward", "passengers", "direct_flight"
        )
        app_label = 'avia_data'
        db_table = 'www_mincountryprice'


class GoodPrice(models.Model):
    DAYS_OF_WEEK_CHOICES = tuple(
        [(v, v) for v in range(0, 7)]
    )

    point_from_type = models.CharField(
        verbose_name=_(u'Тип пункта отправления'),
        max_length=1,
    )

    point_from_id = models.PositiveIntegerField(
        verbose_name=_(u'Идентификатор пункта отправления'),
    )

    point_to_type = models.CharField(
        verbose_name=_(u'Тип пункта прибытия'),
        max_length=1,
    )

    point_to_id = models.PositiveIntegerField(
        verbose_name=_(u'Идентификатор пункта прибытия'),
    )

    route_uid = models.CharField(
        max_length=200,
        verbose_name=_(u'Номера рейсов'),
    )

    departure_weekday = models.PositiveSmallIntegerField(
        verbose_name=_(u'День недели вылета'),
        choices=DAYS_OF_WEEK_CHOICES,
    )

    days_to_flight = models.IntegerField(
        verbose_name=_(u'Дней до вылета'),
    )

    q33 = models.FloatField(
        verbose_name=_(u'33 перцентиль'),
    )

    q67 = models.FloatField(
        verbose_name=_(u'67 перцентиль'),
    )

    def __unicode__(self):
        return u'%s%s - %s%s - %s - %s - %s - (%s - %s)' % (
            self.point_from_type,
            self.point_from_id,
            self.point_to_type,
            self.point_to_id,
            self.route_uid,
            self.departure_weekday,
            self.days_to_flight,
            self.q33,
            self.q67,
        )

    class Meta:
        verbose_name = _(u'хорошая цена')
        verbose_name_plural = _(u'хорошие цены')
        app_label = 'avia_data'
        db_table = 'avia_goodprice'
        unique_together = (
            "point_from_type",
            "point_from_id",
            "point_to_type",
            "point_to_id",
            "departure_weekday",
            "days_to_flight",
            "route_uid",
        )
