# -*- coding: utf-8 -*-

from django.contrib.auth.models import User
from django.core.validators import MaxValueValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models_utils.i18n import L_field


class FlightReviewSource(models.Model):
    L_title = L_field(_(u'название источника'), default=u'', max_length=100, blank=True,
                      add_local_field=True, local_field_critical=True,)
    code = models.CharField(max_length=32, null=False, blank=False)
    enabled = models.BooleanField(verbose_name=_(u'Разрешен'), default=False, db_index=True)

    def __unicode__(self):
        return self.code

    def __str__(self):
        return self.code

    class Meta:
        verbose_name = _(u'источник')
        verbose_name_plural = _(u'источники')
        app_label = 'avia_data'
        db_table = 'avia_flightreviewsource'


class FlightNumber(models.Model):
    airline = models.ForeignKey(
        Company,
        null=True, blank=True, related_name='+',
        verbose_name=_(u'Перевозчик'),
    )

    flight_number = models.CharField(
        max_length=10, null=False, blank=False, db_index=True, default='',
        verbose_name=_(u'Номер рейса')
    )

    def __unicode__(self):
        return '%s (%s)' % (self.id, self.flight_number)

    class Meta:
        verbose_name = _(u'Номер рейса')
        verbose_name_plural = _(u'Номер рейса')
        app_label = 'avia_data'
        db_table = 'avia_flightnumber'


class FlightReview(models.Model):
    review_id = models.CharField(verbose_name=_(u'review_ID'), db_index=True, null=False, blank=False, max_length=64)
    review_content = models.TextField(verbose_name=_(u'Текст отзыва'), null=False, blank=False)
    review_datetime = models.DateTimeField(verbose_name=_(u'Дата отзыва'), null=False,
                                           blank=False, db_index=True)
    review_url = models.CharField(verbose_name=_(u'URL'), max_length=255, null=False, blank=False)

    rating = models.PositiveIntegerField(
        verbose_name=_(u'Рейтинг'),
        validators=[MaxValueValidator(5)],
        null=True, blank=True
    )
    author_name = models.CharField(
        verbose_name=_(u'Имя пользоватя'),
        null=True, blank=True, max_length=255
    )
    author_email = models.EmailField(
        verbose_name=_(u'Email пользователя'), null=True, blank=True
    )
    source = models.ForeignKey(
        FlightReviewSource, null=False, blank=False,
        verbose_name=_(u'Источник'),
        related_name='%(app_label)s_%(class)s_source',
    )
    settlement_from = models.ForeignKey(
        Settlement,
        null=True, blank=True, related_name='%(app_label)s_%(class)s_settlement_from',
        verbose_name=_(u'Город отправления'),
    )
    settlement_to = models.ForeignKey(
        Settlement,
        null=True, blank=True, related_name='%(app_label)s_%(class)s_settlement_to',
        verbose_name=_(u'Город прибытия'),
    )
    station_from = models.ForeignKey(
        Station,
        null=True, blank=True, related_name='%(app_label)s_%(class)s_station_from',
        verbose_name=_(u'Станция отправления'),
    )
    station_to = models.ForeignKey(
        Station,
        null=True, blank=True, related_name='%(app_label)s_%(class)s_station_to',
        verbose_name=_(u'Станция прибытия'),
    )
    airline = models.ForeignKey(
        Company,
        null=False, blank=False, related_name='%(app_label)s_%(class)s_airline',
        verbose_name=_(u'Перевозчик'),
    )

    flight_numbers = models.ManyToManyField('avia_data.FlightNumber', verbose_name=_(u'Номера рейсов'),
                                            blank=True)

    flight_date = models.DateField(verbose_name=_(u'Дата рейса'), null=True, blank=True)

    dirty_lang_class = models.CharField(verbose_name=_(u'Ругань'), max_length=32, null=True, blank=True)

    enable_show = models.NullBooleanField(
        verbose_name=_(u'Разрешить показ'),
        db_index=True
    )
    moderator = models.ForeignKey(
        User,
        null=True, blank=True, related_name='%(app_label)s_%(class)s_moderator',
        verbose_name=_(u'Модератор'),
    )
    moderated_lmt = models.DateTimeField(verbose_name=_(u'Дата модерирования'), null=True, blank=True)

    def __unicode__(self):
        return '%s (%s)' % (self.review_id, self.source)

    class Meta:
        verbose_name = _(u'отзыв')
        verbose_name_plural = _(u'отзывы')
        app_label = 'avia_data'
        db_table = 'avia_flightreview'
