# -*- coding: utf-8 -*-
from django.db import models
from django.utils.translation import ugettext_lazy as _


class SelfBookRule(models.Model):
    description = models.CharField(
        verbose_name=_(u'Описание'), max_length=255)

    exclude_partners = models.BooleanField(_(u'Кроме партнёров'))
    exclude_companies = models.BooleanField(_(u'Кроме компаний'))
    exclude_directions = models.BooleanField(_(u'Кроме направлений'))

    def __unicode__(self):
        return self.description

    class Meta:
        verbose_name = _(u'Правило selfbook')
        verbose_name_plural = _(u'Правила selfbook')
        app_label = 'avia_data'
        db_table = 'avia_selfbookrule'


class SelfBookPartner(models.Model):
    partner = models.ForeignKey(
        'order.Partner', verbose_name=_(u'Партнёр'), related_name='+')

    rule = models.ForeignKey(
        SelfBookRule, verbose_name=_(u'Правило selfbook для партнёра'), related_name='+')

    def __unicode__(self):
        return u'SelfBook [{}] {}'.format(self.partner.code, self.rule.description)

    class Meta:
        verbose_name = _(u'Партнёр правила selfbook')
        verbose_name_plural = _(u'Партнёры правила selfbook')
        app_label = 'avia_data'
        db_table = 'avia_selfbookpartner'
        unique_together = [('rule', 'partner')]


class SelfBookCompany(models.Model):
    rule = models.ForeignKey(
        SelfBookRule, verbose_name=_(u'Правило selfbook для компании'), related_name='+')

    company = models.ForeignKey(
        'www.Company', verbose_name=_(u'Компания'), related_name='+')

    def __unicode__(self):
        return u'SelfBook [{}] {}'.format(self.company.iata, self.rule.description)

    class Meta:
        verbose_name = _(u'Компания правила selfbook')
        verbose_name_plural = _(u'Компании правила selfbook')
        app_label = 'avia_data'
        db_table = 'avia_selfbookcompany'
        unique_together = [('rule', 'company')]


class SelfBookDirection(models.Model):
    rule = models.ForeignKey(
        SelfBookRule, verbose_name=_(u'Правило selfbook для направления'), related_name='+')

    station_from = models.ForeignKey(
        'www.Station', verbose_name=_(u'Аэропорт откуда'), related_name='+',
        null=True, blank=True)

    settlement_from = models.ForeignKey(
        'www.Settlement', verbose_name=_(u'Город откуда'), related_name='+',
        null=True, blank=True)

    country_from = models.ForeignKey(
        'www.Country', verbose_name=_(u'Страна откуда'), related_name='+',
        null=True, blank=True)

    station_to = models.ForeignKey(
        'www.Station', verbose_name=_(u'Аэропорт куда'), related_name='+',
        null=True, blank=True)

    settlement_to = models.ForeignKey(
        'www.Settlement', verbose_name=_(u'Город куда'), related_name='+',
        null=True, blank=True)

    country_to = models.ForeignKey(
        'www.Country', verbose_name=_(u'Страна куда'), related_name='+',
        null=True, blank=True)

    def __unicode__(self):
        def show_airport(a):
            return u'{} {}'.format(a.iata, a) if a else ''

        return u'({}|{}|{}) -> ({}|{}|{})'.format(
            self.country_from or '',
            self.settlement_from or '',
            show_airport(self.station_from),
            self.country_to or '',
            self.settlement_to or '',
            show_airport(self.station_to),
        )

    class Meta:
        verbose_name = _(u'Направление правила selfbook')
        verbose_name_plural = _(u'Направления правила selfbook')
        app_label = 'avia_data'
        db_table = 'avia_selfbookdirection'
        unique_together = [(
            'rule',
            'station_from',
            'settlement_from',
            'country_from',
            'station_to',
            'settlement_to',
            'country_to',
        )]


class SelfBookNationalVersion(models.Model):
    rule = models.ForeignKey(
        SelfBookRule, verbose_name=_(u'Правило selfbook'), related_name='+')

    national_version = models.ForeignKey(
        'avia_data.NationalVersion', verbose_name=_(u'Национальная версия'), related_name='+')

    def __unicode__(self):
        return u'SelfBook {} {}'.format(self.national_version, self.rule.description)

    class Meta:
        verbose_name = _(u'Национальная версия правила selfbook')
        verbose_name_plural = _(u'Национальные версии правила selfbook')
        app_label = 'avia_data'
        db_table = 'avia_selfbooknationalversion'
        unique_together = [('rule', 'national_version')]
