# -*- coding: utf-8 -*-
from django import forms
from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.contrib import admin

from travel.avia.library.python.common.models.transport import TransportType


EVENT_TYPE_CHOICES = (
    ('arrival', 'arrival'),
    ('departure', 'departure')
)


class Tablo(models.Model):
    airport_iata = models.CharField(
        null=False, blank=False, default='',  max_length=3,
        verbose_name=u'Аэропорт'
    )
    event_type = models.CharField(
        null=False, blank=False, choices=EVENT_TYPE_CHOICES, max_length=64,
        verbose_name=u'Тип события',
    )
    flight_number = models.CharField(
        null=False, blank=False, default='', max_length=10,
        verbose_name=u'Номер рейса',
    )
    airline_iata = models.CharField(
        null=False, blank=False, default='',  max_length=3,
        verbose_name=u'Код авиакомпании',
    )
    destination = models.CharField(
        null=False, blank=False, default='',  max_length=128,
        verbose_name=u'Направление',
    )
    date_scheduled = models.DateField(
        null=True, blank=True,
        verbose_name=u'Дата',
    )
    time_scheduled = models.TimeField(
        null=True, blank=True,
        verbose_name=u'Время, расп.',
    )
    time_actual = models.TimeField(
        null=True, blank=True,
        verbose_name=u'Время, факт.',
    )
    status = models.CharField(
        null=False, blank=False, default='',  max_length=64,
        verbose_name=u'Статус',
    )

    class Meta:
        verbose_name = u'Табло'
        verbose_name_plural = u'Табло'
        app_label = 'avia_data'
        db_table = 'avia_tablo'


class TabloSource(models.Model):
    code = models.CharField(
        null=False, blank=False,  max_length=20,
        verbose_name=_(u'Код источника'),
    )
    description = models.CharField(
        null=False, blank=False, default='', max_length=512,
        verbose_name=_(u'Описание источника'),
    )

    class Meta:
        verbose_name = _(u'Источник данных для табло')
        verbose_name_plural = _(u'Источники данных для табло')
        app_label = 'avia_data'
        db_table = 'avia_tablo_source'

    def __unicode__(self):
        return u'[{}] {}{}'.format(self.code, self.description[:32],
                                   u' ...' if len(self.description) > 32 else u'')


class AirportTabloSource(models.Model):
    source = models.ForeignKey(
        TabloSource, verbose_name=_(u'Источник'), related_name='+',
        null=True, blank=True, default=None
    )
    station = models.ForeignKey(
        'www.Station', verbose_name=_(u'Аэропорт'), related_name='+',
        limit_choices_to={'t_type': TransportType.PLANE_ID}
    )
    trusted = models.BooleanField(
        null=False, blank=False, default=False,
        verbose_name=_(u'Проверенный источник'),
    )

    class Meta:
        verbose_name = _(u'Источник табло для Аэропорта')
        verbose_name_plural = _(u'Источники табло для Аэропорта')
        app_label = 'avia_data'
        db_table = 'avia_airport_tablo_source'


class TabloSourceForm(forms.ModelForm):
    description = forms.CharField(widget=forms.Textarea)

    class Meta:
        model = TabloSource
        fields = '__all__'


class TabloSourceAdmin(admin.ModelAdmin):
    form = TabloSourceForm


admin.site.register(TabloSource, TabloSourceAdmin)


def get_change_source_action(tablo_source):
    def change_source(modeladmin, request, queryset):
        queryset.update(source=tablo_source)

    change_source.short_description = _(u'Задать источник {}'.format(tablo_source))
    change_source.__name__ += str(tablo_source.code)
    action = (change_source, change_source.__name__,
              getattr(change_source, 'short_description',
                      change_source.__name__.replace('_', ' ')))
    return change_source.__name__, action


class AirportTabloSourceAdmin(admin.ModelAdmin):
    fieldsets = (
        (None, {
            'fields': ('source', 'station', 'trusted')
        }),
    )

    raw_id_fields = ('station',)
    list_display = ('source', 'get_station_title', 'trusted')
    list_filter = ('source', 'trusted')
    search_fields = ('station__title',)

    def get_station_title(self, obj):
        return obj.station.title

    get_station_title.admin_order_field = 'station'
    get_station_title.short_description = _(u'Название аэропорта')

    def get_actions(self, request):
        actions = super(AirportTabloSourceAdmin, self).get_actions(request)
        actions.update(map(get_change_source_action, TabloSource.objects.all()))
        return actions

admin.site.register(AirportTabloSource, AirportTabloSourceAdmin)
