# -*- coding: utf-8 -*-
import logging
from requests import Session
from requests.adapters import HTTPAdapter
from urllib3 import Retry

from six.moves.urllib_parse import urljoin

logger = logging.getLogger(__name__)


class Client(object):
    def __init__(self, host):
        self._host = host
        self._session = Session()
        self._session.mount('http://', HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
        )))

    def direction(self, national_version, limit=1000000, timeout=5):
        raw_response = self._session.get(
            urljoin(
                self._host,
                'rest/direction?national_version={}&limit={}'.format(national_version, limit)
            ),
            timeout=timeout,
        )
        raw_response.raise_for_status()
        response = raw_response.json()
        if response['status'] != 'ok':
            logger.error('bad avia-backend response: %s', response)
            return []
        return response['data']

    def nearest_settlements(self, settlement_id, lang='ru', max_distance=200, count=10, timeout=20):
        raw_response = self._session.post(
            self._host,
            json=[{
                'name': 'nearestSettlements',
                'params': {
                    'point_id': settlement_id,
                    'max_distance': max_distance,
                    'count': count,
                    'lang': lang,
                },
            }],
            headers={
                'Connection': 'close',
                'Accept': 'application/json',
                'Content-Type': 'application/json',
            },
            timeout=timeout,
        )
        raw_response.raise_for_status()
        response = raw_response.json()
        if response['status'] != 'success':
            logger.error('bad avia-backend response: %s', response)
            return []
        return [s['settlement_id'] for s in response['data'][0]]

    def popular_airlines(self, national_version, lang, limit=-1, timeout=5):
        raw_response = self._session.get(
            urljoin(
                self._host,
                'rest/airlines/popular/{national_version}/{lang}/{limit}'.format(
                    national_version=national_version,
                    lang=lang,
                    limit=limit,
                )
            ),
            timeout=timeout,
        )
        raw_response.raise_for_status()
        response = raw_response.json()
        if response['status'] != 'ok':
            logger.error('bad avia-backend response: %s', response)
            return []
        return response['data']
