# -*- encoding: utf-8 -*-
from __future__ import absolute_import

try:
    import xmlrpclib
except ImportError:
    import xmlrpc.client as xmlrpclib

import logging
from datetime import datetime

from travel.avia.library.python.balance_client.transport import TvmTransport

BALANCE_PRODUCTION_URL = 'https://balance-xmlrpc-tvm.paysys.yandex.net:8004/xmlrpctvm'
BALANCE_TESTING_URL = 'https://balance-xmlrpc-tvm-ts.paysys.yandex.net:8004/xmlrpctvm'

BALANCE_AVIA_SERVICE_ID = 114
BALANCE_AVIA_PRODUCT_ID = 502981


class BalanceException(Exception):
    pass


class Client(object):
    TVM_ALIAS = 'balance'

    """
    Клиент к https://wiki.yandex-team.ru/balance/xmlrpc
    """
    def __init__(self, env, client_tvm_id, logger=None):
        # type: (str, int, logging.Logger) -> None
        self._url = BALANCE_PRODUCTION_URL if env == 'production' else BALANCE_TESTING_URL
        self._logger = logger or logging.getLogger(__name__)

        self._server = xmlrpclib.ServerProxy(
            self._url,
            transport=TvmTransport(self.TVM_ALIAS, client_tvm_id),
        )

    def create_avia_client(self, uid, bid, name, operator_uid):
        """
        Создаем компанию в балансе и связываем ее с нами
        """
        self._logger.info('Method: create_avia_client, Params:  %r', [uid, bid, name, operator_uid])

        try:
            client_id = self._create_client(operator_uid, name)
            self._create_user_client_association(operator_uid, client_id, uid)

            try:
                self._create_or_update_orders_batch(operator_uid, {
                    'ServiceOrderID': bid,
                    'ServiceID': BALANCE_AVIA_SERVICE_ID,
                    'ClientID': client_id,
                    'ProductID': BALANCE_AVIA_PRODUCT_ID,
                })
                return int(client_id)
            except Exception as e:
                try:
                    self._remove_user_client_association(operator_uid, client_id, uid)
                except Exception as e:
                    self._logger.exception(
                        'Can\'t rollback client association (%s): %r', e, [operator_uid, client_id, uid],
                    )
                raise e
        except Exception as e:
            self._logger.exception('%s, Method: create_avia_client, Params: %r', e, [uid, bid, name, operator_uid])

            raise BalanceException('create_avia_client: {}'.format(e))

    def create_avia_helper(self, uid, operator_uid, client_id):
        """
        Создаем связь между пользователем и компанией
        """
        self._logger.info('Method: create_avia_client, Params:  %r', [uid, operator_uid])

        try:
            self._create_user_client_association(operator_uid, client_id, uid)

        except Exception as e:
            self._logger.exception('%s, Method: create_avia_helper, Params: %r', e, [uid, client_id, operator_uid])

            raise BalanceException('create_avia_helper: {}'.format(e))

    def remove_avia_helper(self, operator_uid, client_id, uid):
        """
        Удаляем связь между пользователем и компанией
        """
        self._logger.info('Method: remove_avia_client, Params:  %r', [uid, operator_uid])

        try:
            self._remove_user_client_association(operator_uid, client_id, uid)

        except Exception as e:
            self._logger.exception('%s, Method: remove_avia_helper, Params: %r', e, [uid, client_id, operator_uid])

            raise BalanceException('remove_avia_helper: {}'.format(e))

    def get_passport_by_login(self, operator_uid, login):
        try:
            return self._get_passport_by_login(operator_uid, login)
        except Exception as e:
            self._logger.exception('Error get passport by login: %s', e)
            return None

    def get_orders(self, params):
        try:
            return self._get_orders(params)
        except Exception as e:
            self._logger.exception('Error get passport by login: %s', e)
            return None

    def _create_client(self, operator_uid, name):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient
        """
        method = 'Balance.CreateClient'
        client_info = {
            'SERVICE_ID': BALANCE_AVIA_SERVICE_ID,
            'NAME': name,
        }
        status, status_message, client_id = self.__send_request(method, str(operator_uid), client_info)
        self.__check_status(status, method, operator_uid, client_info)
        return client_id

    def _create_user_client_association(self, operator_uid, client_id, uid):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createuserclientassociation
        """
        method = 'Balance.CreateUserClientAssociation'
        status, response = self.__send_request(method, str(operator_uid), client_id, str(uid))
        self.__check_status(status, method, operator_uid, client_id, uid)

    def _remove_user_client_association(self, operator_uid, client_id, uid):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.removeuserclientassociation
        """
        method = 'Balance.RemoveUserClientAssociation'
        status, response = self.__send_request(method, str(operator_uid), client_id, str(uid))
        self.__check_status(status, method, operator_uid, client_id, uid)
        return response

    def _create_or_update_orders_batch(self, operator_uid, order):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createorupdateordersbatch
        """
        method = 'Balance.CreateOrUpdateOrdersBatch'
        status, response = self.__send_request(method, str(operator_uid), [order])[0]
        self.__check_status(status, method, operator_uid, order)
        return response

    def _get_passport_by_login(self, operator_uid, login):
        """
        https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.getpassportbylogin
        """
        method = 'Balance.GetPassportByLogin'
        response = self.__send_request(method, str(operator_uid), login)

        return response['Uid']

    def _get_orders(self, params):
        """
        https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.getordersinfo
        """
        method = 'GetOrdersInfo'
        return self.__send_request(method, params)

    def __send_request(self, method, *params):
        start_time = datetime.now()
        self._logger.info('Start: Host: %s, Method: %s, Params:  %r', self._url, method, params)

        try:
            return getattr(self._server, method)(*params)

        except Exception as e:
            self._logger.exception('%s, Host: %s, Method: %s, Params:  %r', e, self._url, method, params)
            raise BalanceException('{}: {}'.format(method, e))
        finally:
            seconds = (datetime.now() - start_time).total_seconds()
            self._logger.info('Done: Host: %s, Method: %s, Params:  %r, Time: %d', self._url, method, params, seconds)

    def __check_status(self, status, method, *params):
        if status:
            self._logger.error('Status: %s, Host: %s, Method: %s, Params:  %r', status, self._url, method, params)
            raise BalanceException('{}: {}'.format(method, status))
