# -*- coding: utf-8 -*-
from datetime import datetime  # noqa


CPA = 0.024
INDIRECT_REDIRECT_TYPE = 'inderect'
INDIRECT_WIZARD_REDIRECT_TYPE = 'indirect_wizard'


class UnknownRedirectTypeException(Exception):
    pass


class PriceListCache(object):
    def __init__(self):
        # type: () -> None
        self._price_list = {}
        self._default_click_price = {}
        self._conversions = {}
        self._avg_check_coeffs = {}
        self._minimal_avg_check_coeff = None
        self._maximal_avg_check_coeff = None

    def pre_cache(self):
        from travel.avia.library.python.common.models.partner import (
            ClickPriceMultiplierByRedirectType,
            ConversionByRedirectType,
            DefaultClickPrice,
            MINIMAL_AVG_CHECK_COEFF,
            MAXIMAL_AVG_CHECK_COEFF,
            PriceList,
        )

        # type: () -> None
        default_click_price = dict(DefaultClickPrice.objects.all().values_list(
            'national_version__code', 'billing_price'
        ))

        price_list = {}
        prices = PriceList.objects.all().values(
            'settlement_from_id', 'settlement_to_id',
            'month', 'is_one_way', 'price', 'national_version__code'
        )

        _key = (
            'settlement_from_id', 'settlement_to_id',
            'month', 'is_one_way', 'national_version__code'
        )
        for price in prices:
            price_list[tuple(map(price.__getitem__, _key))] = price['price']

        self._price_list = price_list
        self._default_click_price = default_click_price

        self._conversions = dict(ConversionByRedirectType.objects.filter(
            national_version__code='ru'
        ).all().values_list(
            'redirect_type__code', 'conversion',
        ))

        self._avg_check_coeffs = dict(ClickPriceMultiplierByRedirectType.objects.filter(
            national_version__code='ru'
        ).all().values_list(
            'redirect_type__code', 'multiplier',
        ))

        self._minimal_avg_check_coeff = MINIMAL_AVG_CHECK_COEFF
        self._maximal_avg_check_coeff = MAXIMAL_AVG_CHECK_COEFF

    def get_click_price(
        self, from_id, to_id, national_version, date_from, date_to,
        adults, children, utm_source=None, utm_medium=None, utm_content=None, wizard_redir_key=None,
        use_new_pricing=False,  # RASPTICKETS-14112
    ):
        """
        :return: Стоимость клика с учетом пассажиров и стоимость клика по прайс-листу.
        Стоимость клика преводим из фишкоцентов в фишки
        """
        # type: (int, int, basestring, datetime.date, datetime.date, int, int) -> dict
        key = (
            from_id, to_id, date_from.month, not bool(date_to),
            national_version
        )

        if national_version == 'ru' and key in self._price_list:
            # RASPTICKETS-10933 формула работает только для нового ценообразования которое только в ru
            # RASPTICKETS-11413 формула работает только если цена есть в прайслисте
            price_cpa = self._price_list[key]
            price = price_cpa * (adults + children)
        else:
            price = price_cpa = self._default_click_price[national_version]

        if use_new_pricing and national_version == 'ru':
            # correction_coefficient = 1.1        # RASPTICKETS-17160
            # correction_coefficient = 1.1 * 1.5  # RASPTICKETS-22856
            # correction_coefficient = 1.1 * 1.3  # RASPTICKETS-23090
            correction_coefficient = 1.1 * self._get_avg_check_coeff(  # RASPTICKETS-23164
                utm_source, utm_medium, utm_content, wizard_redir_key,
            )
            price_cpa = price_cpa / 0.9 / 0.1 / 0.02 * CPA * correction_coefficient * self._get_conversion(
                utm_source, utm_medium, utm_content, wizard_redir_key,
            )
            price = price_cpa * (adults + children)

        return self._format_price(price, price_cpa)

    def _format_price(self, price, price_cpa):
        return {
            'price': round(price / 100., 2),
            'price_cpa': round(price_cpa),
        }

    def _get_conversion(self, utm_source, utm_medium, utm_content, wizard_redir_key):
        return float(self._conversions[self._get_redirect_type(utm_source, utm_medium, utm_content, wizard_redir_key)])

    def _get_avg_check_coeff(self, utm_source, utm_medium, utm_content, wizard_redir_key):
        redirect_type = self._get_redirect_type(utm_source, utm_medium, utm_content, wizard_redir_key)
        coeff = self._avg_check_coeffs.get(redirect_type)
        if not coeff:
            # RASPTICKETS-23164: use 'indirect' coeff if 'indirect_wizard' is not available
            if redirect_type == INDIRECT_WIZARD_REDIRECT_TYPE:
                coeff = self._avg_check_coeffs.get(INDIRECT_REDIRECT_TYPE)
            if not coeff:
                raise UnknownRedirectTypeException('unknown redirect type for avg check coeff: '+redirect_type)
        if coeff < self._minimal_avg_check_coeff:
            coeff = self._minimal_avg_check_coeff
        if coeff > self._maximal_avg_check_coeff:
            coeff = self._maximal_avg_check_coeff
        return float(coeff)

    def _get_redirect_type(self, utm_source, utm_medium, utm_content, wizard_redir_key):
        if utm_source == 'rasp' and utm_medium == 'redirect':
            return 'direct_rasp'

        if utm_source == 'sovetnik' and utm_content == 'redirect':
            return 'direct_sovetnik'

        if utm_source == 'wizard_ru' or utm_source == 'unisearch_ru':
            if wizard_redir_key is not None:
                return 'direct_wizard'
            else:
                return INDIRECT_WIZARD_REDIRECT_TYPE

        return INDIRECT_REDIRECT_TYPE


price_list_cache = PriceListCache()
