# -*- coding: utf-8 -*-

import logging
import urllib
import cookiemy

from django.conf import settings
from django.http import HttpResponseRedirect
from django.utils.encoding import smart_unicode
from django.utils.http import urlencode

from travel.avia.library.python.common.geotargeting import lookup
from travel.avia.library.python.common.geotargeting.utils import get_new_tld_host
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.utils.date import astimezone

log = logging.getLogger(__name__)


class Tune(object):
    """
    Генерируем ссылку для перехода в tune.

    Зависимости от других middleware:
    1. Locale
    """

    def process_request(self, request):
        params = request.GET.copy()
        params['domredir'] = '1'
        retpath = request.build_absolute_uri(request.path + '?' + params.urlencode())

        request.tune_url = 'https://{host}/region/?{query}'.format(host=request.tune_host,
                                                                   query=urlencode({'retpath': retpath}))


class Ip(object):
    """
    Записываем в request ip адрес, и флажок о подсети Яндекса (для вставки всяких экспериментов)

    !!!Можно подключать в любом порядке, так как независимая
    """

    def _is_yandex_network(self, ip):
        try:
            return ip and lookup.lookup.is_in(str(ip), 9999)
        except (RuntimeError, ValueError):
            return None

    def process_request(self, request):
        request.client_ip = request.GET.get('ip') or request.META.get('REMOTE_ADDR')
        request.yandex = request.is_yandex_network = self._is_yandex_network(request.client_ip)


class ClientCity(object):
    """
    Определяем:
    client_city_real - город пользователя
    default_city - город пользователя по умолчанию
    client_city - город пользователя в зависимости от нац версии
    geoid - id города в зависимости от нац версии
    client_time - в зависимости от client_city

    Зависимости от других middleware:
    1. Host
    2. Ip
    3. Now
    """

    def _get_client_city(self, request, ip):
        return lookup.get_city(request, ip)

    def _get_default_city(self, domain):
        return Settlement.get_default_for_root_domain(domain)

    def _get_city_domain(self, root_domain, city):
        if root_domain == 'com':
            return 'com'

        try:
            detected_country_domain = city.country.domain_zone or 'ru'
        except AttributeError:
            detected_country_domain = 'ru'

        # Авиабилеты не имеют домена kz, by и тп
        # Поэтому срабатывает не нужная замен города пользователя для ajax запросов
        allowed_domains = getattr(settings, 'ALLOWED_DOMAINS', None)
        if allowed_domains and detected_country_domain not in allowed_domains:
            detected_country_domain = 'ru'

        return detected_country_domain

    def _get_client_city_now(self, request, city):
        return astimezone(request.now, city)

    def process_request(self, request):
        ip = request.client_ip
        client_city_real, geoid = self._get_client_city(request, ip)
        default_city = self._get_default_city(request.root_domain)

        request.client_city_real = client_city_real
        request.geoid = geoid or default_city._geo_id
        request.client_city = client_city_real or default_city
        request.client_city_domain = self._get_city_domain(request.root_domain, request.client_city)
        request.default_city = default_city
        request.client_time = self._get_client_city_now(request, request.client_city)

        # Оверрайд города по нацдомену
        if request.root_domain != request.client_city_domain:
            # !!! client_city_domain - не переопределяем, чтобы редиректы отработали
            request.client_city = request.default_city
            request.geoid = request.client_city._geo_id
            request.client_time = self._get_client_city_now(request, request.client_city)


class RedirectByTune(object):
    """
    При переходе с tune, если пользователь выставил регион,
    отличный от текущего домена, нужно редиректить
    пользователя на нацдомен или .ru при отсутствии оного

    Зависимости от других middleware:
    1. Host
    2. ClientCity
    """

    def _log(self, message):
        log.debug(message)

    def process_request(self, request):
        if not request.GET.get('domredir') or not settings.SUPPORTED_HOSTS:
            return

        self._log('Detected redirect from tune.yandex.ru')
        redirect_host = get_new_tld_host(request.client_city_domain)

        if not redirect_host:
            redirect_host = settings.SUPPORTED_HOSTS[0]
            self._log('Not supported host for tld {tld},'
                      ' redirecting to {redirect_host}'.format(tld=request.client_city_domain,
                                                               redirect_host=redirect_host))

        params = request.GET.copy()
        del params['domredir']
        if 'ncrnd' in params:
            del params['ncrnd']

        redir_path = '//' + redirect_host + request.port + request.path
        if params:
            redir_path += '?' + params.urlencode()

        return HttpResponseRedirect(redir_path)


class RedirectToNationalVersion(object):
    """
    Если город пользователя принадлежит другой нац. версии, то редиректим на нее. Исключение - ru

    Зависимости от других middleware:
    1. Host
    2. ClientCity
    """
    DISABLE_REDIRECT_COOKIEMY_ID = 50  # http://wiki.yandex-team.ru/MyCookie/NomerBloka

    def _redirect_is_disabled(self, request):
        try:
            cookie = urllib.unquote(request.COOKIES['my'])
        except KeyError:
            return False

        setup = cookiemy.Setup()
        try:
            setup.parse(cookie)
        except Exception as e:
            log.warning(u'Ошибка разбора cookiemy {cookie}: {exception}'.format(
                cookie=cookie,
                exception=smart_unicode(e)
            ))
            return False

        disable_redirect = setup.find(self.DISABLE_REDIRECT_COOKIEMY_ID)
        if disable_redirect and disable_redirect[0]:
            return True

    def process_request(self, request):
        if (
            request.root_domain == 'ru' and
            not self._redirect_is_disabled(request) and
            not request.is_ajax()
        ):
            if request.client_city_domain != 'ru':
                redirect_host = get_new_tld_host(request.client_city_domain)

                if redirect_host:
                    return HttpResponseRedirect('//' + redirect_host + request.port +
                                                request.get_full_path())


class GeoTargeting(object):
    """
    Middelware для обратной совместимости, чтобы все нужные middleware были подключены.
    !!! Этот подход не очень хорош => лучше подключать только нужные middleware руками

    Например:
    1. Api - не нужна логика с Tune
    2. Почти всем сервисам не нужна логика с игнорированием этой middleware

    Зависимости от других Middleware:
    1. Все зависимости внутренних.
    """

    def __init__(self):
        self.tune_middleware = Tune()
        self.redirect_by_tune_middleware = RedirectByTune()
        self.redirect_to_national_version_middleware = RedirectToNationalVersion()
        self.ip_middleware = Ip()
        self.client_city_middleware = ClientCity()

    def process_request(self, request):
        request.client_city = None
        request.client_city_real = None

        if request.path.startswith(settings.GEOTARGETING_EXCLUDE_PREFIXES):
            if not request.path.startswith(settings.GEOTARGETING_INCLUDE_PREFIXES):
                return

        self.ip_middleware.process_request(request)
        self.tune_middleware.process_request(request)
        self.client_city_middleware.process_request(request)

        if request.path.startswith(settings.GEOTARGETING_DISABLE_REDIRECT_PREFIXES):
            return

        redirect = self.redirect_by_tune_middleware.process_request(request)
        if redirect:
            return redirect

        redirect = self.redirect_to_national_version_middleware.process_request(request)
        if redirect:
            return redirect
