# -*- coding: utf-8 -*-

import re

from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.fields import RegExpField


@python_2_unicode_compatible
class Express2Country(models.Model):
    code_re = RegExpField(max_length=255, verbose_name=_(u"Регулярное выражение, код экспресса"))
    country = models.ForeignKey('www.Country', verbose_name=_(u"Страна"))
    priority = models.IntegerField(default=0, verbose_name=_(u"Приоритет"),
                                   help_text=_(u"Чем выше приоритет, тем раньше применяется регулярное выражение,"
                                               u" как только мы находим подходящее регулярное выражение, сразу же выходим"))
    time_zone = models.CharField(verbose_name=_(u"Временная зона"),
                                 help_text=_(u"Если нужно указать конкретную временную зону вне зависимости от страны,"
                                             u" даже если указана страна зона возьмется от сюда."),
                                 null=True, blank=True, max_length=30)
    comment = models.TextField(blank=True, null=True, verbose_name=_(u"Комментарий"))

    _precached_variants = None

    class Meta:
        app_label = 'importinfo'
        ordering = ('-priority', 'code_re')
        verbose_name = _(u'Поезда: Код экспресса - Страна')
        verbose_name_plural = _(u'Поезда: Соответсвие временных зон и кодов Экспресса')

    def __str__(self):
        result = self.time_zone

        if result:
            result += u" %s" % self.country.L_title()

        else:
            result = self.country.L_title()

        return u"express~=%s -> %s, p=%s" % (self.code_re, result, self.priority)

    @classmethod
    def get_match_variants(cls):
        for e2c in cls.objects.all():
            regex = re.compile(e2c.code_re, flags=re.U)

            yield regex, e2c

    @classmethod
    def precache(cls):
        cls._precached_variants = [mv for mv in cls.get_match_variants()]

    @classmethod
    def get(cls, express_code):
        if not express_code:
            return

        for regex, e2c in cls._precached_variants or cls.get_match_variants():
            if regex.match(express_code):
                return e2c

    @classmethod
    def get_country(cls, express_code):
        e2c = cls.get(express_code)

        if e2c:
            return e2c.country

    @classmethod
    def get_tz(cls, express_code):
        e2c = cls.get(express_code)

        if not e2c:
            return

        if e2c.time_zone:
            return e2c.time_zone

        return e2c.country.get_capital_tz()
