# coding=utf-8
from logging import getLogger


class CompanyFinder(object):
    """
    Инструмент по нахождению Компаний по:
     1) номеру самолета
     2) коду авиакомпании + номеру самолета
    """

    def __init__(self, companies_by_iata, companies_by_sirena, companies_by_icao, companies_by_icao_ru,
                 corrections_by_iata,
                 company_by_id,
                 logger=None):
        self._company_by_id = company_by_id
        self._companies_by_iata = companies_by_iata
        self._companies_by_sirena = companies_by_sirena
        self._companies_by_icao = companies_by_icao
        self._companies_by_icao_ru = companies_by_icao_ru
        self._corrections_by_iata = corrections_by_iata

        self._logger = logger or getLogger(__name__)

    @staticmethod
    def _select_applicable_correction(flight_number, corrections):
        # Возьмём только номер без кода авиакомпании в начале
        if not flight_number:
            return None

        flight_number = flight_number.split(' ')[-1]

        return next((
            c
            for c in corrections
            if c.match_number(flight_number)
        ), None)

    def _find_by_iata(self, company_code, flight_number):
        correction = self._select_applicable_correction(
            flight_number,
            self._corrections_by_iata.get(company_code, [])
        )

        if correction:
            return self._company_by_id.get(correction.company_id)

        companies = self._companies_by_iata.get(company_code)

        if not companies:
            return None

        if len(companies) != 1:
            self._logger.warn(
                u'Can not resolve company by %s и %s',
                company_code, flight_number
            )

        # список компаний должен быть отсортирован по приоритету
        return companies[0]

    def _find_by_code_system(self, company_code, flight_number, code_system):
        companies = getattr(self, '_companies_by_{}'.format(code_system), None)
        if companies is None:
            raise RuntimeError('Unknown code system "{}"'.format(code_system))

        if not companies:
            return None

        companies_by_code = companies.get(company_code)
        if not companies_by_code:
            return None

        companies_with_iata = [
            c for c in companies_by_code if c.iata
        ]

        for c in companies_with_iata:
            company = self._find_by_iata(c.iata, flight_number)
            if company is not None:
                return company

        if len(companies_by_code) != 1:
            self._logger.warn(
                u'Can not resolve company by %s %s and %s',
                code_system,
                company_code,
                flight_number
            )

        # список компаний должен быть отсортирован по приоритету
        return companies_by_code[0]

    def find(self, company_code, flight_number):
        # пытаемся найти по IATA коду
        company = self._find_by_iata(
            company_code=company_code,
            flight_number=flight_number
        )
        if company:
            return company

        # пытаемся найти по sirena коду
        company = self._find_by_code_system(
            company_code=company_code,
            flight_number=flight_number,
            code_system='sirena',
        )
        if company:
            return company

        # пытаемся найти по icao коду
        company = self._find_by_code_system(
            company_code=company_code,
            flight_number=flight_number,
            code_system='icao',
        )
        if company:
            return company
        company = self._find_by_code_system(
            company_code=company_code,
            flight_number=flight_number,
            code_system='icao_ru',
        )
        if company:
            return company

        return None

    def find_by_flight_number(self, flight_number):
        if not flight_number:
            return None
        parts = flight_number.split(' ')
        if len(parts) != 2:
            self._logger.warn(
                u'Broken flight number [%s]', flight_number)
            return None

        return self.find(parts[0], flight_number)
