# -*- coding: utf-8 -*-

import datetime
import os
import uuid
from functools import partial

from django.conf import settings
from django.core.mail.message import EmailMultiAlternatives


def mail_message(subject, message, recipients, fail_silently=False, connection=None,
                 html_message=None):
    """Sends a message to the admins, as defined by the ADMINS setting."""
    if not recipients:
        return
    mail = EmailMultiAlternatives(settings.EMAIL_SUBJECT_PREFIX + subject, message,
                                  settings.SERVER_EMAIL, [a[1] for a in recipients],
                                  connection=connection)
    if html_message:
        mail.attach_alternative(html_message, 'text/html')
    mail.send(fail_silently=fail_silently)


mail_datachanges = partial(mail_message, recipients=settings.DATACHANGE_RCPNTS)
mail_error404 = partial(mail_message, recipients=settings.ERROR404_RCPNTS)
mail_tablo_error = partial(mail_message, recipients=settings.TABLO_ERROR_RCPNTS)
mail_process = partial(mail_message, recipients=settings.PROCESS_RCPNTS)
mail_free_stations = partial(mail_message, recipients=settings.RASP_FREE_STATIONS_RCPNTS)


def mail_datachanges(subject, message, fail_silently=False, connection=None, html_message=None):
    """Sends a message to the admins, as defined by the ADMINS setting."""
    if not settings.DATACHANGE_RCPNTS:
        return
    mail = EmailMultiAlternatives(settings.EMAIL_SUBJECT_PREFIX + subject, message,
                                  settings.SERVER_EMAIL, [a[1] for a in settings.DATACHANGE_RCPNTS],
                                  connection=connection)
    if html_message:
        mail.attach_alternative(html_message, 'text/html')
    mail.send(fail_silently=fail_silently)


def mail_with_attaches(subject, message, recipients, fail_silently=False, connection=None,
                       html_message=None, files=None):
    """Sends a message to the recipients, with attached files
    files = ((path, mimetype), ...)
    """
    if not recipients:
        return
    mail = EmailMultiAlternatives(
        u'%s%s' % (settings.EMAIL_SUBJECT_PREFIX, subject),
        message, settings.SERVER_EMAIL, [a[1] for a in recipients],
        connection=connection,
    )

    if html_message:
        mail.attach_alternative(html_message, 'text/html')

    if files:
        for path, mimetype in files:
            mail.attach_file(path, mimetype)

    mail.send(fail_silently=fail_silently)


mail_manager_with_attaches = partial(mail_with_attaches, recipients=settings.MANAGERS)
mail_process_with_attaches = partial(mail_with_attaches, recipients=settings.PROCESS_RCPNTS)


class SmailikException(Exception):
    pass


class SmailikClient(object):
    """
    Smailik: https://wiki.yandex-team.ru/smailik
    mail_template - format according to:
        https://wiki.yandex-team.ru/AleksandrKucherenko/smailik
    """

    attachment_template = (u'<part type="{file_mime}">'
                           u'<file>{file_path}</file></part>')

    mail_template = u'''<?xml version="1.0" encoding="UTF-8"?><mails>
        <mail encoding="utf-8">
        <from>{send_from}</from>
        <to>{send_to}</to>
        <replyto>{reply_to}</replyto>
        <content-type>text/html; charset="UTF-8"</content-type>
        <subject encoding="yes">{subject}</subject>
        <parts>
            <part type="text/html"><body style="margin: 0;">{message}</body></part>
            {attachments}
        </parts>
        {custom_headers}
        </mail></mails>'''

    def __init__(self, service, send_from, smailik_dir, attachments_dir=None, reply_to=None):
        self.service = service
        self.send_from = send_from
        self.reply_to = reply_to or send_from
        self.smailik_dir = smailik_dir
        self.attachments_dir = attachments_dir

    @staticmethod
    def _file_name():
        random_number = str(uuid.uuid4())
        current_date = datetime.datetime.now().strftime('%Y_%m_%dT%H_%M_%S_%f')
        return u"{0}_{1}.xml".format(current_date, random_number)

    def generate_email(self, email, subject, body, attachments=None, headers=None, bcc=None):
        headers = dict(headers or {})

        if not isinstance(bcc, (list, tuple)):
            bcc = [bcc]
        bcc = filter(None, bcc)
        if bcc:
            headers['BCC'] = ','.join(bcc)

        email_str = self.mail_template.format(
            send_from=self.send_from,
            send_to=email,
            reply_to=self.reply_to,
            service=self.service,
            subject=subject,
            message=body.replace("&", "&amp;"),
            attachments=self._format_attachments(attachments),
            custom_headers=self._format_custom_headers(headers),
        )
        return email_str

    def send_email(self, email, subject, body, attachments=None, headers=None, bcc=None):
        """
        @param email: str
        @param subject: str
        @param body: str
        @param attachments: list of tuples like ('csv', '/home/user1/att.csv')
        @return:
        """
        email_str = self.generate_email(email, subject, body, attachments, headers, bcc)
        path_to_file = os.path.join(self.smailik_dir, self._file_name())
        try:
            with open(path_to_file, 'w+') as f:
                f.write(email_str.encode('utf8'))
        except Exception as ex:
            raise SmailikException(
                u'Error while sending email: {}, path: {}'.format(
                    str(ex), path_to_file))
        return True

    @classmethod
    def _format_attachments(cls, attachments):
        if not attachments:
            return u''

        result = []
        for file_mime, file_path in attachments:
            result.append(
                cls.attachment_template.format(
                    file_mime=file_mime,
                    file_path=file_path)
            )
        return u''.join(result)

    @classmethod
    def _format_custom_headers(cls, headers):
        if not headers:
            return u''

        return u'<custom>{}</custom>'.format(u''.join([
            u'<header name="{}">{}</header>'.format(header, value)
            for header, value in dict(headers).items()
        ]))

    def path_for_attachment(self, file_name):
        """ Generate a unique path for given file_name. """
        if not self.attachments_dir:
            raise SmailikException(u'attachments_dir must be specified.')

        dir_path = os.path.join(self.attachments_dir, str(uuid.uuid4()))
        if not os.path.exists(dir_path):
            os.makedirs(dir_path)
        return os.path.join(dir_path, file_name)
