# -*- coding: utf-8 -*-
import json
import logging
from collections import OrderedDict
from datetime import datetime
from functools import partial
from itertools import chain
from time import time
from uuid import uuid1


class Timer(object):
    def __init__(self, start_time=None):
        self.start_time = start_time or time()

    def elapsed(self):
        return time() - self.start_time


class Timeline(object):
    def __init__(self, logger, formatter, timer=None, id=None, defaults=None):
        if isinstance(logger, basestring):
            logger = logging.getLogger(logger)
        self._format = formatter
        self._logger = logger
        self._timer = Timer() if timer is None else timer
        self._id = id or str(uuid1())
        self._defaults = defaults if defaults is not None else {}

    def event(self, event_name, **extra):
        items = [
            ('ts', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
            ('tid', self._id),
            ('elapsed_ms', int(self._timer.elapsed() * 1000)),
            ('event', event_name),
        ]

        items = OrderedDict(chain(
            items,  # Чтобы установить значения в начало
            self._defaults.iteritems(),
            extra.iteritems(),
            items  # Чтобы не затёрлись случайно другими значениями
        ))

        message = self._format(items)
        self._logger.info(message)

    def with_defaults(self, **defaults):
        return Timeline(
            self._logger, self._format, timer=self._timer, id=self._id,
            defaults=OrderedDict(self._defaults, **defaults)
        )


class TskvFormat(object):
    def __call__(self, items):
        return '\t'.join(
            '{}={}'.format(k, _somehow_to_str(v)) for k, v in items.iteritems()
        )


class JsonFormat(object):
    def __call__(self, items):
        return json.dumps(items)


def _somehow_to_str(value):
    try:
        return str(value)
    except Exception:
        return json.dumps(value, default=repr)


TskvTimeline = partial(Timeline, formatter=TskvFormat())
JsonTimeline = partial(Timeline, formatter=JsonFormat())
