# -*- coding: utf-8 -*-
import logging
import re
import sys
import tempfile
from contextlib import closing
from datetime import datetime
from optparse import make_option

import requests
from lxml import html
from django.conf import settings
from django.core.management import BaseCommand, CommandError

from travel.avia.library.python.common.utils.progress import PercentageStatus
from travel.avia.library.python.common.utils.dump import (
    database_exists, load_dump_url, load_dump_to_database, log as dump_log,
    purge_database, save_dump_url
)

log = logging.getLogger(__name__)


LISTING_TIMEOUT = 10
DOWNLOAD_TIMEOUT = 5 * 60  # 5 минут
CHUNK_SIZE = 1024 * 1014

FILENAME_PATTERN = re.compile(
    ur'^dump_.*_(?P<db_type>work_db|service_db|migration_db)_(?P<timestamp>\d{14})\.sql\.gz$'
)


def parse_filename(filename):
    match = FILENAME_PATTERN.match(filename)
    if not match:
        return

    return (
        match.group('db_type'),
        datetime.strptime(match.group('timestamp'), '%Y%m%d%H%M%S')
    )


def get_latest_dump_url(base_url, db_type=None):
    listing_html = requests.get(base_url, timeout=LISTING_TIMEOUT, verify=False).text
    doc = html.document_fromstring(listing_html)
    file_link_els = doc.findall('.//a')

    dump_files = []
    for link_el in file_link_els:
        filename = link_el.get('href')
        parsed = parse_filename(filename)
        if parsed:
            dump_db_type, timestamp = parsed
            dump_files.append((filename, dump_db_type, timestamp))

    dump_files.sort(reverse=True, key=lambda x: x[2])
    for filename, dump_db_type, timestamp in dump_files:
        if db_type and dump_db_type != db_type:
            continue

        return base_url + filename


def confirm(message):
    response = raw_input('{} (yes/no): '.format(message))

    while response not in ('yes', 'no'):
        response = raw_input('Please enter either "yes" or "no": ')

    return response == 'yes'


def download_dump(dump_url):
    dump_file = tempfile.NamedTemporaryFile(suffix='-rasp-dump.sql.gz')

    log.info(u'Загружаем %s в %s...', dump_url, dump_file.name)

    resp = requests.head(dump_url, timeout=LISTING_TIMEOUT, verify=False)
    filesize = int(resp.headers['content-length'])
    log.info(u'Размер %s %s', dump_url, filesize)
    download_status = PercentageStatus(filesize, log)

    with closing(
        requests.get(dump_url, timeout=DOWNLOAD_TIMEOUT, stream=True, verify=False)
    ) as resp:
        for chunk in resp.iter_content(CHUNK_SIZE):
            dump_file.write(chunk)
            download_status.step(len(chunk))

    log.info(u'Загрузили %s в %s', dump_url, dump_file.name)

    return dump_file


def update_database_from_dump_url(db_name, dump_url, exclude_tables=()):
    log.info(u'Загружаем дамп %s в базу %s...', dump_url, db_name)

    dump_file = download_dump(dump_url)
    purge_database(db_name)
    load_dump_to_database(dump_file, db_name, exclude_tables)
    save_dump_url(db_name, dump_url)


class Command(BaseCommand):

    args = '<db_name>'
    help = u"Загружает свежий дамп в указанную базу"

    option_list = BaseCommand.option_list + (
        make_option(
            '--noinput',
            action='store_false', dest='interactive', default=True,
            help=u"Ничего не спрашивать в консоли"
        ),
        make_option(
            '--db-type',
            choices=('work_db', 'service_db', 'migration_db'),
            help=u"База для получения дампа"
        ),
        make_option(
            '--full',
            action='store_true',
            help=u"Не пропускать таблицы из LOAD_DB_EXCLUDE_TABLES"
        ),
        make_option(
            '-x', '--exclude-table',
            action='append', dest='exclude_tables',
            help=u"Пропускать таблицу"
        ),
        make_option(
            '--base-url',
            default=settings.LOAD_DB_BASE_URL,
            help=u"URL страницы с файлами дампов"
        ),
    )

    def handle(self, *db_names, **options):
        if len(db_names) != 1:
            raise CommandError("Укажите название базы")

        db_name = db_names[0]

        if int(options['verbosity']) > 1:
            handler = logging.StreamHandler(sys.stdout)
            log.addHandler(handler)
            dump_log.addHandler(handler)

        db_type = options['db_type']
        latest_dump_url = get_latest_dump_url(options['base_url'], db_type)
        if not latest_dump_url:
            raise CommandError("Нет свежего дампа базы {}".format(db_type))

        loaded_dump_url = load_dump_url(db_name)
        if loaded_dump_url == latest_dump_url:
            raise CommandError("Дамп {} уже загружен".format(latest_dump_url))

        if options['interactive'] and (
            database_exists(db_name) and
            not confirm("Содержимое базы {} будет удалено. Продолжить?".format(db_name))
        ):
            sys.exit(1)

        if loaded_dump_url:
            log.info(u'Затираем предыдущий загруженный дамп %s', loaded_dump_url)

        exclude_tables = [] if options['full'] else list(settings.LOAD_DB_EXCLUDE_TABLES)
        if options['exclude_tables']:
            exclude_tables.extend(options['exclude_tables'])

        update_database_from_dump_url(
            db_name, latest_dump_url, exclude_tables=exclude_tables
        )
