# -*- coding: utf-8 -*-
import logging
import os.path
import signal
import sys

from django.conf import settings
from django.core.management import BaseCommand

log = logging.getLogger('gunicorn')


def read_file(fname):
    with open(fname) as f:
        return f.readlines()


def get_master_pid(pidfile, sockfile):
    log.info('Read the master pid.')
    master_pid = int(read_file(pidfile)[0])
    log.info('Check: The master PID is own:')

    cmdline = read_file(os.path.join('/proc/', str(master_pid), 'cmdline'))[0]
    if sockfile not in cmdline:
        return None

    log.info('Check is done')

    log.info('Check: The master PID is parent')

    status = read_file(os.path.join('/proc/', str(master_pid), 'status'))
    ppid = int([line for line in status if line.startswith('PPid:')][0].split('\t')[1])
    if (ppid != 1):
        return

    log.info('Check is done')

    return master_pid


class Command(BaseCommand):
    help = 'Update pre-cache'

    def _handle(self):
        log.info('Start: Update pre-cache')
        log.info('Try to find the master pid...')

        pidfile = (
            settings.GUNICORN_PIDFILE
        )
        sockfile = (
            settings.GUNICORN_PIDFILE
        )

        master_pid = get_master_pid(pidfile, sockfile)
        if not master_pid:
            log.error('The master pid isn\'t found')
            sys.exit(1)

        log.info('The master pid is %d', master_pid)
        log.info('Send a hup signal to the master process %d', master_pid)

        os.kill(master_pid, signal.SIGHUP)

        log.info('Pre-cache is restarted')
        log.info('Finish: Update pre-cache')

    def handle(self, *db_names, **options):
        setup_logger(log, options.get('verbosity'), options.get('add_stdout_handler'))
        try:
            self._handle()
        except Exception, e:
            log.exception('Exception: %r', e)

    def add_arguments(self, parser):
        parser.add_argument(
            '--stdout', action='store_true', default=False,
            dest='add_stdout_handler', help='Add stdout handler',
        )


def setup_logger(logger, verbosity, add_stdout_handler=False):
    loglevel = {
        0: logging.WARN,
        1: logging.INFO,
        2: logging.DEBUG,
        3: logging.DEBUG,
    }[verbosity]
    logger.setLevel(loglevel)

    if add_stdout_handler:
        h = logging.StreamHandler(sys.stdout)
        h.setLevel(loglevel)
        logger.addHandler(h)

    if verbosity > 2:
        logging.getLogger().setLevel(logging.DEBUG)
