# -*- coding: utf-8 -*-

from urllib2 import URLError

from django.conf import settings
from django.contrib.auth.models import User, AnonymousUser
from django.http import HttpResponseRedirect
from django.utils.http import urlencode

from blackbox import XmlBlackbox, BlackboxError, FIELD_LOGIN
from ylog.utils import log_warning

from travel.avia.library.python.common.utils.ya import make_secret_key

# http://wiki.yandex-team.ru/passport/MDAintro#chtotakoemda

FIELD_LANG = ('userinfo.lang.uid', 'lang')
FIELD_STAFF_LOGIN = ('subscription.login.669', 'staff_login')


class YandexUser(object):
    '''
    Модель яндексового пользователя.
    '''

    def __init__(self, uid, is_lite, fields, need_reset):
        self.uid = uid
        self.is_lite_uid = is_lite
        self.fields = fields
        self.need_reset = need_reset

    def is_authenticated(self):
        return not self.is_lite()

    def is_lite(self):
        return bool(self.is_lite_uid)

    def __getattr__(self, field):
        try:
            return self.fields[field]
        except KeyError:
            raise AttributeError(field)

    def get_django_user(self):
        try:
            return User.objects.get(username=self.staff_login)
        except (User.DoesNotExist, AttributeError):
            return AnonymousUser()

    def __str__(self):
        return self.login


class AnonymousYandexUser(object):
    '''
    Заглушка незалогиненного пользователя.
    '''
    login = '(Not logged in)'
    staff_login = '(Not logged in)'
    uid = None
    lang = None

    def is_authenticated(self):
        return False

    def is_lite(self):
        return False

    def __str__(self):
        return self.login

    def get_django_user(self):
        return AnonymousUser()


def add_yauser(request, blackbox):
    """Возвращаемое значение: нужен-ли редирект на MDA"""

    request.yauser = AnonymousYandexUser()

    if settings.DISABLE_YAUTH:
        return False

    sessionid = request.COOKIES.get('Session_id')
    userip = request.META.get('REMOTE_ADDR', None)
    host = request.META.get('HTTP_HOST')
    user_agent = request.META.get('HTTP_USER_AGENT')

    if user_agent and any(bot in user_agent.lower() for bot in BOT_USER_AGENTS):
        return False

    if sessionid is not None:
        try:
            session = blackbox.sessionid(sessionid, userip, host)

            # Кука Session_id есть и валидна
            if session.valid and session.secure:
                if session.uid is not None:
                    request.yauser = YandexUser(int(session.uid), False, session.fields, session.redirect)

                    # Если нужно обновление сессии, то нужен редирект
                    return session.status == 'NEED_RESET'
                else:
                    # lite-пользователь
                    return False

            # Кука Session_id есть, имеет статус NOAUTH, и Чёрный ящик говорит, что её возраст меньше 2-х часов
            if session.status == 'NOAUTH' and session.age < 7200:
                return False

        except (URLError, BlackboxError):  # если blackbox недоступен, то пользователь анонимен
            log_warning()
            return False

    # среди GET-параметров имеется nocookiesupport=yes
    if request.GET.get('nocookiesupport') == 'yes':
        return False

    # На корневом домене не нужно обращаться за MDA
    if host:
        for domain in ROOT_DOMAINS:
            if host.endswith(domain):
                return False

    # Во всех остальных случаях (включая отсутствие куки Session_id) сервис должен обратиться к Пассу с целью синхронизировать (и, возможно, продлить) авторизацию
    return True


# Корневые домены
ROOT_DOMAINS = [
    '.yandex.ru',
    '.yandex.com',
    '.yandex.com.tr',
]

# Подчиненные домены
RU_EXTRA_DOMAINS = [
    '.yandex.ua',
    '.yandex.kz',
    '.yandex.by',
    '.moikrug.ru',
    '.beta.ya.ru',
    '.ya.ru',
]

STANDALONE_DOMAINS = [
    '.yandex.com',
    '.yandex.com.tr',
]

BOT_USER_AGENTS = [bot.lower() for bot in (
    'YandexBot/',
    'YandexRCA/',
    'YandexImages/',
    'YandexMedia/',
    'YandexVideo/',
    'YandexFavicons/',
    'YandexDirect/',
    'YandexCatalog/',
    'YandexMedia/',
    'YandexBlogs/',
    'YandexFavicons/',
    'YandexPagechecker/',
    'YandexNewslinks/',
    'YandexNews/',
    'Yandex/',
    'YandexDirect/',
    'YandexWebmaster/',
    'Googlebot-',
    'Googlebot/',
    'msnbot-',
    'MSNBot/',
    'Mail.RU_Bot/',
    'Mail.RU/',
    'Google Page Speed Insights'
)]


def get_pass_url(hostname):
    if not hostname:
        return None, None

    for domain in ROOT_DOMAINS + RU_EXTRA_DOMAINS:
        if hostname.endswith(domain):
            if len(domain) > 6:
                return 'https://pass%s' % domain, domain
            else:
                return 'https://%s/pass' % hostname, None

    return None, None


class MDAMiddleware:
    def process_request(self, request):
        need_redirect = add_yauser(request, self._get_blackbox())

        request.pass_url, cookie_domain = get_pass_url(request.host)

        request.secret_key = make_secret_key(request)

        if settings.DISABLE_YAUTH:
            return

        if not need_redirect:
            return

        if not request.pass_url:
            return

        if request.method != 'GET':
            return

        if request.is_ajax():
            return

        if cookie_domain in STANDALONE_DOMAINS:
            resign_url = request.pass_url + '/resign'
        else:
            resign_url = request.pass_url + '/'

        redirect_url = resign_url + '?' + urlencode({'retpath': request.build_absolute_uri()})

        response = HttpResponseRedirect(redirect_url)

        response.set_cookie('Cookie_check', 'CheckCookieCheckCookie', domain=cookie_domain)

        return response

    def _get_blackbox(self):
        return XmlBlackbox(
            url=settings.BLACKBOX_URL,
            dbfields=list([FIELD_LOGIN, FIELD_LANG, FIELD_STAFF_LOGIN] + settings.BLACKBOX_PROJECT_FIELDS)
        )
