# -*- coding: utf-8 -*-
from django.core.files.storage import get_storage_class
from django.core.validators import MinValueValidator, RegexValidator
from django.db import models
from django.utils.deconstruct import deconstructible
from django.utils.translation import ugettext_lazy as _
from simple_history.models import HistoricalRecords

from travel.avia.library.python.avia_data.storage import AvatarsPngStorage, AvatarsSvgStorage
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.environment import get_request
from travel.avia.library.python.common.utils.fields import CodeCharField
from travel.avia.library.python.common.utils.media_fields import CustomImageField, SvgImageField


DEFAULT_CACHE_TTL = 20
PARTNER_CODE_REGEX = '^[0-9a-z_]+$'

svg_storage = AvatarsSvgStorage()
png_storage = AvatarsPngStorage()


@deconstructible
class PartnerLogoFileSystemStorage(get_storage_class()):
    def get_available_name(self, name, max_length=None):
        if self.exists(name):
            self.delete(name)

        return name


class BaseVariantsTTL(models.Model):
    class Meta:
        abstract = True
        verbose_name = _(u'время жизни кеша партнера')
        verbose_name_plural = _(u'время жизни кеша партнеров')

    @property
    def partner(self):
        raise NotImplementedError

    search_depth = models.IntegerField(verbose_name=_(u'Глубина поиска, дней'), null=True, blank=True)
    ttl = models.IntegerField(
        verbose_name=_(u'Время жизни кеша, минут'),
        null=False, blank=False,
        validators=[MinValueValidator(3)],
    )

    def __repr__(self):
        return '<{class_}: partner={partner}, search_depth={search_depth}, ttl={ttl}>'.format(
            class_=self.__class__.__name__,
            partner=str(self.partner),
            search_depth=self.search_depth,
            ttl=self.ttl,
        )


class BasePartner(models.Model):

    class Meta:
        abstract = True

    @staticmethod
    def get_LogoSvgUploadToGetter(national_version):
        pass

    @staticmethod
    def get_LogoSvg2PngUploadToGetter(national_version):
        pass

    @staticmethod
    def get_IconSvgUploadToGetter():
        pass

    L_title = L_field(
        _(u'Название партнера'),
        default=u'',
        max_length=100,
        blank=True,
        add_local_field=True,
        local_field_critical=True,
    )

    site_url = models.CharField(verbose_name=_(u'сайт партнера'), max_length=255, null=True)

    code = CodeCharField(
        verbose_name=_(u'код партнера'),
        max_length=100,
        blank=True,
        default=None,
        unique=True,
        null=True,
        validators=[
            RegexValidator(
                PARTNER_CODE_REGEX,
                _(u'Код партнера может содержать только латиницу, цифры и символ подчеркивания')
            ),
        ],
    )

    review_percent = models.IntegerField(_(u'Процент проверок по-умолчанию'), blank=False, null=False, default=10)
    review_percent_ru = models.IntegerField(_(u'Процент проверок для ru'), blank=True, null=True)
    review_percent_kz = models.IntegerField(_(u'Процент проверок для kz'), blank=True, null=True)
    review_percent_ua = models.IntegerField(_(u'Процент проверок для ua'), blank=True, null=True)
    review_percent_com = models.IntegerField(_(u'Процент проверок для com'), blank=True, null=True)
    review_percent_tr = models.IntegerField(_(u'Процент проверок для tr'), blank=True, null=True)

    is_aviacompany = models.BooleanField(
        _(u'Авиакомпания'), blank=False, null=False, default=False
    )

    L_national_ru_title = L_field(
        _(u'Название в русской версии'),
        default=u'', max_length=100, blank=True, base_lang='ru',
        fallback_fields=['L_title']
    )

    L_national_ua_title = L_field(
        _(u'Название в украинской версии'),
        default=u'', max_length=100, blank=True, base_lang='uk',
        fallback_fields=['L_title']
    )

    L_national_kz_title = L_field(
        _(u'Название в казахской версии'),
        default=u'', max_length=100, blank=True, base_lang='ru',
        fallback_fields=['L_title']
    )

    L_national_tr_title = L_field(
        _(u'Название в турецкой версии'),
        default=u'', max_length=100, blank=True, base_lang='tr',
        fallback_fields=['L_title']
    )

    L_national_com_title = L_field(
        _(u'Название в com версии'),
        default=u'', max_length=100, blank=True, base_lang='en',
        fallback_fields=['L_title']
    )

    enabled_in_ticket_ru = models.BooleanField(
        _(u'Запрашивать из КБ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_ua = models.BooleanField(
        _(u'Запрашивать из КБ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_kz = models.BooleanField(
        _(u'Запрашивать из КБ (kz)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_tr = models.BooleanField(
        _(u'Запрашивать из КБ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_com = models.BooleanField(
        _(u'Запрашивать из КБ (com)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_ru = models.BooleanField(
        _(u'Запрашивать из ПМ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_ua = models.BooleanField(
        _(u'Запрашивать из ПМ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_kz = models.BooleanField(
        _(u'Запрашивать из ПМ (kz)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_tr = models.BooleanField(
        _(u'Запрашивать из ПМ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_ru = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_ua = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_kz = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (kz)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_tr = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_com = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (com)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_ru = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_ua = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_kz = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (kz)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_tr = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_wizard_ru = models.BooleanField(
        _(u'Показывать в колдунщике (ru)'), blank=False, null=False, default=True
    )
    enabled_in_wizard_ua = models.BooleanField(
        _(u'Показывать в колдунщике (ua)'), blank=False, null=False, default=True
    )
    enabled_in_wizard_kz = models.BooleanField(
        _(u'Показывать в колдунщике (kz)'), blank=False, null=False, default=True
    )
    enabled_in_wizard_tr = models.BooleanField(
        _(u'Показывать в колдунщике (tr)'), blank=False, null=False, default=True
    )

    media_storage = PartnerLogoFileSystemStorage()

    logo_svg_ru = SvgImageField(
        verbose_name=_(u'Логотип svg (ru)'),
        upload_to=get_LogoSvgUploadToGetter.__func__('ru'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_ua = SvgImageField(
        verbose_name=_(u'Логотип svg (ua)'),
        upload_to=get_LogoSvgUploadToGetter.__func__('ua'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_kz = SvgImageField(
        verbose_name=_(u'Логотип svg (kz)'),
        upload_to=get_LogoSvgUploadToGetter.__func__('kz'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_tr = SvgImageField(
        verbose_name=_(u'Логотип svg (tr)'),
        upload_to=get_LogoSvgUploadToGetter.__func__('tr'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_com = SvgImageField(
        verbose_name=_(u'Логотип svg (com)'),
        upload_to=get_LogoSvgUploadToGetter.__func__('com'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg2png_ru = CustomImageField(
        verbose_name=_(u'Логотип svg2png (ru)'),
        upload_to=get_LogoSvg2PngUploadToGetter.__func__('ru'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_ua = CustomImageField(
        verbose_name=_(u'Логотип svg2png (ua)'),
        upload_to=get_LogoSvg2PngUploadToGetter.__func__('ua'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_kz = CustomImageField(
        verbose_name=_(u'Логотип svg2png (kz)'),
        upload_to=get_LogoSvg2PngUploadToGetter.__func__('kz'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_tr = CustomImageField(
        verbose_name=_(u'Логотип svg2png (tr)'),
        upload_to=get_LogoSvg2PngUploadToGetter.__func__('tr'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_com = CustomImageField(
        verbose_name=_(u'Логотип svg2png (com)'),
        upload_to=get_LogoSvg2PngUploadToGetter.__func__('com'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo2_svg_ru = SvgImageField(
        verbose_name=_(u'Логотип svg (ru) в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    logo2_svg_ua = SvgImageField(
        verbose_name=_(u'Логотип svg (ua) в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    logo2_svg_kz = SvgImageField(
        verbose_name=_(u'Логотип svg (kz) в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    logo2_svg_tr = SvgImageField(
        verbose_name=_(u'Логотип svg (tr) в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    logo2_svg_com = SvgImageField(
        verbose_name=_(u'Логотип svg (com) в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    logo2_svg2png_ru = CustomImageField(
        verbose_name=_(u'Логотип svg2png (ru) в Аватарнице'),
        storage=png_storage,
        null=True, blank=True, default=None
    )

    logo2_svg2png_ua = CustomImageField(
        verbose_name=_(u'Логотип svg2png (ua) в Аватарнице'),
        storage=png_storage,
        null=True, blank=True, default=None
    )

    logo2_svg2png_kz = CustomImageField(
        verbose_name=_(u'Логотип svg2png (kz) в Аватарнице'),
        storage=png_storage,
        null=True, blank=True, default=None
    )

    logo2_svg2png_tr = CustomImageField(
        verbose_name=_(u'Логотип svg2png (tr) в Аватарнице'),
        storage=png_storage,
        null=True, blank=True, default=None
    )

    logo2_svg2png_com = CustomImageField(
        verbose_name=_(u'Логотип svg2png (com) в Аватарнице'),
        storage=png_storage,
        null=True, blank=True, default=None
    )

    icon_svg = SvgImageField(
        verbose_name=_(u'Иконка svg в Аватарнице'),
        storage=svg_storage,
        null=False, blank=True, default=''
    )

    foreign_currency = models.BooleanField(
        verbose_name=_(u'Иностранные валюты'),
        help_text=_(u'Импортировать предложения в валютах, '
                    u'отличных от валюты национальной версии'),
        blank=False, null=False, default=False
    )

    history = HistoricalRecords(use_base_model_db=True)

    def get_icon_svg(self):
        icon = getattr(self, 'icon_svg', None)
        return icon.url if icon else ''

    def get_pricing_model(self):
        return getattr(self, 'pricing_model', None) or 'cpc'

    def get_review_percent(self, language):
        field_name = 'review_percent_' + language
        if hasattr(self, field_name):
            review_percent = getattr(self, field_name)
            if review_percent is not None:
                return review_percent

        return self.review_percent

    def __unicode__(self):
        return self.code

    @property
    def L_national_title(self):
        field_name = 'L_national_{}_title'.format(
            get_request().NATIONAL_VERSION
        )

        return getattr(self, field_name, self.L_title)
