# -*- coding: utf-8 -*-

import httplib
import socket
import urllib2
import warnings

from django.conf import settings
from django.db import models
from django.utils.http import urlencode
from django.utils.translation import ugettext_lazy as _
from django.utils import translation
from lxml import etree
from ylog.utils import log_warning

from travel.avia.library.python.common.utils.caching import CachingManager
from travel.avia.library.python.common.utils.django_utils.meta_helpers import has_field
from travel.avia.library.python.common.utils.fields import CodeCharField
from travel.avia.library.python.common.utils.http import quick_urlopen
from travel.avia.library.python.common.utils.text import NBSP
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning


class L_pseudo_field(object):
    def contribute_to_class(self, cls, name):
        assert name.startswith('L_')

        self.field_name = name[2:]

        self.name = name
        self.model = cls
        self.is_relation = False
        self.auto_created = False
        self.concrete = False
        self.attname = name

        # FIXME:
        # Тут миститка, если есть желание, разберитесь,
        # и прокомментируйте, что тут происходит.
        # cls._meta.add_field(self, virtual=True)
        # без этого тесты прошли, значит это скорее всего не нужно нам.

        # Connect myself
        setattr(cls, name, self)

    def linguistics(self, obj, lang, case='nominative'):
        if lang == 'tr':
            return getattr(obj, '%s_%s' % (self.field_name, lang))

        return getattr(obj, self.field_name)

    class Translations(object):
        def __init__(self, field, obj):
            self.field = field
            self.obj = obj

        def __call__(self, case='nominative', lang=None):
            return self.field.linguistics(self.obj, lang or translation.get_language(), case)

    def __get__(self, instance, instance_type=None):
        if instance is None:
            return self

        return self.Translations(self, instance)

    def __set__(self, instance, value):
        if instance is None:
            raise AttributeError(u"%s must be accessed via instance" % self.name)

        if isinstance(value, self.Translations):
            for field in self.local_fields():
                field_value = getattr(value.obj, field)

                setattr(instance, field, field_value)

        else:
            raise ValueError("%r assignment is not supported" % type(value))


class Currency(models.Model):
    name = models.CharField(_(u"Название"), help_text=_(u"например: Рубли"), max_length=100)
    code = CodeCharField(_(u"Код"), max_length=3)
    iso_code = CodeCharField(_(u"Код в формате ISO 4217"), max_length=3, blank=True, default=None)
    name_in = models.CharField(_(u'Название "в"'), help_text=_(u"например: в&nbsp;рублях"), max_length=100, blank=True)

    name_uk = models.CharField(_(u'Украинское название'), help_text=_(u"например: Гривні"), max_length=100, blank=True)
    name_uk_in = models.CharField(_(u'Украинское название "в"'), help_text=_(u"например: в&nbsp;гривнях"),
                                  max_length=100, blank=True)

    L_name = L_pseudo_field()
    L_template = L_pseudo_field()
    L_template_whole = L_pseudo_field()
    L_template_cents = L_pseudo_field()

    template = CodeCharField(_(u"Шаблон отображения"), help_text=_(u"например: %d&nbsp;р. %d&nbsp;к."), max_length=100)
    template_whole = CodeCharField(_(u"Шаблон без центов"), help_text=_(u"например: %d&nbsp;р."), max_length=100)
    template_cents = CodeCharField(_(u"Шаблон центов"), help_text=_(u"например: %d&nbsp;к."), max_length=100)

    name_tr = models.CharField(_(u"Турецкое название"), help_text=_(u"например: Рубли"), max_length=100, blank=True)
    template_tr = CodeCharField(_(u"Турецкий шаблон отображения"), help_text=_(u"например: %d&nbsp;р. %d&nbsp;к."),
                                max_length=100, blank=True)
    template_whole_tr = CodeCharField(_(u"Турецкий шаблон без центов"), help_text=_(u"например: %d&nbsp;р."),
                                      max_length=100, blank=True)
    template_cents_tr = CodeCharField(_(u"Турецкий шаблон центов"), help_text=_(u"например: %d&nbsp;к."),
                                      max_length=100, blank=True)

    order = models.IntegerField(blank=False, null=False, default=0)
    order_tr = models.IntegerField(blank=False, null=False, default=0)
    order_ua = models.IntegerField(blank=False, null=False, default=0)

    objects = CachingManager(keys=['pk', 'code'])

    def __unicode__(self):
        return self.name

    def __repr__(self):
        return '<Currency {}>'.format(self.code)

    class Meta:
        ordering = ('order',)
        verbose_name = _(u'валюта')
        verbose_name_plural = _(u'валюты')
        app_label = 'currency'
        db_table = 'currency_currency'

    BASE_CURRENCY = 'RUR'

    def format_value(self, *args, **kwargs):
        try:
            return self._format_value(*args, **kwargs)
        except:
            return 'n/a'

    def _format_value(self, value, show_cents=True):
        CENTS_PER_WHOLE = 100  # количество копеек/центов в рубле/долларе
        SHOW_CENTS_LIMIT = 50  # RASPFRONT-1329, показывать центы при цене меньше 50 единиц

        sign = -1 if value < 0 else 1

        abs_value = value * sign
        total_cents = int(round(abs_value * CENTS_PER_WHOLE))
        sign_string = '-' if sign < 0 and total_cents > 0 else ''

        whole, cents = divmod(total_cents, CENTS_PER_WHOLE)

        if whole < SHOW_CENTS_LIMIT:
            show_cents = True

        if whole > 0:
            whole_str = str(whole)
            whole_len = len(whole_str)
            whole = NBSP.join(whole_str[max(i, 0):i + 3] for i in range(-(-whole_len % 3), whole_len, 3))

        if show_cents and cents:
            if whole:
                return sign_string + self.L_template().replace("%d", "%s", 1) % (whole, cents)

            return sign_string + self.L_template_cents() % cents

        return sign_string + self.L_template_whole().replace("%d", "%s", 1) % whole

    @classmethod
    def get_ordering_field_name(cls, national_version):
        national_order_field = 'order_{}'.format(national_version)
        if has_field(cls, national_order_field):
            return national_order_field
        return 'order'

    @classmethod
    def get_ordered_queryset(cls, national_version):
        return cls.objects.order_by(cls.get_ordering_field_name(national_version))

    @classmethod
    def L_order_field(cls, request):
        warnings.warn(
            '[2016-04-26] use get_ordering_field_name or get_ordered_queryset', RaspDeprecationWarning, stacklevel=2)
        return cls.get_ordering_field_name(request.NATIONAL_VERSION)

    def L_name_in(self):
        if translation.get_language() == 'tr':
            return self.name_tr or self.name

        if translation.get_language() == 'uk':
            return self.name_uk_in or self.name_in

        return self.name_in

    def json(self):
        return {
            'code': self.code,
            'iso_code': self.iso_code,
            'name': self.L_name(),
            'name_in': self.L_name_in(),
            'template': self.L_template(),
            'template_whole': self.L_template_whole(),
            'template_cents': self.L_template_cents(),
        }

    @classmethod
    def _fetch_rates(cls, currencies, geoid, base_currency):
        """ https://wiki.yandex-team.ru/VladimirSavenkov/ConvRasp/ """
        if not settings.CURRENCY_RATES_URL:
            return {}, None

        params = {
            'from': base_currency,
            'to': [c.code for c in currencies if c.code != base_currency],
            'geoid': geoid,
        }

        url = settings.CURRENCY_RATES_URL + '?' + urlencode(params, doseq=True)

        tree = etree.parse(quick_urlopen(url))

        status = tree.xpath("/response/status")[0].text

        if status != 'ok':
            return {}, None

        src = tree.xpath("/response/data/src")[0].text

        rates = dict((to.attrib['id'], float(to.attrib['value'])) for to in tree.xpath("/response/data/rates/to"))

        return rates, src

    @classmethod
    def fetch_rates(cls, currencies, geoid, base_currency=None):
        if not base_currency:
            base_currency = cls.BASE_CURRENCY

        try:
            rates, src = cls._fetch_rates(currencies, geoid, base_currency)
        except (urllib2.HTTPError, urllib2.URLError, IOError, socket.error, TypeError, ValueError, httplib.BadStatusLine):
            log_warning()

            rates = {}
            src = None

        rates[base_currency] = 1

        return src, rates


class Price(object):
    def __init__(self, value, currency=None, iso_currency=None):
        self.value = value
        self.currency = currency or Currency.BASE_CURRENCY
        self.iso_currency = iso_currency

        self.base_value = None

        if self.currency == Currency.BASE_CURRENCY:
            self.base_value = value

    def rebase(self, rates):
        if self.currency == Currency.BASE_CURRENCY:
            return

        rate = rates.get(self.currency)

        if rate is not None:
            self.base_value = self.value * rate

    def __json__(self):
        v = {
            'value': self.value,
            'currency': self.currency,
        }

        if self.base_value:
            v['baseValue'] = self.base_value

        return v

    @classmethod
    def parse(cls, s):
        if not s:
            raise ValueError

        parts = s.split()

        value = float(parts[0])

        currency = None

        if len(parts) > 1:
            currency = parts[1]

        return cls(value, currency)

    def __repr__(self):
        return '<Price base_value=%r currency=%r value=%r>' % (self.base_value, self.currency, self.value)

    def __str__(self):
        return "%s %s" % (self.value, self.currency)

    @property
    def sort_value(self):
        return [self.base_value, self.value]

    def __cmp__(self, other):
        if not isinstance(other, Price):
            other = Price(other)

        return cmp(self.sort_value, other.sort_value)

    def __mul__(self, other):
        return Price(self.value * other, self.currency, self.iso_currency)

    def __nonzero__(self):
        return int(self.value)

    def __add__(self, other):
        if other.currency != self.currency:
            raise ValueError("Can't sum prices with different currencies %s and %s" % (self.currency, other.currency))

        return Price(self.value + other.value, self.currency, self.iso_currency)
