# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging
import pickle
import re
import warnings
import zlib
from itertools import chain

from django.conf import settings
from django.conf.global_settings import LANGUAGES
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import (
    GenericRelation,
    GenericForeignKey
)
from django.core.cache import cache
from django.core.exceptions import ValidationError
from django.db import models
from django.db.models import Q
from django.db.models.signals import pre_save
from django.dispatch import receiver
from django.utils import translation
from django.utils.translation import ugettext_lazy as _
from lxml import etree

from travel.avia.library.python.common.models.translations import TranslationMixin, TranslatedTitle
from travel.avia.library.python.common.models_abstract.geo import (
    BaseStation, BaseDistrict, BaseSettlementImage,
)
from travel.avia.library.python.common.models_utils import fetch_children, HiddenManagerWrapper
from travel.avia.library.python.common.models_utils.geo import (
    TransLocalMixin, TR_SUGGEST_FULL_TITLE_OVERRIDES
)
from travel.avia.library.python.common.models_utils.geo import Point, RE_MDOT, TimeZoneMixin
from travel.avia.library.python.common.models_utils.i18n import (
    AbstractTranslate, new_L_field, new_Geobase_L_title,
)
from travel.avia.library.python.common.precache.manager import PrecachingManager
from travel.avia.library.python.common.settings import SUGGEST_LANGUAGES
from travel.avia.library.python.common.utils.fields import (
    RegExpField, TextFileField, CodeCharField, TrimmedCharField,
)
from travel.avia.library.python.common.utils.geobase import geobase
from travel.avia.library.python.common.utils.i18n import CollateUnicode
from travel.avia.library.python.common.utils.iterrecipes import unique_justseen
from travel.avia.library.python.common.utils.models import SwapModel
from travel.avia.library.python.common.utils.schema import parse_schema, build_schema, initial_schema
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.library.python.common.xgettext.i18n import gettext


log = logging.getLogger(__name__)


class CityMajority(models.Model):
    """ Важность города: столица, деревня, ... """

    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)

    COMMON_CITY_ID = 4
    POPULATION_MILLION_ID = 3
    CAPITAL_ID = 1
    REGION_CAPITAL_ID = 2

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'важность города')
        verbose_name_plural = _(u'важности городов')
        app_label = 'www'
        db_table = 'www_citymajority'


class StationMajority(TranslationMixin, models.Model):
    """ Важность станции: главная в городе, ... """

    MAIN_IN_CITY_ID = 1
    IN_TABLO_ID = 2
    NOT_IN_TABLO_ID = 3
    STATION_ID = 4
    NOT_IN_SEARCH_ID = 5
    EXPRESS_FAKE_ID = 100
    MAX_ID = 100

    IN_TABLO = IN_TABLO_ID

    TANKER_L_FIELDS = ['title']

    L_title = new_L_field(_(u'наименование'))

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    code = models.CharField(verbose_name=_(u'код для питона'), max_length=50,
                            editable=False)

    objects = PrecachingManager(
        keys=(
            'pk',
            'code',
        ),
        select_related=(
            'new_L_title',
        ),
    )

    def __unicode__(self):
        return self.L_title()

    def get_title(self):
        return self.L_title()

    title = property(get_title)

    class Meta:
        verbose_name = _(u'важность станции')
        verbose_name_plural = _(u'важности станций')
        app_label = 'www'
        db_table = 'www_stationmajority'


class Country(TranslationMixin, models.Model, Point):
    """ Страна """

    RUSSIA_GEO_ID = 225

    L_title = new_Geobase_L_title(_(u'наименование'), add_local_field=True, local_field_critical=True, extra={
        'ru': [
            ('preposition_v_vo_na', new_L_field.char_field(_(u'предлог (в, во, на)'), max_length=2)),  # prefix_in
            ('genitive', new_L_field.char_field(_(u'наименование (род. падеж'))),  # title_of
            ('locative', new_L_field.char_field(_(u'наименование (местный. падеж)')))  # title_in
        ],
        'uk': [
            # В геобазе нет винительного падежа у стран
            ('accusative', new_L_field.char_field(_(u'наименование (вин. падеж) (uk)'))),
        ]
    })

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    code = CodeCharField(verbose_name=_(u'код страны ISO 2-alpha'), max_length=4,
                         null=True, blank=True, default=None, unique=True)
    code3 = CodeCharField(verbose_name=_(u'код страны ISO 3166-1 3-alpha'), max_length=4,
                          null=True, blank=True, default=None, unique=True)
    domain_zone = CodeCharField(verbose_name=_(u'имя доменной зоны IANA'), max_length=2,
                                null=True, blank=True, default=None, unique=True,
                                help_text=_(u"без точки: 'ru', 'us', 'uk'"))
    synonyms = GenericRelation('PointSynonym', related_query_name='country')

    currency = models.ForeignKey('currency.Currency', verbose_name=_(u'валюта'),
                                 null=True, blank=True, default=None)

    language = CodeCharField(
        _(u'язык страны'),
        max_length=5,
        choices=[(k, _(v)) for k, v in LANGUAGES],
        default='ru',
    )
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    objects = PrecachingManager(
        keys=(
            'pk',
            'domain_zone',
            '_geo_id',
        ),
        select_related=(
            'new_L_title',
        )
    )

    RUSSIA_ID = 225
    BELARUS_ID = 149
    UKRAINE_ID = 187
    TAJIKISTAN_ID = 209
    TURKEY_ID = 983
    KAZAKHSTAN_ID = 159
    LITVA_ID = 117

    def __unicode__(self):
        return self.L_title()

    def save(self, **kwargs):
        super(Country, self).save(**kwargs)

    def get_local_language(self):
        return self.language

    @property
    def name(self):
        return self.__unicode__()

    @property
    def title_with_prefix(self):
        return self.title

    @property
    def hidden(self):
        return False

    @property
    def suburban_zone_id(self):
        return None

    def get_capital(self):
        try:
            return Settlement.objects.get(country=self, majority__id=CityMajority.CAPITAL_ID)

        except Settlement.DoesNotExist:
            return None

        except Settlement.MultipleObjectsReturned:
            try:
                return Settlement.objects.filter(country=self, majority__id=CityMajority.CAPITAL_ID)[0]

            except KeyError:
                return None

    def get_capital_tz(self):
        capital = self.get_capital()
        return capital and capital.time_zone

    @classmethod
    def mast_have_region_countries(cls):
        return [cls.objects.get(code=u"RU"), cls.objects.get(code=u"KZ"),
                cls.objects.get(code=u"US")]

    def L_omonim_title(self, show_district=True, lang=None, national_version=None):
        return {'title': self.L_title(lang=lang), 'add': ''}

    def L_omonim_title_bem(self, request, direction, **extra_params):
        return {'title': self.L_title()}

    def L_suggest_full_title_tr(self, lang=None):
        """Full title for turkish suggests"""

        return [self.L_title(lang=lang)]

    def get_popular_title(self, lang=None):
        # необходимо для общего интерфейса
        return self.L_title(lang=lang)

    def get_railway_tz_city(self):
        # В Таджикистане на ЖД московское время
        if self.id == self.TAJIKISTAN_ID:
            country = Country.objects.get(id=self.RUSSIA_ID)
        else:
            country = self

        return country.get_capital()

    def get_sign(self):
        return self.L_title(lang='en').replace(' ', '')

    @classmethod
    def find_by_sign(cls, sign):
        countries = cls.objects.all()
        by_slug = dict([(c.get_sign().lower(), c) for c in countries])
        return by_slug.get(sign.replace(' ', '').lower())

    class Meta:
        verbose_name = _(u'страна')
        verbose_name_plural = _(u'страны')
        app_label = 'www'
        db_table = 'www_country'


class Region(TranslationMixin, models.Model, TimeZoneMixin, Point):
    """ Область """

    country = models.ForeignKey(Country, verbose_name=_(u'страна'),
                                null=False, blank=False)

    L_title = new_Geobase_L_title(_(u'наименование'), add_local_field=True, local_field_critical=True)

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 help_text=_(u'временная зона в международном формате'),
                                 null=True, blank=False, max_length=30)
    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    hidden = models.BooleanField(verbose_name=_(u'не показывать нигде!'),
                                 default=False)
    koatuu = CodeCharField(verbose_name=_(u'код КОАТУУ'), max_length=40,
                           null=True, blank=True, default=None, unique=True,
                           help_text=_(u"держа́вний класифіка́тор об'є́ктів адміністрати́вно-територіа́льного у́строю Украї́ни"))
    agent_geo_id = models.IntegerField(verbose_name=_(u'agent Geo ID'), null=True,
                                       blank=True, default=None)
    disputed_territory = models.BooleanField(verbose_name=_(u"спорная территория"), default=False)
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)

    objects = PrecachingManager(
        keys=(
            'pk',
            '_geo_id',
        ),
        select_related=(
            'new_L_title',
        ),
    )
    hidden_manager = HiddenManagerWrapper('objects')

    MOSCOW_REGION_ID = 1

    def get_capital(self):
        try:
            return self.settlement_set.filter(majority__id__in=[1, 2])[0]
        except IndexError:
            return None

    def alone(self):
        """ Проверяет, что регион - единственный в стране """
        return Region.hidden_manager.filter(country=self.country).count() == 1

    def __unicode__(self):
        return self.L_title()

    def get_local_language(self):
        return self.country and self.country.language

    @property
    def name(self):
        return self.__unicode__()

    def short_title(self):
        return re.sub(u"область", u"обл.", re.sub(u"Республика", u"Респ.", self.title))

    def save(self, **kwargs):
        self.time_zone = self.time_zone or None
        super(Region, self).save(**kwargs)

    class Meta:
        verbose_name = _(u'область')
        verbose_name_plural = _(u'области')
        app_label = 'www'
        db_table = 'www_region'


class SettlementProperty(models.Model):
    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    description = models.TextField(verbose_name=_(u'описание'),
                                   null=True, blank=True, default=None)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'свойство города')
        verbose_name_plural = _(u'свойства городов')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_settlementproperty'


class Settlement(TranslationMixin, models.Model, TimeZoneMixin, TransLocalMixin, Point):
    """ Город """

    # Id Москвы в нашей базе.
    # Используется как город по умолчанию во многих случаях.
    MOSCOW_ID = 213
    MOSCOW_GEO_ID = 213

    KIEV_ID = 143
    KIEV_GEO_ID = 143

    ISTANBUL_ID = 11508
    ISTANBUL_GEO_ID = 11508

    SPB_ID = 2

    CITIES_IDS_FOR_YMAPS = set([
        # Россия

        MOSCOW_ID,  # Москва (Московская область)
        SPB_ID,  # Санкт-Петербург (Ленинградская область)
        1107,  # Анапа (Краснодарский край)
        11080,  # Арзамас (Нижегородская область)
        20,  # Архангельск (Архангельская область)
        22174,  # Великий Новгород (Новгородская область)
        192,  # Владимир (Владимирская область)
        10990,  # Геленджик (Краснодарский край)
        # 25066, # Дзержинск (Житомирская область)
        972,  # Дзержинск (Нижегородская область)
        # 20692, # Дзержинск (Красноярский край)
        # 23230, # Дзержинск (Донецкая область)
        # 23261, # Дзержинск (Минская область)
        54,  # Екатеринбург (Свердловская область)
        43,  # Казань (Республика Татарстан)
        46,  # Киров (Кировская область)
        # 20200, # Киров (Калужская область)
        7,  # Кострома (Костромская область)
        35,  # Краснодар (Краснодарский край)
        236,  # Набережные Челны (Республика Татарстан)
        23243,  # Нижний Новгород (Нижегородская область)
        65,  # Новосибирск (Новосибирская область)
        10,  # Орел (Орловская область)
        39,  # Ростов-на-Дону (Ростовская область)
        11,  # Рязань (Рязанская область)
        51,  # Самара (Самарская область)
        42,  # Саранск (Республика Мордовия)
        240,  # Тольятти (Самарская область)
        15,  # Тула (Тульская область)
        172,  # Уфа (Республика Башкортостан)
        45,  # Чебоксары (Республика Чувашия)
        56,  # Челябинск (Челябинская область)
        16,  # Ярославль (Ярославская область)

        # Украина

        KIEV_ID,  # Киев (Киев и Киевская область)
        11471,  # Алушта (Крым и г. Севастополь)
        963,  # Винница (Винницкая область)
        141,  # Днепропетровск (Днепропетровская область)
        # 11038, # Донецк (Ростовская область)
        142,  # Донецк (Донецкая область)
        11463,  # Евпатория (Крым и г. Севастополь)
        10343,  # Житомир (Житомирская область)
        960,  # Запорожье (Запорожская область)
        10345,  # Ивано-Франковск (Ивано-Франковская область)
        11464,  # Керчь (Крым и г. Севастополь)
        20221,  # Кировоград (Кировоградская область)
        23024,  # Коктебель (Крым и г. Севастополь)
        222,  # Луганск (Луганская область)
        20222,  # Луцк (Волынская область)
        144,  # Львов (Львовская область)
        10366,  # Мариуполь (Донецкая область)
        # 25811, # Николаев (Хмельницкая область)
        148,  # Николаев (Николаевская область)
        # 22851, # Николаев (Ростовская область)
        # 23565, # Николаев (Львовская область)
        # 25721, # Новый Свет (Донецкая область)
        26094,  # Новый Свет (Крым и г. Севастополь)
        145,  # Одесса (Одесская область)
        959,  # Севастополь (Крым и г. Севастополь)
        146,  # Симферополь (Крым и г. Севастополь)
        11472,  # Судак (Крым и г. Севастополь)
        965,  # Сумы (Сумская область)
        10358,  # Ужгород (Закарпатская область)
        11469,  # Феодосия (Крым и г. Севастополь)
        147,  # Харьков (Харьковская область)
        962,  # Херсон (Херсонская область)
        # 25733, # Ялта (Донецкая область)
        11470,  # Ялта (Крым и г. Севастополь)
    ])

    majority = models.ForeignKey(CityMajority, verbose_name=_(u'важность города'),
                                 null=True, blank=True, default=None)

    suggest_order = models.IntegerField(_(u'порядок в саджестах'),
                                        help_text=_(u'сначала выводятся города с меньшим значением'),
                                        null=False, default=100)

    country = models.ForeignKey(Country, verbose_name=_(u'страна'),
                                null=True, blank=True, default=None)
    region = models.ForeignKey(Region, verbose_name=_(u'область'),
                               null=True, blank=True, default=None)
    district = models.ForeignKey('District', null=True, blank=True, default=None,
                                 verbose_name=_(u"район"), related_name=u"settlements")

    L_title = new_Geobase_L_title(
        _(u'наименование'),
        add_local_field=True,
        local_field_critical=True,
        db_index=True, extra={
            'ru': [
                ('preposition_v_vo_na', new_L_field.char_field(_(u'предлог (в, во, на)'), max_length=2)),
                ('genitive', new_L_field.char_field(_(u'наименование (род. падеж)'))),
                ('accusative', new_L_field.char_field(_(u'наименование (вин. падеж)'))),
                ('locative', new_L_field.char_field(_(u'наименование (местн. падеж)'))),
            ]
        }
    )

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    L_abbr_title = new_L_field(
        _(u'сокр. наименование'),
        add_local_field=True,
        max_length=4,
        help_text=_(u'максимум 4 символа'),
    )

    new_L_abbr_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    time_zone = models.CharField(verbose_name=_(u'временная зона'),
                                 help_text=_(u'временная зона в международном формате'),
                                 null=False, blank=False, max_length=30)
    phone_info = models.CharField(verbose_name=_(u'тел. справочной'), max_length=20,
                                  null=True, blank=True, default=None)
    phone_info_short = models.CharField(verbose_name=_(u'короткий тел. справочной'),
                                        max_length=20,
                                        null=True, blank=True, default=None)
    _geo_id = models.IntegerField(verbose_name=_(u'geo ID'), null=True,
                                  blank=True, default=None, unique=True)
    _kladr_id = models.CharField(verbose_name=_(u'kladr ID'), max_length=50,
                                 null=True, blank=True, default=None,
                                 editable=False)
    sirena_id = CodeCharField(verbose_name=_(u'код в "Sirena"'), max_length=50,
                              null=True, blank=True, default=None, unique=True)
    iata = CodeCharField(verbose_name=_(u'код IATA'), max_length=20,
                         null=True, blank=True, default=None, unique=True)
    koatuu = CodeCharField(verbose_name=_(u'код КОАТУУ'), max_length=40,
                           null=True, blank=True, default=None, unique=True,
                           help_text=_(u"держа́вний класифіка́тор об'є́ктів адміністрати́вно-територіа́льного у́строю Украї́ни"))
    properties = models.ManyToManyField(SettlementProperty,
                                        verbose_name=_(u'свойства'), blank=True, default=None)
    big_city = models.BooleanField(verbose_name=_(u'город на уровне столицы'),
                                   default=False)
    hidden = models.BooleanField(verbose_name=_(u'не показывать нигде!'),
                                 default=False)
    service_comment = models.TextField(verbose_name=_(u'служебный комментарий'),
                                       help_text=_(u'не показывается в сервисе'),
                                       null=True, blank=True, default=None)
    has_tablo = models.BooleanField(verbose_name=_(u'город имеет табло'),
                                    default=True, blank=False, null=False)
    agent_geo_id = models.IntegerField(verbose_name=_(u'agent Geo ID'), null=True,
                                       blank=True, default=None)
    suburban_zone = models.ForeignKey("SuburbanZone", blank=True, null=True,
                                      verbose_name=_(u"пригородная зона"),
                                      help_text=_(u"пригородная зона, в которую входит город"),
                                      related_name="settlement_set")
    has_many_airports = models.BooleanField(verbose_name=_(u'много аэропортов'),
                                            default=False, null=False)
    has_urban_transport = models.BooleanField(verbose_name=_(u'есть городской транспорт'),
                                              default=False, null=False)
    human_url = CodeCharField(verbose_name=_(u'код города для ЧПУ'), max_length=50,
                              null=True, blank=True, default=None, unique=True)
    threads_amount = models.PositiveIntegerField(verbose_name=_(u'общее количество рейсов в табло'),
                                                 default=0, editable=False)
    synonyms = GenericRelation('PointSynonym', related_query_name='settlement')
    type_choices = models.CharField(verbose_name=_(u'типы ТС'), max_length=50, default='', null=False, blank=True)
    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None, blank=True, null=True,
                                  db_index=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None, blank=True, null=True,
                                 db_index=True)
    _disputed_territory = models.BooleanField(verbose_name=_(u"спорная территория"), default=False,
                                              db_column='disputed_territory')
    modified_at = models.DateTimeField(_(u'Дата-время изменения'), auto_now=True, null=False, blank=True)
    old_avia_id = models.IntegerField(verbose_name=u'Id в Авиа до синхронизации с расписаниями',
                                      null=True, blank=True, default=None)

    objects = PrecachingManager(
        keys=(
            '_geo_id',
            'pk',
        ),
        iexact_keys=(
            'iata',
        ),
        select_related=(
            'new_L_title',
            'new_L_abbr_title',
        ),
    )

    hidden_manager = HiddenManagerWrapper('objects')

    @property
    def type_choices_set(self):
        return self.type_choices and set(self.type_choices.split(',')) or set()

    @property
    def title_with_prefix(self):
        if self.majority_id != 5:  # Поселок или деревня
            return u"г. " + self.title

        return self.title

    def L_title_with_prefix(self, lang=None):
        if not lang:
            lang = translation.get_language()

        if self.majority_id != 5:  # Поселок или деревня
            return gettext(u'г. %s', lang=lang) % self.L_title(lang=lang)

        return self.L_title(lang=lang)

    def L_title_with_type(self, lang=None):
        return self.L_title_with_prefix(lang)

    def get_geobase_region(self):
        if not self._geo_id or not geobase:
            return None

        try:
            return geobase.region_by_id(self._geo_id)
        except Exception:
            log.exception(u'Не удалось получить ename города %s из геобазы', self._geo_id)
            return None

    @classmethod
    def get_default_city(cls):
        """Дефолт-сити"""
        return cls.hidden_manager.get(id=Settlement.MOSCOW_ID)

    @classmethod
    def get_default_for_root_domain(cls, root_domain):
        if root_domain in settings.DEFAULT_CITIES:
            return cls.objects.get(id=settings.DEFAULT_CITIES[root_domain])

        try:
            country = Country.objects.get(domain_zone=root_domain)

            return country.get_capital()
        except Country.DoesNotExist:
            pass

        return cls.get_default_city()

    def save(self, **kwargs):
        self.time_zone = self.time_zone or None
        if not self.country_id:
            if self.region:
                self.country = self.region.country

        super(Settlement, self).save(**kwargs)

    def __unicode__(self):
        return self.L_title()

    def __repr__(self):
        return u'<Settlement id:{}, {!r}, region:{}, country:{}>'.format(
            self.id, self.title, self.region_id, self.country_id
        )

    def get_local_language(self):
        return self.country and self.country.language

    @property
    def name(self):
        return self.title

    def get_popular_title(self, **kwargs):
        return self.L_title()

    def get_short_title(self):
        return self.L_title()

    def L_omonim_title(self, lang=None, show_district=True, national_version=None):
        u"""Название города для саггестов"""
        result = {
            'title': self.L_title_with_prefix(lang),
            'add': ''
        }

        add = []
        if show_district and self.district_id:
            add.append(self.district.title)

        if self.region_id:
            add.append(self.region.L_title(lang=lang))

        country = self.translocal_country(national_version)
        if country:
            add.append(country.L_title(lang=lang))

        if add:
            result['add'] = u", ".join(add)

        return result

    def L_omonim_title_bem(self, request, direction, **extra_params):
        u"""Название города со ссылкой для переключения в омонимах"""
        result = {
            'title': self.L_title_with_prefix(),
        }

        add = []

        if self.district_id:
            add.append(self.district.title)

        if self.region_id:
            add.append(self.region.L_title())

        if self.country_id and not self.disputed_territory:
            add.append(self.country.L_title())

        if add:
            result['add'] = add

        return result

    def L_full_geography(self, show_district=False, national_version=None, lang=None):
        country = self.translocal_country(national_version)

        # Для Москвы, Санкт-Петербурга и Киева регион не указывается, так как регион называется "Москва и Московская область"
        cities_as_regions = [self.MOSCOW_ID, self.SPB_ID, self.KIEV_ID]
        is_region = self.id in cities_as_regions

        breadcrumbs = [
            show_district and self.district and not is_region and self.district.L_title_with_postfix(lang=lang),
            self.region and not is_region and self.region.L_title(lang=lang),
            country and country.L_title(lang=lang)
        ]

        return u', '.join(b for b in breadcrumbs if b)

    def L_title_with_full_geography(self, national_version=None, lang=None):
        """
        Получает строку с минимально допустимым количеством коллизий
        для поиска населенного пункта в сторонних сервисах: blablacar.ru, maps.yandex.ru
        Примеры:
        Москва, Россия
        Первоуральск, Свердловская область, Россия
        Первомайский, Мамонтовский р-н, Алтайский край, Россия
        """
        parts = [self.L_title(lang=lang), self.L_full_geography(True, national_version, lang)]
        return u', '.join(part for part in parts if part)

    def L_suggest_full_title_tr(self, lang=None):
        """Full title for turkish suggests"""

        override = TR_SUGGEST_FULL_TITLE_OVERRIDES.get(self.point_key)

        if override:
            return override

        parts = [
            self.L_title_with_type(lang),
        ]

        if self.region_id and self.majority_id > CityMajority.POPULATION_MILLION_ID:
            parts.append(self.region.L_title(lang=lang))

        country = self.translocal_country('tr')

        if country:
            parts.append(country.L_title(lang=lang))

        return parts

    @property
    def disputed_territory(self):
        """RASP-15442"""
        if self._disputed_territory:
            return True

        if self.region_id and self.region.disputed_territory:
            return True

        return self.district_id and self.district.disputed_territory

    class Meta:
        verbose_name = _(u'город')
        verbose_name_plural = _(u'города')
        app_label = 'www'
        db_table = 'www_settlement'

    is_city = True

    def city_size(self):
        return self.majority_id and ((self.majority_id in [1, 2] and 'big') or
                                     (self.majority_id in [5] and 'small') or '') or ''

    def has_actual_tablo(self):
        """Для городов актуальное табло не строится"""
        return False

    def get_agent_geo_id(self):
        """Определяет agent_geo_id у города"""
        agent_geo_id = self.agent_geo_id

        if agent_geo_id is None:
            agent_geo_id = self.region and self.region.agent_geo_id

        return agent_geo_id

    def get_externaldirections(self):
        return ExternalDirection.objects.filter(externaldirectionmarker__station__settlement=self).distinct()

    def get_directions(self):
        """
        main_directions - если есть пригородная зона, то направления, проходящие через станции города
        directions - остальные направления, либо все, если город не привязан к пригородной зоне
        """

        if self.suburban_zone:
            directions = self.suburban_zone.externaldirection_set.all()
            main_directions = list(directions.filter(externaldirectionmarker__station__settlement=self).distinct())
            directions = [d for d in directions if d not in main_directions]  # Тут только не проходящие через станции города

            return main_directions, directions

        return [], []

    @property
    def has_ymap(self):
        return self.id in Settlement.CITIES_IDS_FOR_YMAPS

    def get_stations(self):
        stations = Station.objects.filter(
            Q(settlement=self) | Q(station2settlement__settlement=self)
        ).filter(hidden=False).distinct()

        return stations

    def get_stations_by_type(self, plane_only=False, use_water=False):
        """ Возвращает станции всех типов транспорта для данного города
            Вход: город - объект Settlement
            Выход: {'plane': [st1, st2], 'train': [st3, st4]}
        """
        from travel.avia.library.python.common.models.transport import TransportType

        if not use_water:
            warnings.warn('[2015-11-30] use_water must be True', RaspDeprecationWarning, stacklevel=2)

        t_types = TransportType.objects.in_bulk_cached()
        result = {
            t_types[t_type_code].code: {'type': t_types[t_type_code], 'stations': [], 'related': []}
            for t_type_code in t_types.keys()
        }

        qs = Station.objects.filter(hidden=False, majority__id__lt=3)

        stations = list(chain(qs.filter(settlement=self),
                              qs.filter(station2settlement__settlement=self)))

        stations.sort(key=lambda s: (s.majority_id, CollateUnicode(s.get_clean_title().strip('"')), s.id))

        terminals = fetch_children(stations, StationTerminal.objects, 'station')

        # Разносим станции по типам ТС
        for st in unique_justseen(stations, key=lambda s: s.id):
            ttype_id = st.t_type_id
            if plane_only and ttype_id != TransportType.PLANE_ID:
                continue

            ttype_code = st.t_type.code
            if ttype_id in TransportType.WATER_TTYPE_IDS:
                ttype_code = 'water'

            result[ttype_code]['stations'].append((st, terminals.getlist(st.id)))

        related = self.related_stations.filter(station__hidden=False, station__majority__id__lt=3).order_by(
            'station__majority__id',
            'station__popular_title',
            'station__title',
        )

        for related_station in related:
            station = related_station.station

            ttype_id = station.t_type_id
            if plane_only and ttype_id != TransportType.PLANE_ID:
                continue

            ttype_code = station.t_type.code
            if ttype_id in TransportType.WATER_TTYPE_IDS:
                ttype_code = 'water'

            result[ttype_code]['related'].append((station, terminals.getlist(station.id)))

        if not use_water and 'water' in result:
            result['sea'] = result.pop('water')

        if use_water and 'sea' in result:
            del result['sea']

        if use_water and 'river' in result:
            del result['river']

        return result

    @property
    def tz_name_abbr(self):
        tz_name = self.title_abbr or self.title_ru_locative or self.title

        return u'время в %s' % tz_name


class SettlementRelatedStations(models.Model):
    """ Соответствия станций и городов """
    settlement = models.ForeignKey('Settlement', related_name='related_stations', verbose_name=_(u'город'))
    station = models.ForeignKey('Station', related_name='related_settlements', verbose_name=_(u'станция'))

    class Meta:
        db_table = 'www_settlement_related_stations'
        verbose_name = _(u'соответствие станции и города для отображения')
        verbose_name_plural = _(u'соответствия станций и городов для отображения')
        ordering = ('settlement', 'station')
        app_label = 'www'


class SettlementNearest(models.Model):
    """ Соответствия станций и городов """
    settlement = models.ForeignKey('Settlement', related_name='related_settlement', verbose_name=_(u'город'))
    nearest = models.ForeignKey('Settlement', related_name='related_nearest', verbose_name=_(u'ближайший'))
    description = models.TextField(verbose_name=_(u"описание проезда"), null=True, blank=True)

    class Meta:
        verbose_name = _(u'ближайший город')
        verbose_name_plural = _(u'ближайшие города')
        app_label = 'www'
        db_table = 'www_settlementnearest'


class StationProperty(models.Model):
    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    description = models.TextField(verbose_name=_(u'описание'),
                                   null=True, blank=True, default=None)
    order = models.IntegerField(default=10, help_text=_(u"чем больше цифра, тем хуже признак"))

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'свойство станции')
        verbose_name_plural = _(u'свойства станций')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_stationproperty'


class StationType(models.Model):
    """ Тип станции """

    TANKER_L_FIELDS = ['name', 'prefix', 'railway_prefix']

    L_name = new_L_field(_(u'название'), max_length=255, base_field_critical=True, base_lang='ru', extra={
        'ru': [
            ('genitive', new_L_field.char_field(_(u'название в родительном падеже'))),  # name_gen
            ('accusative', new_L_field.char_field(_(u'название в винительном падеже'))),  # name_acc
            ('dative',  new_L_field.char_field(_(u'название в дательном падеже'))),  # name_dat
        ]
    })

    new_L_name = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    L_prefix = new_L_field(_(u'префикс'), base_lang='ru', max_length=100)

    new_L_prefix = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    L_railway_prefix = new_L_field(_(u'префикс для жд'), base_lang='ru', max_length=100, help_text=_(u"для списка станций на странице жд рейса"))

    new_L_railway_prefix = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    title_regex = RegExpField(verbose_name=_(u'шаблон названия станции'), null=True, blank=True,
                              default=None, max_length=100)

    default_for_t_type = models.OneToOneField('www.TransportType', null=True,
                                              verbose_name=_(u"по умочанию для типа транспорта"),
                                              default=None, blank=True, related_name='default_station_type')

    types = None
    regexes = None

    objects = PrecachingManager(
        keys=(
            'pk',
        ),
        select_related=(
            'new_L_name',
            'new_L_prefix',
            'new_L_railway_prefix'
        ),
    )

    STATION_ID = 1
    PLATFORM_ID = 2
    STOP_ID = 3
    CHECKPOINT_ID = 4
    POST_ID = 5
    CROSSING_ID = 6
    OVERTAKING_POINT_ID = 7
    TRAIN_STATION_ID = 8
    AIRPORT_ID = 9
    BUS_STATION_ID = 10
    BUS_STOP_ID = 11
    UNKNOWN_ID = 12
    PORT_ID = 13
    PORT_POINT_ID = 14
    WHARF_ID = 15
    RIVER_PORT_ID = 16
    MARINE_STATION_ID = 17

    @classmethod
    def types_map(cls):
        if not cls.types:
            cls.types = dict((t.name_ru, t) for t in cls._default_manager.all())

        return cls.types

    @classmethod
    def regex_list(cls):
        if not cls.regexes:
            cls.regexes = [
                (t, re.compile(t.title_regex, re.U | re.I))
                for t in cls._default_manager.filter(title_regex__isnull=False).exclude(title_regex='')
            ]

        return cls.regexes

    def __unicode__(self):
        return self.L_name()

    @property
    def L_title(self):
        return self.L_name

    class Meta:
        verbose_name = _(u'тип станции')
        verbose_name_plural = _(u'типы станций')
        app_label = 'www'
        db_table = 'www_stationtype'


class Station(BaseStation):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.geo'

    class Meta(BaseStation.Meta):
        app_label = 'www'


@receiver(pre_save, sender=Station)
def station_change_tablo_state_handler(sender, **kwargs):
    """
    RASP-13834 При смене на значение "Нет данных" нужно запоминать предыдущее значение и выводить его под селектом
    """

    instance = kwargs['instance']

    if not instance.id:
        return

    try:
        old_value = Station.objects.get(id=instance.id)

        if instance.tablo_state != old_value.tablo_state:
            instance.tablo_state_prev = old_value.tablo_state

    except Station.DoesNotExist:
        pass


class StationTerminal(models.Model):
    u"""Терминал станции (аэропорта)"""

    name = TrimmedCharField(_(u'Название'), max_length=50, blank=False, null=False)
    station = models.ForeignKey(Station, verbose_name=_(u'станция'), null=False, blank=False)
    sirena_id = CodeCharField(_(u'Диана код терминала'), max_length=6,
                              blank=True, null=True)
    sirena_letter = CodeCharField(_(u'Сирена буква терминала'), max_length=6,
                                  blank=True, null=True)

    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None,
                                  blank=True, null=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None,
                                 blank=True, null=True)

    objects = PrecachingManager(keys=['pk', 'station'])

    def __unicode__(self):
        return "%s-%s" % (self.station.title or '', self.name)

    @property
    def export_uid(self):
        u"""ID терминала для экспорта"""
        return "exit__%d" % self.id

    @property
    def export_type(self):
        u"""Тип терминала для экспорта"""
        return "exit"

    def as_xml(self):
        element = etree.Element('terminal',
                                name=self.name,
                                lat=str(self.latitude),
                                lon=str(self.longitude))

        return element

    class Meta:
        verbose_name = _(u'терминал')
        verbose_name_plural = _(u'терминалы')
        app_label = 'www'
        db_table = 'www_stationterminal'

    @classmethod
    def get_airport_terminal(cls, airport, letter):
        """
        Правильно работает только для аэропортов
        """
        if not letter:
            return

        try:
            return airport.stationterminal_set.get(name=letter)
        except StationTerminal.DoesNotExist:
            pass

        for terminal in airport.stationterminal_set.all():
            if terminal.sirena_letter:
                for sirena_letter in terminal.sirena_letter.split(u'/'):
                    if sirena_letter.upper() == letter.upper():
                        return terminal


class StationPassage(models.Model):
    u"""Пеший переход между станциями"""

    station_from = models.ForeignKey(Station, verbose_name=_(u'с какой станции'),
                                     null=False, blank=False,
                                     related_name='outgoing_passages')
    station_to = models.ForeignKey(Station, verbose_name=_(u'до какой станции'),
                                   null=False, blank=False,
                                   related_name='incoming_passages')
    terminal_from = models.ForeignKey(StationTerminal, verbose_name=_(u'с какого выхода'),
                                      null=True, blank=True, default=None,
                                      related_name='outgoing_passages')
    terminal_to = models.ForeignKey(StationTerminal, verbose_name=_(u'до какого выхода'),
                                    null=True, blank=True, default=None,
                                    related_name='incoming_passages')
    duration = models.IntegerField(verbose_name=_(u'длительность перехода'),
                                   null=True, blank=True, default=None)
    is_dual = models.BooleanField(verbose_name=_(u'переход двусторонний'),
                                  null=False, blank=False, default=True)

    def __unicode__(self):
        return "%s-%s" % (self.station_from.title, self.station_to.title)

    def as_xml(self, code_system=None):
        link = etree.Element('link',
                             time=str(self.duration),
                             bidirectional=str(int(self.is_dual)),
                             enabled='1')

        _from = etree.Element('from')
        _to = etree.Element('to')

        if code_system is not None:
            if self.station_from.get_code(code_system):
                _from.set('vendor', code_system)
                _from.set('vendor_id', self.station_from.get_code(code_system))
            else:
                _from.set('station_id', str(self.station_from_id))

            if self.station_to.get_code(code_system):
                _to.set('vendor', code_system)
                _to.set('vendor_id', self.station_to.get_code(code_system))
            else:
                _to.set('station_id', str(self.station_to_id))
        else:
            _from.set('station_id', str(self.station_from_id))
            _to.set('station_id', str(self.station_to_id))

        if self.terminal_from_id:
            _from.set('terminal', self.terminal_from.code)

        if self.terminal_to_id:
            _to.set('terminal', self.terminal_to.code)

        link.append(_from)
        link.append(_to)

        return link

    class Meta:
        verbose_name = _(u'переход между станциями')
        verbose_name_plural = _(u'переходы между станциями')
        app_label = 'www'
        db_table = 'www_stationpassage'


class Station2Settlement(models.Model):
    """ Соответствия станций и городов """

    station = models.ForeignKey(Station, verbose_name=_(u'станция'))
    settlement = models.ForeignKey(Settlement, verbose_name=_(u'город'))

    class Meta:
        verbose_name = _(u'соответствиe станции и города')
        verbose_name_plural = _(u'соответствия станций и городов')
        ordering = ('settlement', 'station')
        app_label = 'www'
        db_table = 'www_station2settlement'


class StationPhone(models.Model):
    """ Телефон станции """

    station = models.ForeignKey(Station, verbose_name=_(u'станция'))
    phone = models.CharField(verbose_name=_(u'телефон'), max_length=255)
    note = models.TextField(verbose_name=_(u'примечание'),
                            null=True, blank=True, default=None)

    def __unicode__(self):
        return self.phone

    class Meta:
        verbose_name = _(u'телефон станции')
        verbose_name_plural = _(u'телефоны станций')
        ordering = ('phone',)
        app_label = 'www'


class CodeSystem(models.Model):
    u"""Система кодирования"""
    title = models.CharField(verbose_name=_(u'название системы кодирования'),
                             max_length=100, blank=False)
    code = CodeCharField(verbose_name=_(u'код системы кодирования'),
                         max_length=100, blank=False, unique=True)

    objects = PrecachingManager(keys=['pk', 'code'])

    def __unicode__(self):
        return unicode(self.title)

    class Meta:
        ordering = ('id',)
        verbose_name = _(u'система кодирования')
        verbose_name_plural = _(u'cистемы кодирования')
        app_label = 'www'
        db_table = 'www_codesystem'


class StationCode(models.Model):
    u"""Код станции"""
    station = models.ForeignKey(Station, verbose_name=_(u'станция'), blank=False,
                                null=False, related_name='code_set')
    code = CodeCharField(verbose_name=_(u"код"), blank=False, null=False,
                         max_length=50, db_index=True)
    system = models.ForeignKey(CodeSystem, verbose_name=_(u'система кодирования'),
                               blank=False, null=False)

    @classmethod
    def id_to_code_dict(cls, system):
        return dict(
            (sc.station_id, sc.code)
            for sc in cls.objects.filter(system__code=system)
        )

    @classmethod
    def id_to_code_dict_for(cls, system, stations):
        cs = CodeSystem.objects.get(code=system)

        return dict(
            (sc.station_id, sc.code)
            for sc in cls.objects.filter(system=cs, station__in=stations)
        )

    class StationCodeByStationIdGetter(object):
        """
        Класс при инициализации получает одним запросом
        данные о кодах переданных станций в заданной системе кодирования.
        И позволяет получать код станции по ее id.
        """

        def __init__(self, system_id, station_ids, default=None):
            self.default = default

            # order_by() для того, чтобы сбросить сортировку по-умолчанию и избавится от JOIN
            self.station_code_by_station_id = dict(
                StationCode.objects
                .filter(system_id=system_id, station_id__in=station_ids)
                .order_by()
                .values_list('station_id', 'code')
            )

        def get(self, station_id):
            return self.station_code_by_station_id.get(station_id, self.default)

    def __unicode__(self):
        return self.station.title + u': ' + self.system.code + u' = ' + self.code

    class Meta:
        ordering = ('station__id',)
        verbose_name = _(u'код станции (обновляется из единых справочников)')
        verbose_name_plural = _(u'коды станций (обновляются из единых справочников)')
        # В каждой системе кодирования станции имеет только один код
        unique_together = (('station', 'system'), ('system', 'code'))
        app_label = 'www'
        db_table = 'www_stationcode'


class SettlementCode(models.Model):
    u"""Код города"""
    settlement = models.ForeignKey(Settlement, verbose_name=_(u'город'), blank=False,
                                   null=False, related_name='code_set')
    code = CodeCharField(verbose_name=_(u"код"), blank=False, null=False,
                         max_length=50, db_index=True)
    system = models.ForeignKey(CodeSystem, verbose_name=_(u'система кодирования'),
                               blank=False, null=False)

    def __unicode__(self):
        return self.settlement.title + u': ' + self.system.code + u' = ' + self.code

    class Meta:
        ordering = ('settlement__id',)
        verbose_name = _(u'код города')
        verbose_name_plural = _(u'коды городов')
        # В каждой системе кодирования город имеет только один код
        unique_together = (('settlement', 'system'), ('system', 'code'))
        app_label = 'www'
        db_table = 'www_settlementcode'


class StationExpressAlias(models.Model):
    u"""Название станции в системе Экспресс"""

    station = models.ForeignKey(
        Station, verbose_name=_(u'станция'), blank=False,
        null=False, related_name='expressalias_set'
    )

    alias = CodeCharField(
        _(u'название'),
        blank=False, null=False,
        max_length=50, db_index=True, unique=True
    )

    class Meta:
        app_label = 'www'
        db_table = 'www_stationexpressalias'
        verbose_name = _(u'название в системе Экспресс')


class DirectionTranslate(AbstractTranslate):
    """
    Переводы названий направлений
    """

    @classmethod
    def get_keys(cls):
        return set(Direction.objects.values_list('title', flat=True))

    class Meta:
        ordering = ('value',)
        verbose_name = _(u'перевод названия направления')
        verbose_name_plural = _(u'переводы названий направлений')
        app_label = 'www'
        db_table = 'www_directiontranslate'


class DirectionFromTranslate(AbstractTranslate):
    """
    Переводы названий направлений туда (от Москвы) и обратно (на Москву)
    """
    @classmethod
    def get_keys(cls):
        from travel.avia.library.python.common.models.schedule import RTStation

        values = set(Direction.objects.values_list('title_from', flat=True))

        values = values.union(Direction.objects.values_list('title_to', flat=True))

        values = values.union(DirectionMarker.objects.values_list('title_from', flat=True))

        values = values.union(DirectionMarker.objects.values_list('title_to', flat=True))

        values = values.union(RTStation.objects.values_list('arrival_subdir', flat=True))

        return values.union(RTStation.objects.values_list('departure_subdir', flat=True))

    class Meta:
        ordering = ('value',)
        verbose_name = _(u'перевод названия направления туда и обратно')
        verbose_name_plural = _(u'переводы названий направлений туда и обратно')
        app_label = 'www'
        db_table = 'www_directionfromtranslate'


class SuburbanZone(models.Model):
    """ Пригородная зона """

    L_title = new_L_field(
        verbose_name=_(u"название"),
        max_length=200,
        add_local_field=True,
        local_field_critical=True,
    )

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    code = models.CharField(verbose_name=_(u"код зоны"),
                            max_length=100, blank=False, null=False,
                            unique=True)
    settlement = models.ForeignKey(Settlement, verbose_name=_(u"основной город зоны"),
                                   related_name="zone_settlement")
    title_from = models.CharField(verbose_name=_(u'название направления туда(от Москвы)'),
                                  max_length=100, blank=False)
    title_to = models.CharField(verbose_name=_(u'название направления обратно(на Москву)'),
                                max_length=100, blank=False)

    objects = PrecachingManager(
        select_related=(
            'new_L_title',
        ),
    )

    TANKER_L_FIELDS = ['title']

    def __unicode__(self):
        return self.title + u' <' + self.code + u'>'

    def markers(self):
        cache_key = settings.CACHEROOT + 'zone/%s/markers' % self.pk

        compressed = cache.get(cache_key)
        if compressed:
            pickled = zlib.decompress(compressed)
            return pickle.loads(pickled)

        markers = ExternalDirectionMarker.objects \
                                         .filter(external_direction__suburban_zone=self) \
                                         .select_related('external_direction') \
                                         .order_by()
        markers = list(markers)

        pickled = pickle.dumps(markers, protocol=pickle.HIGHEST_PROTOCOL)
        compressed = zlib.compress(pickled)
        cache.set(cache_key, compressed, settings.CACHES['default']['LONG_TIMEOUT'])

        return markers

    class Meta:
        verbose_name = _(u'пригородная зона')
        verbose_name_plural = _(u'пригородные зоны')
        app_label = 'www'
        db_table = 'www_suburbanzone'
        ordering = ('title',)


class ExternalDirection(models.Model):
    u"""Направления для электричек для показа в городах"""
    L_title = new_L_field(_(u'название направления'), add_local_field=True, local_field_critical=True, max_length=100)

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    L_full_title = new_L_field(_(u'полное название'), add_local_field=True, local_field_critical=True, max_length=100)

    new_L_full_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    code = models.CharField(verbose_name=_(u'код направления'),
                            max_length=50, blank=False, null=False, unique=True)
    suburban_zone = models.ForeignKey("SuburbanZone", blank=True, null=True,
                                      verbose_name=_(u"пригородная зона"),
                                      help_text=_(u"пригородная зона, в которую входит направление"))
    base_station = models.ForeignKey(Station, blank=True, null=True,
                                     verbose_name=_(u"базовый вокзал"))

    reduced_tariff = models.FloatField(_(u'множитель льготного тарифа'), default=0)

    use_synchronous_tariff = models.BooleanField(_(u'использовать синхронный тариф'), null=False, default=False)

    objects = PrecachingManager(
        keys=(
            'pk',
            'code',
        ),
        select_related=(
            'new_L_title',
            'new_L_full_title',
        ),
    )

    def L_cutted_full_title(self, lang=None):
        # RASP-11929
        # TODO адаптировать для переводов
        # if not lang:
        #     lang = translation.get_language()

        return self.L_title().replace(u'направление', u'напр.')

    @property
    def schema_file_name(self):
        return "%s.csv" % self.code

    schema_file = TextFileField(blank=True, verbose_name=_(u"файл с картой направления"),
                                encoding='cp1251',
                                help_text=_(u"принимаются только текстовые файлы в кодировке cp1251"))

    @property
    def title_gen(self):
        return self.title[:-2] + u'ого'

    @property
    def title_dat(self):
        return self.title[:-2] + u'ому'

    @property
    def stations(self):
        return Station.objects.filter(hidden=False,
                                      externaldirectionmarker__external_direction=self).order_by('title')

    def __unicode__(self):
        return "%s (%s) " % (self.title, self.code)

    def schema(self, edit=False):
        if not self.schema_file:
            if edit:
                return initial_schema(self)

            return None

        zones = parse_schema(self.schema_file, self.code, edit)
        # Заполним автобусные остановки моделями
        bus_sids = [int(sid)
                    for num, rows in zones
                    for row in rows
                    for item, href, label, metro, busstations in row if busstations
                    for sid in busstations]

        bus_mapping = Station.hidden_manager.filter(pk__in=bus_sids).all()
        bus_mapping = dict([(s.id, s) for s in bus_mapping])

        for num, rows in zones:
            for row in rows:
                for i, item in enumerate(row):
                    item, href, label, metro, busstations = item

                    if busstations:
                        busstations = [bus_mapping[sid] for sid in busstations if sid in bus_mapping]
                        busstations = [(s.get_popular_title(), s.get_schedule_url('bus')) for s in busstations]
                        busstations.sort(key=lambda (title, u): RE_MDOT.sub('', title).lower())

                        row[i] = item, href, label, metro, busstations

        return zones

    def parse_schema_json(self, json):
        self.schema_file = build_schema(json)

    @classmethod
    def find_for_stations(cls, *stations):
        """id направления для данных станций"""

        directions = None

        for s in stations:
            ds = set(m.external_direction_id for m in s.externaldirectionmarker_set.all())

            if directions is None:
                directions = ds
            else:
                directions.intersection_update(ds)

                # Если сет уже пустой, то дальше
                # он не пополнится
                if not directions:
                    return []

        return directions

    class Meta:
        ordering = ('title',)
        verbose_name = _(u'внешнее направление')
        verbose_name_plural = _(u'внешние направления')
        app_label = 'www'
        db_table = 'www_externaldirection'


class ExternalDirectionMarker(models.Model):
    u"""Пометка для направления(вершина графа направления)"""
    external_direction = models.ForeignKey(ExternalDirection, blank=False,
                                           verbose_name=_(u"направление"))
    station = models.ForeignKey(Station, blank=False,
                                verbose_name=_(u"станция"))
    order = models.IntegerField(verbose_name=_(u"порядок"), db_index=True)

    zone = TrimmedCharField(verbose_name=_(u'тарифная зона'), max_length=10, null=True, blank=True)
    zgroup = TrimmedCharField(verbose_name=_(u'тарифная группа'), max_length=50, null=True, blank=True)
    zprice = TrimmedCharField(verbose_name=_(u'въезд в зону'), max_length=50, null=True, blank=True)
    zprice_inner = TrimmedCharField(verbose_name=_(u'проезд внутри зоны'), max_length=50, null=True, blank=True)
    zprice_exit = TrimmedCharField(verbose_name=_(u'выезд из зоны в соседнюю'), max_length=50, null=True, blank=True)
    branch = CodeCharField(verbose_name=_(u'ветка'), max_length=50, null=True, blank=True)

    def __unicode__(self):
        return self.station.title

    def direction_name(self):
        return self.direction.title

    direction_name.short_description = _(u"направление")

    class Meta:
        ordering = ('order',)
        verbose_name = _(u'маркер внешнего направления')
        verbose_name_plural = _(u'маркеры внешних направлений')
        unique_together = (('external_direction', 'station'),
                           ('external_direction', 'order'))
        app_label = 'www'
        db_table = 'www_externaldirectionmarker'


class PointSynonym(models.Model):
    title = models.CharField(_(u'синоним'), max_length=100)
    content_type = models.ForeignKey(ContentType)
    object_id = models.PositiveIntegerField()
    point = GenericForeignKey()
    language = CodeCharField(
        _(u'язык синонима'),
        max_length=5,
        choices=[(lang, _(lang)) for lang in SUGGEST_LANGUAGES],
        null=True,
        blank=True,
    )
    search_type = models.CharField(
        max_length=20,
        null=False,
        editable=False,
        default='synonym',
        db_index=True,
        choices=[
            ('synonym', _(u'синоним')),
            ('normalized', _(u'нормализованная форма')),
            ('tokenized', _(u'токены')),
            ('exact', _(u'точная копия')),
        ],
    )

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'синоним')
        verbose_name_plural = _(u'синонимы')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_pointsynonym'
        unique_together = ('title', 'content_type', 'object_id')


class PseudoRegion(models.Model):
    title = models.CharField(_(u"название"), max_length=255, null=False, blank=False)
    countries = models.ManyToManyField(Country, blank=True, verbose_name=_(u"страны"))

    class Meta:
        verbose_name = _(u"псевдо-регион")
        verbose_name_plural = _(u"псевдо-регионы")
        app_label = 'www'
        db_table = 'www_pseudoregion'

    def __unicode__(self):
        return self.title


PseudoRegion.countries.through._meta.verbose_name = _(u"страна псевдорегиона")
PseudoRegion.countries.through._meta.verbose_name_plural = _(u"страны псевдорегиона")


class ReplaceException(models.Model):
    u"""Исключения из колдунств [RASP-5359]"""
    city_from = models.ForeignKey(Settlement, verbose_name=_(u"город отправления"), related_name="city_from")
    city_to = models.ForeignKey(Settlement, verbose_name=_(u"город прибытия"), blank=True, null=True, related_name="city_to")
    station_to = models.ForeignKey(Station, verbose_name=_(u"станция прибытия"), blank=True, null=True, related_name="station_to")

    class Meta:
        verbose_name = _(u'исключение колдунства')
        verbose_name_plural = _(u'исключения колдунства')
        app_label = 'www'
        db_table = 'www_replaceexception'

    _exceptions = None

    @classmethod
    def precache(cls):
        cls._exceptions = list(cls.objects.all())

    @classmethod
    def check_points(cls, point_from, point_to):
        if cls._exceptions is None:
            cls.precache()

        for ex in cls._exceptions:
            if point_from == ex.city_from and (point_to == ex.city_to or point_to == ex.station_to):
                return True

        return False


class District(BaseDistrict):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.geo'

    class Meta(BaseDistrict.Meta):
        app_label = 'www'


def check_point_in_point(examining_point, container_point):
    examining_class = examining_point.__class__.__name__.lower()
    container_class = container_point.__class__.__name__.lower()

    return get_compare_function(examining_class, container_class)(examining_point, container_point)


def get_compare_function(examining_class, container_class):
    try:
        return globals()['check_%s_in_%s' % (examining_class, container_class)]
    except KeyError:
        raise Exception("Can not check that %s in %s" % (examining_class, container_class))


def check_point_in_settlement(point, settlement):
    if isinstance(point, Settlement):
        return point == settlement
    elif isinstance(point, Station):
        return point.settlement_id == settlement.id
    elif isinstance(point, (Country, Region)):
        return False
    else:
        raise TypeError('Can not compare %s and %s' % (point.__class__.__name__, settlement.__class__.__name__))


check_station_in_settlement = check_settlement_in_settlement = check_point_in_settlement
check_region_in_settlement = check_country_in_settlement = check_point_in_settlement


def check_point_in_country(point, country):
    if isinstance(point, (Station, Settlement, Region)):
        return point.country_id == country.id
    elif isinstance(point, Country):
        return point == country
    else:
        raise TypeError('Can not compare %s and %s' % (point.__class__.__name__, country.__class__.__name__))

check_station_in_country = check_settlement_in_country = check_region_in_country = check_country_in_country =\
    check_point_in_country


def check_point_in_pseudoregion(point, pseudo_region):
    countries = list(country.id for country in pseudo_region.countries.all())
    if isinstance(point, (Station, Settlement, Region)):
        return point.country_id in countries
    elif isinstance(point, Country):
        return point.id in countries
    else:
        raise TypeError('Can not compare %s and %s' % (point.__class__.__name__, pseudo_region.__class__.__name__))


check_station_in_pseudoregion = check_settlement_in_pseudoregion = check_point_in_pseudoregion
check_region_in_pseudoregion = check_country_in_pseudoregion = check_point_in_pseudoregion


class GortransCityLink(models.Model):
    city = models.ForeignKey(Settlement, null=False, blank=False, verbose_name=_(u'город'))
    url = models.CharField(u"Ссылка", max_length=255, null=False, blank=False)

    def url_a(self):
        return '<a href="%s" target="_blank">%s</a>' % (self.url, self.url)

    url_a.allow_tags = True

    def __unicode__(self):
        return self.city.title

    class Meta:
        verbose_name = _(u"ссылка на гортранс")
        verbose_name_plural = _(u"ссылки на гортранс")
        app_label = 'www'
        db_table = 'www_gortranscitylink'


class Direction(models.Model):
    """
    Внутренние направления для электричек, используются для расчетов направлений в расписании.
    Больше нигде отображаться не должны
    """
    title = models.CharField(verbose_name=_(u'название направления'),
                             max_length=100, blank=False)
    L_title = DirectionTranslate.get_L_method(key_field='title')
    title_from = models.CharField(verbose_name=_(u'название направления туда(от Москвы)'),
                                  max_length=100, blank=False)
    L_title_from = DirectionFromTranslate.get_L_method(key_field='title_from')
    title_to = models.CharField(verbose_name=_(u'название направления обратно(на Москву)'),
                                max_length=100, blank=False)
    L_title_to = DirectionFromTranslate.get_L_method(key_field='title_to')
    code = models.CharField(verbose_name=_(u'код направления'),
                            max_length=50, blank=False, null=False, unique=True)
    suburban_zone = models.ForeignKey("SuburbanZone", blank=True, null=True,
                                      verbose_name=_(u"пригородная зона"),
                                      help_text=_(u"пригородная зона, в которую входит направление"))

    objects = PrecachingManager(keys=['pk', 'code'])

    @property
    def title_gen(self):
        return self.title[:-2] + u'ого'

    @property
    def stations(self):
        return Station.objects.filter(hidden=False,
                                      directionmarker__direction=self).order_by('title')

    def __unicode__(self):
        return self.title

    class Meta:
        ordering = ('title',)
        verbose_name = _(u'направление')
        verbose_name_plural = _(u'направления')
        app_label = 'www'
        db_table = 'www_direction'


ARROW_CHOICES = (
    ('', _(u"не указано")),
    ('right', _(u"направо")),
    ('left', _(u"налево")),
    ('up', _(u"вверх")),
    ('down', _(u"вниз")),
    ('right-up', _(u"направо-вверх")),
    ('right-down', _(u"направо-вниз")),
    ('left-up', _(u"налево-вверх")),
    ('left-down', _(u"налево-вниз"))
)


class DirectionMarker(models.Model):
    u"""Пометка для направления(вершина графа направления)"""
    direction = models.ForeignKey(Direction, blank=False,
                                  verbose_name=_(u"направление"))
    station = models.ForeignKey(Station, blank=True,
                                verbose_name=_(u"станция"), null=True,
                                help_text=_(u"если станция не указана, то это стрелка"))
    title_from = models.CharField(verbose_name=_(u"туда"),
                                  max_length=100, blank=True, null=True,
                                  help_text=_(u"название направления туда(от Москвы). "
                                              u"Используется для определения направления "
                                              u"на предыдущей станции((от Москвы) на Катышку)"))
    L_title_from = DirectionFromTranslate.get_L_method(key_field='title_from')
    title_to = models.CharField(verbose_name=_(u"обратно"),
                                max_length=100, blank=True, null=True,
                                help_text=_(u"название направления обратно(на Москву). "
                                            u"Используется для определения направления "
                                            u"на предыдущей станции((на Москву) на Бутку)"))
    L_title_to = DirectionFromTranslate.get_L_method(key_field='title_to')
    order = models.IntegerField(verbose_name=_(u"порядок"), db_index=True)
    arrival_title_from = models.CharField(verbose_name=_(u"прибытие Туда"),
                                          max_length=100, blank=True, null=True,
                                          help_text=_(u"название направления туда(от Москвы). "
                                                      u"Используется для определения направления прибытия "
                                                      u"на текущей станции((от Москвы) на Катышку) иначе берется текст из Туда"))
    arrival_title_to = models.CharField(verbose_name=_(u"прибытие Обратно"),
                                        max_length=100, blank=True, null=True,
                                        help_text=_(u"название направления обратно(на Москву). "
                                                    u"Используется для определения направления прибытия "
                                                    u"на текущей станции((на Москву) на Бутку) иначе берется текст из Обратно"))
    prev = models.ForeignKey("DirectionMarker", blank=True, null=True,
                             verbose_name=_(u"предыдущая станция"),
                             related_name='next_set')
    next = models.ForeignKey("DirectionMarker", blank=True, null=True,
                             verbose_name=_(u"следующая станция"),
                             related_name='prev_set')
    # Задаем в mysql как
    arrow_type = models.CharField(verbose_name=_(u"тип ответвления"),
                                  choices=ARROW_CHOICES,
                                  max_length=100, null=True, blank=True)
    arrow_title = models.CharField(blank=True, null=True, max_length=100,
                                   verbose_name=_(u"название стрелки"))

    def save(self, **kwargs):
        self.arrow_type = self.arrow_type or None
        super(DirectionMarker, self).save(**kwargs)

    def __unicode__(self):
        return self.station and self.station.title or (u'-> ' + self.arrow_title)

    def direction_name(self):
        return self.direction.title

    direction_name.short_description = _(u"направление")

    class Meta:
        ordering = ('order',)
        verbose_name = _(u'маркер направления')
        verbose_name_plural = _(u'маркеры направлений')
        unique_together = (('direction', 'station'), ('direction', 'order'))
        app_label = 'www'
        db_table = 'www_directionmarker'


class CompanyOffice(models.Model):
    title = models.CharField(verbose_name=_(u'Название'), max_length=255)
    company = models.ForeignKey('www.Company', verbose_name=_(u'Компания перевозчик'))
    settlement = models.ForeignKey('www.Settlement', verbose_name=_(u'Город'))
    address = models.CharField(max_length=255, verbose_name=_(u'Адресс'))

    main_station = models.ForeignKey('www.Station', verbose_name=_(u'Главная станция'),
                                     null=True, blank=True, default=None)
    contact_info = models.TextField(verbose_name=_(u'Контактная информация'),
                                    null=False, blank=True, default='')
    phone = TrimmedCharField(verbose_name=_(u'Контактный телефон'), max_length=255,
                             null=False, blank=True, default='')
    phone_booking = TrimmedCharField(verbose_name=_(u'Телефон для бронирования'),
                                     max_length=255, blank=True, default='')
    description = models.TextField(verbose_name=_(u'Описание'), blank=True, default='')
    longitude = models.FloatField(verbose_name=_(u'Долгота'), null=True, blank=True, default=None)
    latitude = models.FloatField(verbose_name=_(u'Широта'), null=True, blank=True, default=None)
    is_main = models.BooleanField(verbose_name=_(u'Главный офис'), default=False)

    def __unicode__(self):
        return self.title

    def clean(self):
        # У перевозчика может быть только один главный офис
        if self.is_main and self.__class__.objects.filter(company=self.company, is_main=True).exists():
            raise ValidationError(_(u'У компании-перевозчика %s уже есть главый офис' % self.company))

    class Meta:
        verbose_name = _(u'Офис компании-перевозчика')
        verbose_name_plural = _(u'Офисы компаний-перевозчиков')
        app_label = 'www'


class SettlementImage(BaseSettlementImage):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.api.models'

    class Meta(BaseSettlementImage.Meta):
        verbose_name = _(u'Изображение города')
        verbose_name_plural = _(u'Изображение городов')
        app_label = 'api'
        db_table = 'api_settlementimage'
