# -*- coding: utf-8 -*-
from datetime import datetime

from django.conf import settings
from django.core.exceptions import MultipleObjectsReturned, ObjectDoesNotExist
from django.core.files.storage import get_storage_class
from django.core.validators import MaxValueValidator, MinValueValidator, ValidationError
from django.db import models
from django.db import transaction
from django.utils.deconstruct import deconstructible
from django.utils.translation import ugettext_lazy as _
from simple_history.models import HistoricalRecords

from travel.avia.library.python.avia_data.storage import AvatarsPngStorage, AvatarsSvgStorage
from travel.avia.library.python.common.models.base_partner_model import BasePartner, BaseVariantsTTL, DEFAULT_CACHE_TTL
from travel.avia.library.python.common.models.geo import check_point_in_point
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils import ConstField
from travel.avia.library.python.common.utils.caching import cache_until_switch_thread_safe
from travel.avia.library.python.common.utils.environment import get_request
from travel.avia.library.python.common.utils.fields import CodeCharField, TrimmedCharField
from travel.avia.library.python.common.utils.mathematic import round_dispersed
from travel.avia.library.python.common.utils.media_fields import CustomImageField
from travel.avia.library.python.common.utils import environment


if settings.ENVIRONMENT not in ['dev', 'testing', 'production']:
    raise EnvironmentError(u'Неверно указана среда выполнения')

MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES = 1 * 1024 * 1024

# Минимальный допустимый поправочный коэффициент стоимости клика вследствие падения размера среднего чека.
MINIMAL_AVG_CHECK_COEFF = 0.5
# Максимальный допустимый поправочный коэффициент стоимости клика вследствие роста размера среднего чека.
MAXIMAL_AVG_CHECK_COEFF = 2.0


svg_storage = AvatarsSvgStorage()
png_storage = AvatarsPngStorage()


@deconstructible
class PartnerLogoFileSystemStorage(get_storage_class()):
    def get_available_name(self, name, max_length=None):
        if self.exists(name):
            self.delete(name)

        return name


@deconstructible
class PartnerLogoUploadToGetter(object):
    def __init__(self, national):
        self.national = national

    def __call__(self, instance, filename):
        return 'avia/partners/p/logo/pngold/{}.png'.format(
            '_'.join(filter(None, [instance.code, self.national])),
        )


@deconstructible
class PartnerLogoSvgUploadToGetter(object):
    def __init__(self, national):
        self.national = national

    def __call__(self, instance, filename):
        return 'avia/partners/p/logo/svg/id{}_{}.svg'.format(instance.id, self.national)


@deconstructible
class PartnerIconSvgUploadToGetter(object):
    def __call__(self, instance, filename):
        return 'avia/partners/p/logo/icon/id{}.svg'.format(instance.id)


@deconstructible
class LogoSvg2PngUploadToGetter(object):
    def __init__(self, national):
        self.national = national

    def __call__(self, instance, filename):
        return 'avia/partners/p/logo/png/id{}_{}.png'.format(instance.id, self.national)


@deconstructible
class DohopVendorLogoSvgUploadToGetter(object):
    def __init__(self, national):
        self.national = national

    def __call__(self, instance, filename):
        return 'avia/partners/dohop/logo/svg/id{}_{}.svg'.format(instance.id, self.national)


@deconstructible
class DohopVendorLogoSvg2PngUploadToGetter(object):
    def __init__(self, national):
        self.national = national

    def __call__(self, instance, filename):
        return 'avia/partners/dohop/logo/png/id{}_{}.png'.format(instance.id, self.national)


@deconstructible
class DohopVendorIconSvgUploadToGetter(object):
    def __call__(self, instance, filename):
        return 'avia/partners/dohop/logo/icon/id{}.svg'.format(instance.id)


class Partner(BasePartner):

    history = HistoricalRecords(use_base_model_db=True)

    @staticmethod
    def get_LogoSvgUploadToGetter(national_version):
        return PartnerLogoSvgUploadToGetter(national_version)

    @staticmethod
    def get_LogoSvg2PngUploadToGetter(national_version):
        return LogoSvg2PngUploadToGetter(national_version)

    @staticmethod
    def get_IconSvgUploadToGetter():
        return PartnerIconSvgUploadToGetter()

    variant_cache_ttl = models.PositiveSmallIntegerField(
        verbose_name=_(u'Время жизни кеша предложений, в минутах'),
        blank=False,
        null=False,
        default=DEFAULT_CACHE_TTL,
        validators=[MinValueValidator(3), MaxValueValidator(180)],
    )

    billing_datasource_id_dev = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга dev'),
                                                    blank=True, null=True, default=None, help_text=_(u"datasource ID"))

    billing_datasource_id_testing = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга testing'),
                                                        blank=True, null=True, default=None,
                                                        help_text=_(u"datasource ID"))

    billing_datasource_id_production = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга production'),
                                                           blank=True, null=True, default=None,
                                                           help_text=_(u"datasource ID"))

    click_price = models.FloatField(verbose_name=_(u'стоимость клика, у.е.'), blank=True, null=False, default=0)

    click_price_ru = models.FloatField(
        verbose_name=_(u'стоимость клика (ru) у.е.'),
        blank=True, null=False, default=0
    )

    click_price_ua = models.FloatField(
        verbose_name=_(u'стоимость клика (ua) у.е.'),
        blank=True, null=False, default=0
    )

    click_price_kz = models.FloatField(
        verbose_name=_(u'стоимость клика (kz) у.е.'),
        blank=True, null=False, default=0
    )

    click_price_tr = models.FloatField(
        verbose_name=_(u'стоимость клика (tr) у.е.'),
        blank=True, null=False, default=0
    )

    click_price_com = models.FloatField(
        verbose_name=_(u'стоимость клика (com) у.е.'),
        blank=True, null=False, default=0
    )

    media_storage = PartnerLogoFileSystemStorage()

    logo = CustomImageField(
        verbose_name=_(u'Основной логотип'),
        upload_to=PartnerLogoUploadToGetter(''),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_ru = CustomImageField(
        verbose_name=_(u'Логотип (ru)'),
        upload_to=PartnerLogoUploadToGetter('ru'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_ua = CustomImageField(
        verbose_name=_(u'Логотип (ua)'),
        upload_to=PartnerLogoUploadToGetter('ua'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_tr = CustomImageField(
        verbose_name=_(u'Логотип (tr)'),
        upload_to=PartnerLogoUploadToGetter('tr'),
        storage=media_storage, null=False, blank=True, default=''
    )

    current_balance = models.FloatField(verbose_name=_(u'текущий баланс, у.е.'), blank=True, null=False, default=0.0)

    notify_balance_threshold = models.FloatField(
        verbose_name=_(u'порог баланса для уведомления, у.е.'), blank=True, null=True, default=None)
    low_balance_notification_sent = models.DateTimeField(verbose_name=_(u"Уведомление о низком балансе послано"),
                                                         blank=False, null=False, default=datetime(2011, 1, 1))
    null_balance_notification_sent = models.DateTimeField(
        verbose_name=_(u"Уведомление о нулевом балансе послано"),
        blank=False, null=False, default=datetime(2011, 1, 1)
    )
    statistic_reminder_sent = models.DateTimeField(
        verbose_name=_(u"Напоминание про статистику продаж послано"),
        blank=False, null=False, default=datetime(2011, 1, 1)
    )

    balance_updated_at = models.DateTimeField(
        verbose_name=_(u'баланс обновлен'),
        blank=True, null=True, default=None
    )

    enabled_with_negative_balance = models.BooleanField(
        verbose_name=_(u'Не отключать партнёра при достижении им отрицательного баланса'),
        blank=False, null=False, default=False,
    )

    disabled = models.BooleanField(verbose_name=_(u'принудительно отключить партнера'),
                                   blank=False, null=False, default=False)
    hidden = models.BooleanField(verbose_name=_(u'скрываем партнера'),
                                 blank=False, null=False, default=False)

    start_unavailability_datetime = models.DateTimeField(
        verbose_name=_(u'Дата начала технических работ'),
        blank=True, null=True, default=None
    )

    end_unavailability_datetime = models.DateTimeField(
        verbose_name=_(u'Дата конца технических работ'),
        blank=True, null=True, default=None
    )

    t_type = models.ForeignKey(
        "www.TransportType", verbose_name=_(u'тип транспорта'),
        default=2, related_name='order_partner'
    )

    _billing_datasource_id_attr = 'billing_datasource_id_%s' % settings.ENVIRONMENT

    query_module_name = models.CharField(
        _(u"Версия модуля"), max_length=255, default=None,
        blank=True, null=True
    )

    can_fetch_by_daemon = models.BooleanField(
        _(u'Может быть запрошен через демона'), blank=False, null=False, default=True,
        help_text=_(u'Обычно этот чекбокс должен быть включен.'
                    u'Отключайте только для ж/д партнеров, которые запрашиваются не через демон.')
    )

    billing_order_id = models.IntegerField(
        verbose_name=_(u'Номер заказа в партнерке'),
        blank=True,
        null=True,
        default=None,
        unique=True,
    )

    billing_client_id = models.IntegerField(
        verbose_name=_(u'ID клиента у баланса'),
        blank=True,
        null=True,
        default=None,
        unique=True,
    )

    marker = CodeCharField(
        verbose_name=_(u'Имя get-параметра для партнерского маркера'),
        max_length=50,
        blank=True,
        default='marker',
        null=True,
        help_text=_(
            u'Если партнер не поддерживает маркеры, то поле должно быть пустым.'
            u'Нельзя добавлять произвольное имя маркера, так как вы можете сломать работу партнера.'
        )
    )

    pricing_model = models.CharField(
        verbose_name=_(u'Модель ценообразования'),
        max_length=5,
        choices=[
            ('cpc', 'cpc'),
            ('cpa', 'cpa'),
        ],
        null=True,
        blank=False,
        default='cpc',
    )

    use_in_update_conversions = models.BooleanField(
        verbose_name=_(u'Учитывать в расчете конверсии'),
        null=False, blank=False, default=False,
    )

    can_sale_by_installments = models.BooleanField(
        verbose_name=_(u'Есть возможность продавать билеты в рассрочку'),
        null=False, blank=False, default=False,
    )

    is_deleted = models.BooleanField(
        verbose_name=_(u'Удаленный партнер'),
        help_text=_(
            u'Остался в админке по чисто техническим причинам. Перед тем, как редактировать это поле, надо '
            u'проставить партнеру disabled в партнерке.'),
        null=False, blank=False, default=False,
    )

    class Meta(BasePartner.Meta):
        app_label = 'order'
        verbose_name = _(u"партнер")
        verbose_name_plural = _(u"партнеры")

    def disable(self, yandex_login, role):
        self._update_status(yandex_login, role, disabled=True)

    def enable(self, yandex_login, role):
        self._update_status(yandex_login, role, disabled=False)

    def _update_status(self, yandex_login, role, disabled):
        if self.disabled == disabled:
            return

        self.disabled = disabled
        with transaction.atomic():
            UpdateHistoryRecord(
                partner=self,
                updater_yandex_login=yandex_login,
                updater_role=role,
                action=UpdateHistoryRecord.CHANGE_ACTIVE_STATUS_ACTION,
                description='disabled: {}'.format(disabled)
            ).save(using='writable')
            self.save(using='writable')

    def update_unavailability_range(self, start_unavailability_datetime,
                                    end_unavailability_datetime, yandex_login, updater_role, db_name='writable'):
        if (self.start_unavailability_datetime == start_unavailability_datetime and
                self.end_unavailability_datetime == end_unavailability_datetime):
            return

        with transaction.atomic():
            description = ('start_unabaliable_datetime:[{}], end_unabaliable_datetime:[{}]'
                           .format(start_unavailability_datetime, end_unavailability_datetime))
            UpdateHistoryRecord(
                partner=self,
                updater_yandex_login=yandex_login,
                updater_role=updater_role,
                action=UpdateHistoryRecord.CHANGE_UNAVAILABILITY_RANGE_ACTION,
                description=description
            ).save(using=db_name)

            self.start_unavailability_datetime = start_unavailability_datetime
            self.end_unavailability_datetime = end_unavailability_datetime
            self.save(using=db_name)

    @classmethod
    def find_by_billing_datasource_id(cls, id_):
        return cls.objects.get(**{cls._billing_datasource_id_attr: id_})

    @property
    def billing_datasource_id(self):
        return getattr(self, self._billing_datasource_id_attr)

    def set_billing_datasource_id(self, value):
        setattr(self, self._billing_datasource_id_attr, value)

    def get_national_price(self, national_version):
        cache_field = 'clk_price_obj_%s' % national_version

        if not hasattr(self, cache_field):
            setattr(
                self, cache_field,
                ClickPrice(self, national_version)
            )

        return getattr(self, cache_field)

    @property
    def price(self):
        if not hasattr(self, 'clk_price_obj'):
            self.clk_price_obj = ClickPrice(self)

        return self.clk_price_obj

    @property
    def enabled(self):
        if self.code in ['dohop']:
            return True

        if self.start_unavailability_datetime and self.end_unavailability_datetime:
            now = environment.now()
            if self.start_unavailability_datetime <= now <= self.end_unavailability_datetime:
                return False

        return not self.disabled

    def get_national_click_price(self, national_version):
        field_name = 'click_price_{}'.format(national_version)

        return getattr(self, field_name)

    def get_national_logo(self, national_version=None):
        if not national_version:
            national_version = get_request().NATIONAL_VERSION

        field_name = 'logo_{}'.format(
            national_version
        )

        national_logo = getattr(self, field_name)

        logo = national_logo or self.logo

        return logo.url if logo else None

    def get_national_logo_svg(self, national_version=None):
        if not national_version:
            national_version = get_request().NATIONAL_VERSION

        field_name = 'logo2_svg_{}'.format(
            national_version
        )

        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_national_logo_svg2png(self, national_version):
        field_name = 'logo2_svg2png_{}'.format(national_version)
        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_staticpages_partner(self, national_version, lang, t_type_code, services):
        try:
            return self.staticpages_partners.get(
                national_version=national_version,
                t_type__code=t_type_code,
                is_published=True,
                services__in=services
            )

        except (MultipleObjectsReturned, ObjectDoesNotExist):
            return None

    def enabled_in(self, service, national_version, mobile=False):
        if mobile:
            return getattr(self, 'enabled_in_mobile_%s_%s' % (service, national_version), False)

        return getattr(self, 'enabled_in_%s_%s' % (service, national_version), False)

    def __json__(self, extended=False):
        return {
            'title': self.L_national_title() or self.title,
            'logo': self.get_national_logo(),
        }

    @classmethod
    def get_actual(cls, t_type_code, national_version='ru', from_rasp=False, mobile=False):
        partners = Partner.objects.filter(t_type__code=t_type_code, can_fetch_by_daemon=True)

        service = 'rasp' if from_rasp else 'ticket'

        return [
            p for p in partners
            if p.enabled and p.enabled_in(service, national_version, mobile)
        ]

    @classmethod
    @cache_until_switch_thread_safe
    def get_info(cls, code):
        partner = cls.objects.get(code=code)
        return partner.title, partner.site_url

    @classmethod
    def get_train_partner_code(cls, point_from, point_to, national_version):
        if not point_from.country or not point_to.country:
            return None

        # RASP-14147
        if national_version == 'ru':
            if point_from.country.code == point_to.country.code == 'UA':
                return 'ukrmintrans_train'

            # RASP-8130: не обращаться к УФС если обе станции вне РФ
            if point_from.country.code == 'RU' or point_to.country.code == 'RU':
                return 'ufs'

        elif national_version == 'ua':
            if point_from.country.code == point_to.country.code == 'RU':
                return 'ufs'

            if point_from.country.code == 'UA' or point_to.country.code == 'UA':
                return 'ukrmintrans_train'

        return None


# Deprecated, please use UpdateHistoryRecord
# TODO kill after 11.01.2016
class PartnerStatusUpdate(models.Model):
    class Meta:
        app_label = 'order'
        db_table = 'order_partnerstatusupdate'
        verbose_name = _(u'обновление статуса партнёра')
        verbose_name_plural = _(u'обновления статусов партнёров в партнёрке')

    UPDATER_ROLES = tuple(
        [(v, v) for v in ['admin', 'manager', 'helper', 'owner']]
    )

    partner = models.ForeignKey(
        Partner, blank=False, null=False, db_index=True,
        related_name='status_updates'
    )
    disabled = models.BooleanField(
        blank=False, null=False
    )
    updater_yandex_login = models.CharField(
        blank=False, null=False, max_length=255
    )
    updater_role = models.CharField(
        blank=False, null=False, choices=UPDATER_ROLES, max_length=255
    )
    time = models.DateTimeField(
        blank=False, default=datetime.utcnow, db_index=True
    )


class UpdateHistoryRecord(models.Model):
    class Meta:
        app_label = 'order'
        db_table = 'order_updatehistoryrecords'
        verbose_name = _(u'информация о обновлении чего-либо кем-то')
        verbose_name_plural = _(u'информация о обновлении чего-либо кем-то')

    UPDATER_ROLES = tuple(
        [(v, v) for v in ['admin', 'manager', 'helper', 'owner']]
    )

    CHANGE_ACTIVE_STATUS_ACTION = 'change-active-status'
    CHANGE_UNAVAILABILITY_RANGE_ACTION = 'change-unavailability-range'
    CHANGE_HIDDEN_STATUS = 'change-hidden-status'
    ACTIONS = tuple(
        [(v, v) for v in [CHANGE_ACTIVE_STATUS_ACTION, CHANGE_UNAVAILABILITY_RANGE_ACTION, CHANGE_HIDDEN_STATUS]]
    )

    partner = models.ForeignKey(
        Partner, blank=False, null=False, db_index=True,
        related_name='history_update_records'
    )
    action = models.CharField(
        blank=False, null=False, choices=ACTIONS, max_length=100
    )
    description = models.CharField(
        blank=False, null=False, max_length=100
    )
    updater_yandex_login = models.CharField(
        blank=False, null=False, max_length=255
    )
    updater_role = models.CharField(
        blank=False, null=False, choices=UPDATER_ROLES, max_length=255
    )
    time = models.DateTimeField(
        blank=False, default=datetime.utcnow, db_index=True
    )


class ClickPrice(object):
    def __init__(self, partner_object, national_version=None):
        self.partner = partner_object
        self.national_version = national_version

    @property
    def cents(self):
        return int(self.dollars * 100)

    def get_cents_dispersed(self):
        return int(round_dispersed(float(self.dollars) * 100))

    @property
    def dollars(self):
        if self.national_version:
            return self.partner.get_national_click_price(self.national_version)

        return self.partner.click_price

    def __str__(self):
        return self.dollars


class EmailType(models.Model):
    code = models.CharField(
        verbose_name=u'Код', max_length=24, null=False, blank=False,
        unique=True,
    )

    title = models.CharField(
        verbose_name=u'Тип емейла',
        null=False, blank=True, default='', max_length=64
    )

    class Meta:
        app_label = 'order'

    def __unicode__(self):
        return u'{}'.format(self.title)


def get_base_email_type():
    return EmailType.objects.get(code='base').id


class PartnerEmail(models.Model):
    partner = models.ForeignKey(Partner, related_name='emails')
    emailtype = models.ForeignKey(EmailType, related_name='emails', default=get_base_email_type)
    email = models.EmailField(blank=False)

    class Meta:
        app_label = 'order'
        verbose_name = _(u"Email поставщика")
        verbose_name_plural = _(u"Email'ы поставщика")


class RegionalizePartnerQueryRule(models.Model):
    check_fields = ('settlement', 'country', 'pseudo_region')

    partner = models.ForeignKey(Partner, null=False, blank=False, verbose_name=_(u"партнер"))
    exclude = models.BooleanField(verbose_name=_(u'кроме'), default=False)

    settlement_from = models.ForeignKey('www.Settlement', null=True, blank=True,
                                        related_name=u"regionalize_query_from",
                                        verbose_name=_(u"город отправления"))

    country_from = models.ForeignKey('www.Country', null=True, blank=True,
                                     related_name=u"regionalize_query_from",
                                     verbose_name=_(u"страна отправления"))

    pseudo_region_from = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                           related_name=u"regionalize_query_from",
                                           verbose_name=_(u"псевдорегион отправления"))

    settlement_to = models.ForeignKey('www.Settlement', null=True, blank=True,
                                      related_name=u"regionalize_query_to",
                                      verbose_name=_(u"город прибытия"))

    country_to = models.ForeignKey('www.Country', null=True, blank=True,
                                   related_name=u"regionalize_query_to",
                                   verbose_name=_(u"страна прибытия"))

    pseudo_region_to = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                         related_name=u"regionalize_query_to",
                                         verbose_name=_(u"псевдорегион прибытия"))

    user_settlement = models.ForeignKey('www.Settlement', null=True, blank=True,
                                        related_name=u"regionalize_query_user",
                                        verbose_name=_(u"город пользователя"))

    user_country = models.ForeignKey('www.Country', null=True, blank=True,
                                     related_name=u"regionalize_query_user",
                                     verbose_name=_(u"страна пользователя"))

    user_pseudo_region = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                           related_name=u"regionalize_query_user",
                                           verbose_name=_(u"псевдорегион пользователя"))

    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=(('*', '*'),) + settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='*',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    start_date = models.DateField(
        verbose_name=_(u'начало периода'), null=True, blank=True, default=None
    )
    end_date = models.DateField(
        verbose_name=_(u'конец периода'), null=True, blank=True, default=None
    )
    week_days = models.CharField(
        verbose_name=_(u'дни недели'), null=True, blank=True, default=None,
        max_length=7,
        help_text=_(u'строка с номерами дней недели (1 - пн, 2 - вт, ..., 7 - вс). Пример: 123456'),
    )

    class Meta:
        app_label = 'order'
        verbose_name = _(u"правило регионализации")
        verbose_name_plural = _(u"регионализация запросов к партнерам")

    def __unicode__(self):
        return _(u"Регионализация %s") % self.partner.code

    def get_point_from(self):
        for field_name in self.check_fields:
            if getattr(self, '%s_from_id' % field_name):
                return getattr(self, '%s_from' % field_name)

    def get_point_to(self):
        for field_name in self.check_fields:
            if getattr(self, '%s_to_id' % field_name):
                return getattr(self, '%s_to' % field_name)

    def get_user_point(self):
        for field_name in self.check_fields:
            if getattr(self, 'user_%s_id' % field_name):
                return getattr(self, 'user_%s' % field_name)

    def is_applicable_to_query(self, query):
        point_from_container = self.get_point_from()
        if point_from_container and not check_point_in_point(query.point_from, point_from_container):
            return False

        point_to_container = self.get_point_to()
        if point_to_container and not check_point_in_point(query.point_to, point_to_container):
            return False

        user_point_container = self.get_user_point()
        if user_point_container and query.user_settlement and \
                not check_point_in_point(query.user_settlement, user_point_container):
            return False

        return True


class BlablacarToken(models.Model):
    token = models.TextField(verbose_name=_(u'Токен'), null=False, blank=False)

    expire_msk_dt = models.DateTimeField(verbose_name=_(u'Дата окончания действия токена'),
                                         blank=False, null=False)

    receive_msk_dt = models.DateTimeField(verbose_name=_(u'Дата получения токена'),
                                          blank=False, null=False)

    @classmethod
    def get_latest_token_or_none(cls):
        try:
            return cls.objects.latest('expire_msk_dt')

        except cls.DoesNotExist:
            return None

    def __unicode__(self):
        return u'<Token: %s %s>' % (self.expire_msk_dt, self.token[:8])

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Токен для блаблакар')
        verbose_name_plural = _(u'Токены для блаблакар')


class DohopVendor(BasePartner):

    history = HistoricalRecords(use_base_model_db=True)

    description_ru_ru = models.TextField(
        verbose_name=_(u'Описание в русской версии'), null=True, blank=True
    )

    description_ua_uk = models.TextField(
        verbose_name=_(u'Описание в украинской версии (uk)'), null=True, blank=True
    )

    description_ua_ru = models.TextField(
        verbose_name=_(u'Описание в украинской версии (ru)'), null=True, blank=True
    )

    description_tr_tr = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (tr)'), null=True, blank=True
    )

    description_tr_en = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (en)'), null=True, blank=True
    )

    dohop_code = models.CharField(_(u'Код вендора'), max_length=100, null=False, blank=False)
    dohop_id = models.PositiveIntegerField(_(u'ID вендора'), null=False, blank=False, db_index=True)

    # FIXME drop deprecated
    dohop_vendor_info = models.TextField(null=False, blank=False, )
    dohop_residences = models.TextField(null=False, blank=False)
    dohop_langs = models.CharField(max_length=255, null=False, blank=False)

    enabled = models.BooleanField(verbose_name=_(u'Может быть запрошен через демона'),
                                  blank=False, null=False, default=False)
    dohop_cache_ttl = models.PositiveSmallIntegerField(verbose_name=_(u'Время жизни кеша, минут'),
                                                       blank=False, null=False, default=DEFAULT_CACHE_TTL,
                                                       validators=[MinValueValidator(3), MaxValueValidator(180)])

    t_type = models.ForeignKey("www.TransportType", verbose_name=_(u'тип транспорта'),
                               default=2, related_name='+')

    partner = models.ForeignKey(
        Partner, verbose_name=_(u'Основной партнер'), null=True
    )

    billing_datasource_id_dev = models.IntegerField(
        verbose_name=_(u'iD партнера в системе биллинга dev'),
        blank=True, null=True, default=None,
        help_text=_(u"datasource ID")
    )

    billing_datasource_id_testing = models.IntegerField(
        verbose_name=_(u'iD партнера в системе биллинга testing'),
        blank=True, null=True, default=None,
        help_text=_(u"datasource ID")
    )

    billing_datasource_id_production = models.IntegerField(
        verbose_name=_(u'iD партнера в системе биллинга production'),
        blank=True, null=True, default=None,
        help_text=_(u"datasource ID")
    )

    # Deprecated
    click_price = models.FloatField(
        verbose_name=_(u'стоимость клика, у.е.'),
        blank=False, null=False, default=0
    )

    pricing_model = ConstField('cpa')

    @property
    def marker(self):
        return self.partner.marker

    @property
    def query_module_name(self):
        return self.partner.query_module_name

    def enabled_in(self, service, national_version, mobile=False):
        if mobile:
            return getattr(self, 'enabled_in_mobile_%s_%s' % (service, national_version), False)

        return getattr(self, 'enabled_in_%s_%s' % (service, national_version), False)

    # Автоматически заполняется из dohop_id в save()
    @classmethod
    def get_code_from_dohop_id(cls, dohop_id):
        return 'dohop_%s' % dohop_id

    def save(self, *args, **kwargs):
        self.code = self.get_code_from_dohop_id(self.dohop_id)
        if self.partner is None:  # Set default reference
            self.partner = Partner.objects.get(code='dohop')

        super(DohopVendor, self).save(*args, **kwargs)

    @property
    def billing_datasource_id(self):
        return getattr(self, 'billing_datasource_id_%s' % settings.ENVIRONMENT)

    def get_national_logo_svg(self, national_version):
        field_name = 'logo2_svg_{}'.format(
            national_version
        )

        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_national_price(self, national_version):
        cache_field = 'clk_price_obj_%s' % national_version

        if not hasattr(self, cache_field):
            setattr(
                self, cache_field,
                ClickPrice(
                    Partner.objects.get(code='dohop'),
                    national_version
                )
            )

        return getattr(self, cache_field)

    def get_staticpages_partner(self, national_version, lang, t_type_code, services):
        try:
            return self.staticpages_partners.get(
                national_version=national_version,
                t_type__code=t_type_code,
                is_published=True,
                services__in=services
            )

        except (MultipleObjectsReturned, ObjectDoesNotExist):
            return None

    # Заглушка
    def get_national_logo(self, national_version):
        return None

    @classmethod
    def index_by_national_version(cls, national_version):
        return 'yandexcom' if national_version == 'com' else 'yandex'

    @classmethod
    def get_actual(cls, national_version='ru', from_rasp=False, mobile=False):
        vendors = DohopVendor.objects.filter(
            enabled=True,
            index_infos__index__code=cls.index_by_national_version(national_version)
        ).select_related('partner')

        service = 'rasp' if from_rasp else 'ticket'

        return [
            p for p in vendors
            if p.enabled_in(service, national_version, mobile)
        ]

    @staticmethod
    def get_LogoSvgUploadToGetter(national_version):
        return DohopVendorLogoSvgUploadToGetter(national_version)

    @staticmethod
    def get_LogoSvg2PngUploadToGetter(national_version):
        return DohopVendorLogoSvg2PngUploadToGetter(national_version)

    @staticmethod
    def get_IconSvgUploadToGetter():
        return DohopVendorIconSvgUploadToGetter()

    class Meta(BasePartner.Meta):
        app_label = 'order'
        verbose_name = _(u"вендор Dohop")
        verbose_name_plural = _(u"вендоры Dohop")


class DohopIndex(models.Model):
    code = TrimmedCharField(_(u'Код индекса'), max_length=12, null=False, blank=False, unique=True)

    def __unicode__(self):
        return self.code

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Индекс Dohop')
        verbose_name_plural = _(u'Индексы Dohop')


class DohopVendorIndexInfo(models.Model):
    vendor = models.ForeignKey(DohopVendor, related_name='index_infos')
    index = models.ForeignKey(DohopIndex, related_name='vendors_infos')

    vendor_info = models.TextField(null=False, blank=False)
    residences = models.TextField(null=False, blank=False)
    langs = models.CharField(max_length=255, null=False, blank=False)

    def __unicode__(self):
        return '{} {}'.format(self.index, self.vendor.code)

    class Meta:
        unique_together = (('vendor', 'index'),)
        app_label = 'order'
        verbose_name = _(u'Информация по вендору дохоп в индексе')
        verbose_name_plural = _(u'Информация по вендору дохоп в индексах')


class PartnerUser(models.Model):
    ROLES = tuple([(v, v) for v in ['admin', 'manager', 'helper', 'owner', 'assessor', 'yandex-assessor']])

    role = models.CharField(choices=ROLES, max_length=100, blank=False, null=False, default=None)
    login = models.CharField(unique=True, max_length=100, blank=False, null=False, default=None)
    name = models.CharField(blank=False, null=False, default=None, max_length=255)
    partner = models.ForeignKey(Partner, related_name='partner_users', null=True, blank=True)
    passportuid = models.CharField(max_length=100, null=True, blank=True)

    def clean(self):
        if self.role in ['admin', 'manager', 'yandex-assessor'] and self.partner_id:
            raise ValidationError(_(u'у роли {} не может быть partner_id'.format(self.role)))

        if self.role in ['owner', 'helper', 'assessor'] and not self.partner_id:
            raise ValidationError(_(u'у роли {} должен быть partner_id'.format(self.role)))

        if (self.role == 'owner'):

            has_other_owners = (
                self.__class__.objects
                    .filter(role='owner', partner_id=self.partner_id)
                    .exclude(id=self.id)
                    .exists()
            )
            if has_other_owners:
                error = u'пара (role=owner, partner_id={}) должна быть уникальна'.format(self.partner_id)
                raise ValidationError(_(error))

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Пользователь партнерки')
        verbose_name_plural = _(u'Пользователи партнерки')


class PartnerRegionalizationFile(models.Model):
    url = models.CharField(max_length=255, blank=False, null=False)
    partner = models.ForeignKey(Partner, related_name='regionalization_files', null=False, blank=False)
    national_version = models.CharField(
        _(u'Нац. версия'),
        choices=settings.AVIA_NATIONAL_VERSIONS_CHOICES,
        blank=False,
        default='ru',
        null=False,
        max_length=settings.AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN
    )

    class Meta:
        unique_together = (('partner', 'national_version'),)
        app_label = 'order'
        verbose_name = _(u'Файл регионализации')
        verbose_name_plural = _(u'Файлы регионализации')


class PriceList(models.Model):
    settlement_from = models.ForeignKey('www.Settlement', related_name='+', verbose_name=_(u'город откуда'))
    settlement_to = models.ForeignKey('www.Settlement', related_name='+', verbose_name=_(u'город куда'))
    month = models.IntegerField(verbose_name=_(u'месяц вылета'), null=False, blank=False)
    is_one_way = models.BooleanField(verbose_name=_(u'только туда'), null=False, blank=False)
    price = models.IntegerField(verbose_name=_(u'цена в фишкоцентах'), null=False, blank=False)
    national_version = models.ForeignKey('avia_data.NationalVersion', related_name='+', verbose_name=_(u'Нац. версия'))

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Цена клика по направлению')
        verbose_name_plural = _(u'Цена клика по направлениям')
        unique_together = ('settlement_from', 'settlement_to', 'month', 'is_one_way', 'national_version')


class DefaultClickPrice(models.Model):
    billing_price = models.IntegerField(verbose_name=_(u'цена в фишкоцентах'), null=False, blank=False)
    national_version = models.ForeignKey('avia_data.NationalVersion', related_name='+', verbose_name=_(u'Нац. версия'))

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Дефолтная цена клика')
        verbose_name_plural = _(u'Дефолтные цены клика')


class CPCPrice(models.Model):
    partner = models.ForeignKey(
        Partner,
        related_name='+',
        verbose_name=_(u'Партнер'),
        limit_choices_to={
            't_type': TransportType.PLANE_ID,
        },
    )
    national_version = models.ForeignKey('avia_data.NationalVersion', related_name='+', verbose_name=_(u'Нац. версия'))
    n_redirects = models.IntegerField(_(u'Число редиректов всего'), null=False, blank=False)
    cpa_sum = models.DecimalField(_(u'Сумма по CPA всего'), max_digits=18, decimal_places=2)
    eCPC = models.IntegerField(_(u'Расчитанное CPC (в копейках) (остальные)'), null=False, blank=False)
    eCPC_rasp_direct = models.IntegerField(
        _(u'Расчитанное CPC (в копейках) (расписания прямые)'),
        null=False,
        blank=False,
        default=0,
    )
    eCPC_sovetnik_direct = models.IntegerField(
        _(u'Расчитанное CPC (в копейках) (советник прямые)'),
        null=False,
        blank=False,
        default=0,
    )
    eCPC_wizard_direct = models.IntegerField(
        _(u'Расчитанное CPC (в копейках) (колдун прямые)'),
        null=False,
        blank=False,
        default=0,
    )
    eCPC_wizard_indirect = models.IntegerField(
        _(u'Расчитанное CPC (в копейках) (колдун непрямые)'),
        null=False,
        blank=False,
        default=0,
    )

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Цена клика для СPA партнера')
        verbose_name_plural = _(u'Цены кликов для CPA партнеров')
        unique_together = ('partner', 'national_version')


class RedirectType(models.Model):
    code = models.CharField(verbose_name=_(u'Код'), null=False, blank=False, max_length=20, unique=True)
    name = models.CharField(verbose_name=_(u'Описание'), null=False, blank=False, max_length=100)

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Тип редиректа для расчета цен')
        verbose_name_plural = _(u'Типы редиректов для расчета цен')

    def __unicode__(self):
        return u'[{}] {}'.format(self.code, self.name)


class ConversionByRedirectType(models.Model):
    national_version = models.ForeignKey('avia_data.NationalVersion', related_name='+', verbose_name=_(u'Нац. версия'))
    redirect_type = models.ForeignKey(RedirectType, related_name='+', verbose_name=_(u'Тип редиректа'))
    conversion = models.DecimalField(max_digits=4, decimal_places=4, verbose_name=_(u'Конверсия (в долях)'))
    updated_at = models.DateTimeField(verbose_name=_(u'Время обновления'))

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Конверсия по типу редиректа')
        verbose_name_plural = _(u'Конверсии по типам редиректов')
        unique_together = ('national_version', 'redirect_type')


class ClickPriceMultiplierByRedirectType(models.Model):
    national_version = models.ForeignKey('avia_data.NationalVersion', related_name='+', verbose_name=_(u'Нац. версия'))
    redirect_type = models.ForeignKey(RedirectType, related_name='+', verbose_name=_(u'Тип редиректа'))
    multiplier = models.DecimalField(max_digits=5, decimal_places=3, verbose_name=_(u'Корректировка стоимости клика'))
    updated_at = models.DateTimeField(verbose_name=_(u'Время обновления'))

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Корректировка стоимости клика по типу редиректа в зависимости от среднего чека')
        verbose_name_plural = _(u'Корректировки стоимости клика по типам редиректов в зависимости от среднего чека')
        unique_together = ('national_version', 'redirect_type')


class PartnerVariantsTTL(BaseVariantsTTL):
    class Meta(BaseVariantsTTL.Meta):
        app_label = 'order'

    partner = models.ForeignKey('Partner', verbose_name=_(u'Партнер'), related_name='variants_ttl')


class AmadeusMerchantVariantsTTL(BaseVariantsTTL):
    class Meta(BaseVariantsTTL.Meta):
        app_label = 'order'
        db_table = 'order_amadeusmerchant_variantsttl'

    partner = models.ForeignKey('avia_data.AmadeusMerchant', verbose_name=_(u'Партнер'), related_name='variants_ttl')


class DohopVariantsTTL(BaseVariantsTTL):
    class Meta(BaseVariantsTTL.Meta):
        app_label = 'order'

    partner = models.ForeignKey('DohopVendor', verbose_name=_(u'Партнер'), related_name='variants_ttl')
