# -*- coding: utf-8 -*-
import json
import logging
import warnings
from datetime import datetime, timedelta, time

import pytz
from django.conf import settings
from django.core.validators import MinLengthValidator
from django.db import models
from django.utils import translation
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _
from django.utils.text import slugify
from pytils.dt import ru_strftime

from travel.avia.library.python.avia_data.storage import AvatarsPngStorage, AvatarsSvgStorage
from travel.avia.library.python.common.models.geo import DirectionFromTranslate
from travel.avia.library.python.common.models.translations import TranslatedTitle, TranslatedText, TranslationMixin
from travel.avia.library.python.common.models_abstract.schedule import BaseRThread, BaseSupplier, BaseRoute
from travel.avia.library.python.common.models_utils import HiddenManagerWrapper
from travel.avia.library.python.common.models_utils.i18n import L_field, new_L_field
from travel.avia.library.python.common.precache.manager import PrecachingManager
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.caching import cache_until_switch
from travel.avia.library.python.common.utils.date import RunMask, NBSP, DateTimeFormatter
from travel.avia.library.python.common.utils.date import smart_localize
from travel.avia.library.python.common.utils.fields import CodeCharField, TrimmedCharField
from travel.avia.library.python.common.utils.media_fields import CustomImageField, SvgImageField
from travel.avia.library.python.common.utils.models import SwapModel
from travel.avia.library.python.common.utils.text import transliterate
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.library.python.common.xgettext.i18n import gettext, tgettext


log = logging.getLogger(__name__)


class Supplier(BaseSupplier):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.schedule'

    class Meta(BaseSupplier.Meta):
        app_label = 'www'


def get_svg2png_logo_upload_to(instance, filename):
    return 'data/company/svg2png/id%s.png' % instance.id


def make_slug(airline):
    if airline.hidden:
        return

    l_title = airline.new_L_title

    en_nominative = l_title.en_nominative
    if not en_nominative and l_title.ru_nominative:
        en_nominative = transliterate(l_title.ru_nominative, 'cyr-lat')

    if not (airline.iata and en_nominative):
        return

    key = u'{}_{}'.format(airline.iata, en_nominative)

    return slugify(key)


class Company(TranslationMixin, models.Model):
    """ Компания-перевозчик """
    AEROFLOT_ID = 26
    POBEDA_ID = 9144

    aeroexpress_id = 162

    address = models.CharField(verbose_name=_(u'адрес'), max_length=255, null=True,
                               blank=True)
    home_station = models.ForeignKey('www.Station', verbose_name=_(u'главная станция'),
                                     null=True, blank=True)

    supplier_code = CodeCharField(verbose_name=_(u'код Поставщика Данных'), max_length=255,
                                  null=True, blank=True, default=None, unique=True)
    sirena_id = CodeCharField(verbose_name=_(u'код Сирена'), max_length=100,
                              null=True, blank=True, default=None, unique=True)
    iata = CodeCharField(verbose_name=_(u'код IATA'), max_length=100,
                         null=True, blank=True, default=None, db_index=True)
    icao = CodeCharField(verbose_name=_(u'код ICAO'), max_length=100, null=True,
                         blank=True, unique=True)
    icao_ru = CodeCharField(verbose_name=_(u'код ICAO(рус)'), max_length=100, null=True,
                            blank=True, unique=True)
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'тип транспорта'),
                               null=True, blank=True, default=None)
    is_freight = models.BooleanField(_(u"Грузовая компания"), default=False,
                                     help_text=_(u"Не импортируем такие компании от OAG"))

    priority = models.IntegerField(
        null=False, blank=True, default=0,
        help_text=_(u'Приоритет для различения компаний с одним иата-кодом')
    )

    L_title = new_L_field(_(u'название'), add_local_field=True)

    new_L_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    L_short_title = new_L_field(_(u'краткое название'), add_local_field=True)

    new_L_short_title = models.ForeignKey(
        TranslatedTitle, related_name='+'
    )

    url = models.URLField(verbose_name=_(u'урл сайта'), max_length=255, null=True,
                          blank=True)

    L_registration_url = new_L_field(_(u'Url регистрации'), add_local_field=True, default='', blank=True, null=True, max_length=255)

    new_L_registration_url = models.ForeignKey(
        TranslatedText, related_name='+'
    )

    L_registration_phone = new_L_field(_(u'Телефон регистрации'), add_local_field=True, default='', blank=True, null=True, max_length=255)

    new_L_registration_phone = models.ForeignKey(
        TranslatedText, related_name='+'
    )

    email = models.EmailField(verbose_name=_(u'email'), max_length=255, null=True, blank=True)
    contact_info = models.TextField(verbose_name=_(u'контактная информация'),
                                    null=True, blank=True, default=None)
    phone = TrimmedCharField(verbose_name=_(u'контактный телефон'), max_length=255, null=False,
                             default='', blank=True)
    phone_booking = TrimmedCharField(verbose_name=_(u'телефон для бронирования'), max_length=255, null=False,
                                     default='', blank=True)
    description = models.TextField(verbose_name=_(u'описание'),
                                   null=True, blank=True, default=None)

    logo = models.ImageField(verbose_name=_(u'логотип'),
                             upload_to='data/company/logo', null=True, blank=True,
                             default=None)
    icon = models.ImageField(verbose_name=_(u'иконка'), upload_to='data/company/icon',
                             null=True, blank=True, default=None)

    logo_mono = models.ImageField(verbose_name=_(u'логотип без фона'), upload_to='data/company/logo_mono',
                                  null=True, blank=True, default=None)

    logo_bgcolor = TrimmedCharField(verbose_name=_(u'цвет фона логотипа'), max_length=7, null=False,
                                    help_text=_(u'В формате #FF0044'),
                                    default='', blank=True)

    svg_logo2 = SvgImageField(
        verbose_name=_(u'SVG логотип в Аватарнице'),
        storage=AvatarsSvgStorage(),
        null=True, blank=True, default=None,
    )

    svg2png_logo2 = CustomImageField(
        verbose_name=_(u'PNG логотип в Аватарнице'),
        storage=AvatarsPngStorage(),
        null=True, blank=True, default=None,
    )

    country = models.ForeignKey('Country', verbose_name=_(u'страна'),
                                null=True, blank=True, default=None)
    hidden = models.BooleanField(verbose_name=_(u'не показывать нигде!'), default=False)
    strange = models.BooleanField(verbose_name=_(u'не понятные перевозчики'),
                                  default=False)

    meta_title = models.CharField(max_length=255, verbose_name=_(u'meta title'), null=True, blank=True)
    meta_description = models.TextField(verbose_name=_(u'meta description'), null=True, blank=True)

    alliance = models.ForeignKey('AviaAlliance', verbose_name=_(u'Авиа-альянс'), null=True,
                                 blank=True, related_name='companies')

    L_bonus_name = L_field(_(u'название программы для часто летающих пассажиров'),
                           default=u'', max_length=255, blank=True, add_local_field=True)

    slug = CodeCharField(verbose_name=_(u'идентификатор для ЧПУ'), max_length=255,
                         null=True, blank=True, default=None, unique=True)

    popular_score = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании'), null=False, default=0
    )
    popular_score_for_ru = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании для ru версии'), null=False, default=0
    )
    popular_score_for_ua = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании для ua версии'), null=False, default=0
    )
    popular_score_for_tr = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании для tr версии'), null=False, default=0
    )
    popular_score_for_kz = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании для kz версии'), null=False, default=0
    )
    popular_score_for_com = models.PositiveIntegerField(
        verbose_name=_(u'Очки популярности компании для com версии'), null=False, default=0
    )
    seo_description_i18n = models.CharField(
        max_length=255, verbose_name=_(u'Ссылка на ключ в танкере'), null=True, blank=True
    )
    seo_description_approved = models.BooleanField(
        verbose_name=_(u'Показать текст на морде'), default=False
    )
    objects = PrecachingManager(
        iexact_keys=(
            'sirena_id',
            'iata',
            'icao',
            'icao_ru',
        ),
        select_related=(
            'new_L_title',
            'new_L_short_title',
            'new_L_registration_url',
            'new_L_registration_phone',
        ),
    )

    hidden_manager = HiddenManagerWrapper('objects')

    type = "company"

    def __unicode__(self):
        return u'<Company: {} {}>'.format(self.id, self.title)

    class Meta:
        verbose_name = _(u'компания-перевозчик')
        verbose_name_plural = _(u'компании-перевозчики')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_company'


class CompanySynonym(models.Model):
    language = models.CharField(
        choices=settings.LANGUAGES,
        max_length=2,
        default='ru',
        verbose_name=_(u'язык')
    )

    synonym = models.CharField(max_length=255, verbose_name=_(u'синоним'))

    company = models.ForeignKey(Company, verbose_name=_(u'компания'))

    class Meta:
        verbose_name = _(u'синоним компании-перевозчика')
        verbose_name_plural = _(u'синонимы компаний-перевозчиков')
        app_label = 'www'
        db_table = 'www_companysynonym'


class Route(BaseRoute):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.schedule'

    class Meta(BaseRoute.Meta):
        app_label = 'www'


class RThreadType(models.Model):
    """ Тип нитки: ОСН, ТРАФ, .. """

    BASIC_ID = 1
    THROUGH_TRAIN_ID = 3
    CHANGE_ID = 9
    CANCEL_ID = 10
    ASSIGNMENT_ID = 11
    INTERVAL_ID = 12
    PSEUDO_BACK_ID = 13

    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    code = models.CharField(verbose_name=_(u'код'), max_length=100, unique=True)

    objects = PrecachingManager(keys=['pk', 'code'])

    def __unicode__(self):
        return self.title

    @classmethod
    @cache_until_switch
    def get_by_code(cls, code):
        return RThreadType.objects.get(code=code)

    class Meta:
        verbose_name = _(u'тип нитки')
        verbose_name_plural = _(u'типы ниток')
        ordering = ('id',)
        app_label = 'www'
        db_table = 'www_rthreadtype'


class RThread(BaseRThread):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.schedule'

    class Meta(BaseRThread.Meta):
        app_label = 'www'


class RTStation(models.Model):
    """ Станция нитки """

    tz_arrival = models.IntegerField(null=True, blank=True,
                                     help_text=_(u"расст. от нач. маршр. до приб. (в минутах). Прибытие на первую станцию заполнять нельзя(оставить пустым)"))
    tz_departure = models.IntegerField(null=True, blank=True,
                                       help_text=_(u"расст. от нач. маршр. до отпр. (в минутах). Отправление с последней станции заполнять нельзя(оставить пустым)"))
    time_zone = models.CharField(verbose_name=_(u'временная зона'), max_length=30)

    thread = models.ForeignKey('RThread', verbose_name=_(u"нитка"))
    station = models.ForeignKey('Station', verbose_name=_(u"станция"))
    arrival_code_sharing = models.BooleanField(verbose_name=_(u"прибытие code sharing"),
                                               null=False, blank=True, default=False)
    departure_code_sharing = models.BooleanField(verbose_name=_(u"отправление code sharing"),
                                                 null=False, blank=True, default=False)
    arrival_direction = models.ForeignKey("Direction", blank=True, null=True,
                                          verbose_name=_(u"arr_direction"),
                                          help_text=_(u"направление прибытия"),
                                          related_name='arrival_rtstation_set')
    departure_direction = models.ForeignKey("Direction", blank=True, null=True,
                                            verbose_name=_(u"dep_direction"),
                                            help_text=_(u"направление отправления"),
                                            related_name='departure_rtstation_set')
    arrival_subdir = models.CharField(verbose_name=_(u'arr_subdir'), max_length=100,
                                      null=True, blank=True)
    L_arrival_subdir = DirectionFromTranslate.get_L_method(key_field='arrival_subdir')
    departure_subdir = models.CharField(verbose_name=_(u'dep_subdir'), max_length=100,
                                        null=True, blank=True)
    L_departure_subdir = DirectionFromTranslate.get_L_method(key_field='departure_subdir')
    is_from_subdir = models.BooleanField(verbose_name=_(u"поднаправление \"туда\""),
                                         help_text=_(u"от Москвы"),
                                         null=False, blank=True, default=True)
    is_technical_stop = models.BooleanField(verbose_name=_(u"ТО"),
                                            help_text=_(u"техническая остановка"),
                                            null=False, blank=True, default=False)
    platform = TrimmedCharField(verbose_name=_(u'платформа'),
                                max_length=20, null=True, default=None, blank=True)

    terminal = models.ForeignKey('StationTerminal', verbose_name=_(u'терминал'),
                                 null=True, blank=True, default=None)

    arrival_t_model = models.ForeignKey('TransportModel', verbose_name=_(u"модель прибытия"),
                                        null=True, default=None, blank=True,
                                        related_name='arrival_rtstation_set')
    departure_t_model = models.ForeignKey('TransportModel', verbose_name=_(u"модель отправления"),
                                          null=True, default=None, blank=True,
                                          related_name='departure_rtstation_set')
    is_virtual_end = models.BooleanField(verbose_name=_(u"является виртуальной конечной остановкой"),
                                         null=False, blank=True, default=False)

    # fuzzy flags group
    is_fuzzy = models.BooleanField(_(u"нечеткое время отправления/прибытия"), default=False)
    is_searchable_to = models.BooleanField(_(u"искать до станции"), default=True)
    is_searchable_from = models.BooleanField(_(u"искать от станции"), default=True)
    in_station_schedule = models.BooleanField(_(u"в расписании по станции"), default=True)
    in_thread = models.BooleanField(_(u'показывать на странице нитки'), default=True)

    # next_station и prev_station заполнены только у самолетов. см. update_plane_rtstations.py
    next_station = models.ForeignKey('Station', verbose_name=_(u"следующая станция"), null=True, related_name="rtstation_next_station_set")
    prev_station = models.ForeignKey('Station', verbose_name=_(u"предыдущая станция"), null=True, related_name="rtstation_prev_station_set")

    is_combined = models.BooleanField(_(u"Станция согласованной пересадки"), default=False)

    def L_platform(self, lang=None):
        return PlatformTranslation.get_translation(self.platform, lang)

    class Meta:
        verbose_name = _(u'станция нитки')
        verbose_name_plural = _(u'станции ниток')
        ordering = ('id',)
        app_label = 'www'
        db_table = 'www_rtstation'

    @cached_property
    def pytz(self):
        return pytz.timezone(self.time_zone)

    def get_arrival_mask_shift(self, thread_start_time):
        if self.tz_arrival is None:
            return None

        return (self.tz_arrival + thread_start_time.hour * 60 + thread_start_time.minute) / 1440

    def get_departure_mask_shift(self, thread_start_time):
        if self.tz_departure is None:
            return None

        return (self.tz_departure + thread_start_time.hour * 60 + thread_start_time.minute) / 1440

    def get_arrival_dt(self, naive_start_dt, out_tz=None):
        """
        @return: Aware Arrival datetime
        """
        if self.tz_arrival is None:
            return

        arrival_dt = smart_localize(naive_start_dt + timedelta(minutes=self.tz_arrival), self.pytz)

        if out_tz:
            return arrival_dt.astimezone(out_tz)
        else:
            return arrival_dt

    def get_arrival_loc_dt(self, naive_start_dt):
        return self.get_arrival_dt(naive_start_dt, self.station.pytz)

    def get_loc_arrival_dt(self, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_arrival_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_arrival_loc_dt(naive_start_dt)

    def get_departure_dt(self, naive_start_dt, out_tz=None):
        """
        @return: Aware departure datetime
        """
        if self.tz_departure is None:
            return

        departure_dt = smart_localize(naive_start_dt + timedelta(minutes=self.tz_departure), self.pytz)

        if out_tz:
            return departure_dt.astimezone(out_tz)
        else:
            return departure_dt

    def get_departure_loc_dt(self, naive_start_dt):
        return self.get_departure_dt(naive_start_dt, self.station.pytz)

    def get_loc_departure_dt(self, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_departure_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_departure_loc_dt(naive_start_dt)

    def get_event_dt(self, event, naive_start_dt, out_tz=None):
        return getattr(self, 'get_{}_dt'.format(event))(naive_start_dt, out_tz)

    def get_event_loc_dt(self, event, naive_start_dt):
        return self.get_event_dt(event, naive_start_dt, self.station.pytz)

    def get_loc_event_dt(self, event, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_event_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_event_loc_dt(event, naive_start_dt)

    def calc_days_shift(self, event='departure', start_date=None, event_tz=None):
        """
        Вычисляет сдвиг в днях
            от старта нитки
                во временной зоне старта нитки
            до отправления со станции (прибытия на станцию) self
                во временной зоне event_tz (по-умолчанию во временной зоне self.station)
        если
            известен день старта нитки start_date во временной зоне нитки
            по-умолчанию берем дату во временной зоне нитки на сейчас
        """

        return self.calc_days_timedelta(event, start_date, event_tz).days

    def calc_days_timedelta(self, event='departure', start_date=None, event_tz=None):
        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if start_date is None:
            start_date = environment.now_aware().astimezone(self.thread.pytz).date()

        naive_start_dt = datetime.combine(start_date, self.thread.tz_start_time)

        event_dt = self.get_event_dt(event, naive_start_dt, out_tz=event_tz)

        return event_dt.date() - naive_start_dt.date()

    def calc_days_shift_by_event_date(self, event='departure', event_date=None, event_tz=None):
        """
        Вычисляет сдвиг в днях
            от старта нитки
                во временной зоне старта нитки
            до отправления со станции (прибытия на станцию) self
                во временной зоне event_tz (по-умолчанию во временной зоне self.station)
        если
            известена дата отправления(прибытия)
        """

        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if event_date is None:
            event_date = environment.now_aware().astimezone(event_tz).date()

        start_date = self.calc_thread_start_date(event, event_date, event_tz)

        return self.calc_days_shift(event, start_date, event_tz)

    def calc_thread_start_date(self, event='departure', event_date=None, event_tz=None):
        """
        Вычисляем время старта нитки,
        так, чтобы дата отправления(прибытия) совпала с нашей
        """

        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if event_date is None:
            event_date = environment.now_aware().astimezone(event_tz).date()

        try_days_timedelta = self.calc_days_timedelta(event, start_date=event_date, event_tz=event_tz)

        try_naive_start_dt = datetime.combine(event_date, self.thread.tz_start_time)

        try_start_date = try_naive_start_dt.date() - try_days_timedelta

        # Сдвиг даты старта нитки по thread.pytz от даты отправления со станции по from_date_tz,
        # при переходе на летнее(зимнее) время может отличатся на ± 1 день

        naive_start_dt = datetime.combine(try_start_date, self.thread.tz_start_time)
        try_from_date = self.get_event_dt(event, naive_start_dt, out_tz=event_tz).date()

        if try_from_date == event_date:
            return try_start_date

        elif try_from_date < event_date:
            return try_start_date + timedelta(1)

        else:
            return try_start_date - timedelta(1)


class PlatformTranslation(models.Model):
    TANKER_L_FIELDS = ('platform',)
    TANKER_KEY_FIELD = 'platform'
    TANKER_HELP_FIELDS = ('description',)

    description = models.CharField(_(u'Где встречается'), max_length=255, null=True, default=None,
                                   blank=True)

    L_platform = new_L_field(
        _(u'платформа'),
        add_local_field=True,
        max_length=30,
        null=True,
        default=None,
        blank=True,
    )

    new_L_platform = models.ForeignKey(TranslatedTitle, related_name='+')

    objects = PrecachingManager(
        keys=(
            'platform',
        ),
        select_related=(
            'new_L_platform',
        ),
    )

    def __unicode__(self):
        return u'Переводы платформы "{}"'.format(self.platform)

    @classmethod
    def get_translation(cls, platform, lang=None):
        platform = (platform or u'').strip()

        if not platform:
            return platform

        try:
            pt = cls.objects.filter(platform=platform)[0]

            return pt.L_platform(lang=lang)

        except IndexError:
            return platform

    class Meta:
        verbose_name = _(u'Перевод платформы')
        verbose_name_plural = _(u'Переводы платформ')
        app_label = 'www'
        db_table = 'www_platformtranslation'


class DeLuxeTrain(models.Model):
    numbers = TrimmedCharField(verbose_name=_(u'номера'), max_length=50)
    L_title = new_L_field(
        _(u'название'),
        add_local_field=True, base_lang='ru',
        field_cls=TrimmedCharField, max_length=100, null=False, blank=True,
        db_index=True
    )

    new_L_title = models.ForeignKey(TranslatedTitle, related_name='+')

    deluxe = models.BooleanField(_(u"фирменный"), default=False)
    high_speed = models.BooleanField(_(u"скоростной"), default=False)

    TANKER_L_FIELDS = ('title',)

    _number2deluxe_train = {}

    objects = PrecachingManager(
        select_related=(
            'new_L_title',
        ),
    )

    @classmethod
    def get_by_number(cls, number):
        """
        Возвращает фирменный поезд по его номеру
        """

        def init():
            for train in DeLuxeTrain.objects.all():
                numbers = train.numbers.split('/')

                for n in numbers:
                    cls._number2deluxe_train[n] = train

        if not cls._number2deluxe_train:
            init()

        return cls._number2deluxe_train.get(number)

    def L_title_short(self, lang=None):
        title = self.L_title(lang=lang)

        if title and self.deluxe:
            # Фраза "фирменный поезд «Невский Экспресс»". Сокращение без слова 'поезд'
            return tgettext(u'фирменный «<special-title/>»', special_title=title)

        elif self.deluxe:
            # Фраза "фирменный поезд". Сокращение без слова 'поезд'
            return gettext(u'фирменный')

        elif title:
            return u'«%s»' % title

        return u''

    class Meta:
        verbose_name = _(u'фирменный или именной поезд')
        verbose_name_plural = _(u'фирменные и именные поезда')
        app_label = 'www'
        db_table = 'www_deluxetrain'


class TrainSchedulePlan(models.Model):
    """
    График поездов и электричек
    """

    title = TrimmedCharField(verbose_name=_(u"название граффика"),
                             default=None, blank=True, null=True,
                             max_length=255)
    start_date = models.DateField(verbose_name=_(u"дата начала(включительно)"),
                                  null=False, blank=False)
    end_date = models.DateField(verbose_name=_(u"дата окончания(включительно)"),
                                null=False, blank=False)
    code = CodeCharField(verbose_name=_(u"код"), max_length=50,
                         null=False, blank=False, unique=True)

    APPENDIX_FROM = 'from'
    APPENDIX_TO = 'to'
    APPENDIX_CHOICES = (
        (APPENDIX_FROM, _(u'Использовать "c"')),
        (APPENDIX_TO, _(u'Использовать "по"')),
    )

    appendix_type = models.CharField(_(u'селектор с/по'), max_length=4,
                                     choices=APPENDIX_CHOICES, default=APPENDIX_FROM,
                                     null=False, blank=False)

    @classmethod
    def fill_thread_plans(cls, threads):
        plans_ids = set(t.schedule_plan_id for t in threads)

        plans_by_id = TrainSchedulePlan.objects.in_bulk(list(plans_ids))

        for t in threads:
            t.schedule_plan = plans_by_id.get(t.schedule_plan_id)

        return plans_by_id.values()

    @classmethod
    def add_to_threads(cls, threads, today):
        """
        @Deprecated
        На каждый thread устанавливается соответствующий schedule_plan.
        Возвращается текущий и следующий планы (они одинаковы для всех ниток).
        """
        plans = cls.fill_thread_plans(threads)

        if plans:
            plans_qs = cls.objects.filter(id__in=[p.id for p in plans])
            current_plan, next_plan = cls.get_current_and_next(today, qs=plans_qs)
            return current_plan, next_plan
        else:
            return None, None

    @classmethod
    def get_current_and_next(cls, today, qs=None):
        """
        Отображаемые графики зависят от сегоднящнего дня.
        Не от дня отображаемого расписания!
        """
        if qs is None:
            qs = cls.objects

        current = next_plan = None

        # Расчитываем на то, что планы не пересекаются и идут по порядку
        plans = list(qs.filter(end_date__gte=today).order_by('start_date')[:2])

        if not plans:
            return current, next_plan

        if plans[0].start_date <= today:
            current = plans.pop(0)
            current.name = None

        if plans:
            next_plan = plans[0]
            start_date_str = DateTimeFormatter(next_plan.start_date, lang='ru').L().replace(' ', NBSP)
            next_plan.name = u'с&nbsp;{}'.format(start_date_str)

        if current and (next_plan or (current.end_date - today).days < 7):
            end_date_str = DateTimeFormatter(current.end_date, lang='ru').L().replace(' ', NBSP)
            current.name = u'по&nbsp;{}'.format(end_date_str)

        return current, next_plan

    def is_current(self, today):
        return self.start_date <= today <= self.end_date

    class Meta:
        verbose_name = _(u"график поездов и электричек")
        verbose_name_plural = _(u"графики поездов и электричек")
        app_label = 'www'
        db_table = 'www_trainscheduleplan'

    def get_appendix(self, thread_start_date, next_plan):
        """
        @Deprecated
        Первоначальный таск - RASP-6846 ?
        """

        warnings.warn("Use TrainSchedulePlan.get_L_appendix() or L_appendix()",
                      RaspDeprecationWarning, stacklevel=2)

        if self == next_plan:
            return ru_strftime(u"c %d %B", self.start_date, inflected=True)

        elif next_plan or self.end_date - thread_start_date < timedelta(days=20):
            return ru_strftime(u"по %d %B", self.end_date, inflected=True)

    def L_appendix(self):
        if self.appendix_type == self.APPENDIX_FROM:
            return gettext(u'с {human_date}').format(human_date=DateTimeFormatter(self.start_date).L())

        else:
            return gettext(u'по {human_date}').format(human_date=DateTimeFormatter(self.end_date).L())

    def get_L_appendix(self, thread_start_date, next_plan):
        """
        TODO: разобраться и сделать правильную логику работы
        1. Логика работы - взята старая, чтобы ничего не поломать
        2. Вообще странно, что передается thread_start_date
        3. Сейчас по факту в thread_start_date передается first_run или локализованый today
        """
        if self == next_plan:
            return self.L_appendix()

        elif next_plan or thread_start_date + timedelta(days=20) > self.end_date:
            return self.L_appendix()

    @classmethod
    def get_schedule_end_period(cls, today=None):
        today = today or environment.today()

        end_dates = [
            tsp.end_date for tsp in
            cls.objects.filter(end_date__gte=today, end_date__isnull=False)
        ]

        if end_dates:
            return min(end_dates)


class ScheduleExclusion(models.Model):
    """
    Исключения из табло и расписаний
    """
    number = TrimmedCharField(verbose_name=_(u"Номер рейса"), max_length=255, null=True, blank=True,
                              help_text=_(u"Если номера нет, берутся все рейсы между станциями"))
    start_station = models.ForeignKey('www.Station', verbose_name=_(u"Начальная станция маршрута"),
                                      null=False, blank=False, related_name='scheduleexclusion_start_set')
    end_station = models.ForeignKey('www.Station', verbose_name=_(u"Конечная станция маршрута"),
                                    null=False, blank=False, related_name='scheduleexclusion_end_set')
    exclude_station = models.ForeignKey('www.Station', verbose_name=_(u"Исключаемая стания"),
                                        null=False, blank=False)

    departure = models.TimeField(_(u"Оптправление с начальной станции"), null=True, blank=True)
    departure_interval = models.TimeField(_(u"Конец интервала отправления"), null=True, blank=True,
                                          default=None)

    def get_thread_station_pairs(self):
        from django.db import connection

        cursor = connection.cursor()
        cursor.execute("""
        SELECT DISTINCT t.id FROM www_rthread t
            JOIN www_rtstation rts_start ON rts_start.thread_id = t.id AND rts_start.tz_arrival IS NULL
            JOIN www_rtstation rts_end ON rts_end.thread_id = t.id AND rts_end.tz_departure IS NULL
        WHERE rts_start.station_id = %s AND rts_end.station_id = %s
        """, [self.start_station.id, self.end_station.id])

        ids = [row[0] for row in cursor.fetchall()]

        threads = RThread.objects.filter(pk__in=ids)

        if self.number:
            threads = threads.filter(number=self.number)

        if self.departure is not None and self.departure_interval is not None:
            threads = threads.filter(tz_start_time__range=(self.departure, self.departure_interval))
        elif self.departure is not None:
            threads = threads.filter(tz_start_time=self.departure)

        pairs = []

        for t in threads:
            pairs.append((t, self.exclude_station))

        return pairs

    @classmethod
    def get_all_exclusion_pairs(cls):
        pairs = []
        for exclusion in cls.objects.all():
            pairs += exclusion.get_thread_station_pairs()

        return pairs

    @classmethod
    def get_excluded_thread_for_station(cls, station):
        threads = []

        for exclusion in cls.objects.filter(exclude_station=station):
            for pair in exclusion.get_thread_station_pairs():
                threads.append(pair[0])

        return threads

    class Meta:
        verbose_name = _(u'Исключение из расписания по станции')
        verbose_name_plural = _(u'Исключения из расписания по станции')
        app_label = 'www'
        db_table = 'importinfo_scheduleexclusion'


class VehicleProducer(models.Model):
    """ Производитель ТС: Як, Ту, .. """

    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    t_type = models.ForeignKey('TransportType', verbose_name=_(u'тип транспорта'))

    def __unicode__(self):
        return self.title

    def name(self):
        return self.__unicode__()

    type = "vehicleproducer"

    class Meta:
        verbose_name = _(u'производитель ТС')
        verbose_name_plural = _(u'производители ТС')
        ordering = ('title',)
        app_label = 'www'
        db_table = 'www_vehicleproducer'


class ExpressTypeLite(models.Model):
    title = TrimmedCharField(_(u'Название'), max_length=100, null=False, blank=True, default='')
    code = TrimmedCharField(_(u'Код'), max_length=40, null=False, blank=True, default='')
    color = TrimmedCharField(_(u'Цвет'), max_length=10, null=False, blank=True, default='')

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'подтип экспрессов')
        verbose_name_plural = _(u'подтипы экспрессов')
        app_label = 'www'
        db_table = 'www_expresstypelite'


class StationSchedule(models.Model):
    station = models.ForeignKey('Station')
    route = models.ForeignKey('Route')
    thread = models.ForeignKey('RThread', null=True)
    rtstation = models.OneToOneField(RTStation, related_name='schedule')

    stops = models.TextField(null=True)
    stops_translations = models.TextField(null=True)

    is_fuzzy = models.BooleanField(default=False)

    def L_stops(self):
        if not getattr(self, 'stops_translations', None):
            return ''

        lang = translation.get_language()

        stops = json.loads(self.stops_translations)

        if lang in stops:
            return stops[lang]

        return ''

    @classmethod
    def get(cls, station, thread, event):
        # Получить расписание треда по станции
        schedule_set = cls.objects.filter(station=station, thread=thread)

        try:
            if event == 'departure':
                return schedule_set.order_by('rtstation__id')[0]

            return schedule_set.order_by('-rtstation__id')[0]

        except IndexError:
            raise cls.DoesNotExist("No StationSchedule matching query exists")

    def times(self, event):
        if not getattr(self, "%s_times" % event):
            # не отправляется или не прибывает на эту станцию
            return []

        return sorted(time(int(t) / 60, int(t) % 60) for t in getattr(self, "%s_times" % event).split(","))

    def mask(self, event, today=None):
        return RunMask(getattr(self, "%s_mask" % event), today=today)

    def __unicode__(self):
        return u'id=%s' % self.id

    class Meta:
        app_label = 'www'


class AviaAlliance(models.Model):
    L_title = new_L_field(
        _(u'название альянса'), default=u'', max_length=100, blank=True,
        add_local_field=True, local_field_critical=True
    )

    new_L_title = models.ForeignKey(TranslatedTitle, related_name='+')

    logo_svg = models.FileField(verbose_name=u'SVG логотип', null=False, blank=True,
                                upload_to='data/alliance/logo')
    L_description = new_L_field(
        verbose_name=_(u"описание"), add_local_field=True,
        field_cls=models.TextField, blank=True
    )

    new_L_description = models.ForeignKey(TranslatedText, related_name='+')

    enabled = models.BooleanField(_(u'Показывать'), blank=False, null=False, default=False)

    objects = PrecachingManager(
        select_related=(
            'new_L_title',
            'new_L_description',
        ),
    )

    class Meta:
        ordering = ('title',)
        verbose_name = _(u"авиа альянс")
        verbose_name_plural = _(u"авиа альянсы")
        app_label = 'www'

    def __unicode__(self):
        return self.L_title()


class RouteSchedule(models.Model):
    station_from = models.ForeignKey('www.Station', related_name='+')
    station_to = models.ForeignKey('www.Station', related_name='+')
    run_mask = models.CharField(
        max_length=12 * 31,
        validators=[MinLengthValidator(12 * 31)]
    )
    route_number = models.CharField(max_length=16)
    t_model = models.ForeignKey('www.TransportModel', null=True)

    class Meta:
        app_label = 'www'
        db_table = 'www_routeschedule'
        unique_together = (
            'station_from', 'station_to', 'route_number', 't_model'
        )
