# -*- coding: utf-8 -*-

import logging
from datetime import date, datetime

from django.db import models, transaction
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.common.utils.unicode_csv import unicode_reader
from travel.avia.library.python.common.utils.date import fix_date


log = logging.getLogger(__name__)


class CacheSet(models.Model):
    title = models.TextField(_(u'Заголовок'), default='', null=False)

    date1 = models.DateField(_(u'Дата 1'), null=False, default=date.today)
    date2 = models.DateField(_(u'Дата 2'), null=False, default=date.today)
    date3 = models.DateField(_(u'Дата 3'), null=False, default=date.today)

    sort_order = models.IntegerField(_(u'Порядок для сортировки'), default=65535)

    @property
    def dates(self):
        return (self.date1, self.date2, self.date3,)

    def clean(self):
        self.date1, self.date2, self.date3 = sorted([self.date1, self.date2, self.date3])

    def __unicode__(self):
        return u'%s: %s - %s - %s' % tuple([self.title] + map(lambda d: d.strftime('%d.%m'), self.dates))

    class Meta:
        app_label = 'tariffs'
        verbose_name = _(u"блок направлений")
        verbose_name_plural = _(u"блоки направлений")
        ordering = ['sort_order', 'date1', 'date2', 'date3']

    @classmethod
    @transaction.atomic
    def import_from_csv(cls, filename):
        today = date.today()

        def get_point(express_code):
            station = Station.get_by_code('express', express_code)

            if station.settlement:
                return station.settlement

            return station

        reader = unicode_reader(open(filename, 'rb'), encoding='utf-8', delimiter='\t')

        cls.objects.all().delete()

        sort_order_index = 0
        for row in reader:
            if not any(row):
                continue

            try:
                from_name, to_name, from_express_code, to_express_code, alt_link, date1, date2, date3, partner_code, block_title, id = row
            except Exception:
                log.error(u'Не разбирается строка %s', row)
                continue

            date1, date2, date3 = sorted(fix_date(datetime.strptime(d, '%d.%m').date(), today) for d in [date1, date2, date3])

            cacheset, created = cls.objects.get_or_create(pk=id)

            if created:
                sort_order_index += 10
                cacheset.sort_order = sort_order_index

            cacheset.date1 = date1
            cacheset.date2 = date2
            cacheset.date3 = date3
            cacheset.title = block_title

            cacheset.save()

            point_from = get_point(from_express_code)
            point_to = get_point(to_express_code)

            cacheset.directions.create(
                title='%s - %s' % (from_name, to_name),
                point_from_key=point_from.point_key,
                point_to_key=point_to.point_key,
                alt_link=alt_link,
                partner_code=partner_code,
            )


class SpecialOfferDisplayManager(models.Manager):
    use_for_related_fields = True

    def get_queryset(self):
        return super(SpecialOfferDisplayManager, self).get_queryset() \
            .exclude(company__iata=None) \
            .exclude(company__hidden=True) \
            .exclude(company__strange=True) \
            .exclude(settlement_to__country=None)


SERVICE_CLASS_CHOICES = (
    ('e', _(u'Эконом')),
    ('b', _(u'Бизнес')),
    ('f', _(u'Первый')),
)


class SpecialOffer(models.Model):
    u"""Спецпредложения на Авиабилеты"""
    settlement_from = models.ForeignKey('Settlement', verbose_name=_(u'город отправления'), related_name='special_offers_from')
    airport_from = models.ForeignKey('Station', verbose_name=_(u'аэропорт отправления'), related_name='special_offers_from', null=True, blank=True)

    settlement_to = models.ForeignKey('Settlement', verbose_name=_(u'город прибытия'), related_name='special_offers_to')
    airport_to = models.ForeignKey('Station', verbose_name=_(u'аэропорт прибытия'), related_name='special_offers_to', null=True, blank=True)

    company = models.ForeignKey('Company', verbose_name=_(u'авиакомпания'))
    tariff = models.FloatField(_(u'Цена'))
    service_class = models.CharField(verbose_name=_(u'класс обслуживания'), choices=SERVICE_CLASS_CHOICES,
                                     max_length=1, null=False, blank=False, default='e')

    date_start = models.DateField(verbose_name=_(u"дата начала предложения"))
    date_end = models.DateField(verbose_name=_(u"дата окончания предложения"))

    period_start = models.DateField(verbose_name=_(u"дата начала периода путешествия"))
    period_end = models.DateField(verbose_name=_(u"дата окончания периода путешествия"))

    reverse = models.BooleanField(verbose_name=_(u"туда-обратно"), default=False)
    online_booking = models.BooleanField(verbose_name=_(u"онлайн бронирование"), default=False)

    tariff_code = TrimmedCharField(verbose_name=_(u"код тарифа"), max_length=20, default='', null=False, blank=True)

    text = models.TextField(verbose_name=_(u"Описание"), null=False, blank=True, default='')

    objects = models.Manager()
    display_objects = SpecialOfferDisplayManager()

    @property
    def point_to(self):
        return self.settlement_to

    @property
    def point_from(self):
        return self.settlement_from

    def __unicode__(self):
        return u'%s. %s - %s' % (
            self.company.iata,
            self.point_from and self.point_from.L_title() or
            'hidden[%s]' % self.settlement_from.L_title(),
            self.point_to and self.point_to.L_title() or
            'hidden[%s]' % self.settlement_to.L_title(),
        )

    class Meta:
        verbose_name = _(u'Спецпредложение на Авиабилеты')
        verbose_name_plural = _(u'Спецпредложения на Авиабилеты')
        app_label = 'www'
