# -*- coding: utf-8 -*-

from django.conf import settings
from django.core.urlresolvers import reverse
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.environment import get_request


class FrontendManager(models.Manager):
    def get_queryset(self):
        request = get_request()

        national_version = request.NATIONAL_VERSION if request else 'ru'

        return super(FrontendManager, self).get_queryset().filter(national_version=national_version)


class StaticPagesManager(FrontendManager):

    def __init__(self, is_ticket):
        super(StaticPagesManager, self).__init__()

        self.is_ticket = is_ticket

    def get_queryset(self):
        return super(StaticPagesManager, self).get_queryset().filter(is_ticket_page=self.is_ticket)


class StaticPage(models.Model):
    u""" Статические страницы """
    parent = models.ForeignKey('self', verbose_name=_(u"родитель"), null=True, blank=True)
    L_title = L_field(verbose_name=_(u"название"), max_length=200, add_local_field=True)
    slug = models.CharField(verbose_name=_(u"код"), max_length=200, null=True, blank=True)
    VISIBILITY_TYPES = (
        (0, _(u'Скрыта')),
        (1, _(u'Опубликована')),
        (2, _(u'Устарела')),
    )
    is_published = models.IntegerField(choices=VISIBILITY_TYPES,
                                       verbose_name=_(u"опубликовано"), default=0)
    L_content = L_field(verbose_name=_(u"содержание"), add_local_field=True,
                        field_cls=models.TextField, default=None, null=True, blank=True)
    template = models.CharField(verbose_name=_(u"файл шаблона"), max_length=200,
                                help_text=_(u'например "yandex_dengi.html"'),
                                null=True, blank=True)
    order = models.IntegerField(verbose_name=_(u"порядок"), default=0)
    materialized_path = models.CharField(verbose_name=_(u"родители"), max_length=200,
                                         help_text=_(u'materialized path'), default='',
                                         null=True, blank=True)

    PAGE_TYPES = (
        ('page', _(u'Обычная страница')),
        ('template', _(u'Jinja шаблон')),
        ('link', _(u'Ссылка на внешнюю страницу')),
        ('geo', _(u'Страница стран')),
        ('carrier', _(u'Страница перевозчиков по странам')),
        ('transport', _(u'Страница типов ТС')),
    )
    type = models.CharField(choices=PAGE_TYPES, max_length=10, default='page', blank=False, null=False)

    is_ticket_page = models.BooleanField(verbose_name=_(u'страница из КБ?'), default=False)

    L_meta_title = L_field(max_length=255, verbose_name=_(u'title страницы'), add_local_field=True, default=None)
    L_meta_description = L_field(verbose_name=_(u'meta description страницы'), field_cls=models.TextField,
                                 add_local_field=True, default=None, null=True, blank=True)

    national_version = models.CharField(
        choices=[('ru', 'ru'), ('ua', 'ua'), ('tr', 'tr')],
        max_length=25,
        default='ru',
        verbose_name=_(u'Национальная версия')
    )

    admin_manager = models.Manager()
    objects = StaticPagesManager(False)
    ticket_objects = StaticPagesManager(True)

    @classmethod
    def get_page(cls, **selectors):
        return cls.objects.get(**selectors)

    @classmethod
    def get_ticket_page(cls, **selectors):
        return cls.ticket_objects.get(**selectors)

    def __unicode__(self):
        return self.title

    def get_local_language(self):
        lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(self.national_version, 'ru')

        return lang

    def path_title(self):
        return u"&nbsp;" * (len(self.materialized_path or "999") - 3) + self.title
    path_title.short_description = _(u'Заголовок')
    path_title.allow_tags = True

    def add_child(self):
        return _(
            u'<a href="/admin/staticpages/staticpage/add/?parent=%(id)s" '
            u'title="%(mp)s">добавить потомка</a>'
        ) % {
            'id': self.id,
            'mp': self.materialized_path
        }
    add_child.short_description = _(u'Добавить потомка')
    add_child.allow_tags = True

    def set_path(self, count):
        if self.parent:
            self.materialized_path = self.parent.materialized_path + "%03d" % count

        else:
            self.materialized_path = "%03d" % count

    def save(self, *args, **kwargs):
        # Порядок
        if self.order == 0:
            try:
                order = StaticPage.admin_manager.filter(parent__exact=self.parent).order_by('-order')[0].order + 100
            except IndexError:
                order = 100

            self.order = order

        # Пересчитываем materialized_path
        upper_count = StaticPage.admin_manager.filter(parent__exact=self.parent, order__lt=self.order).count()

        self.set_path(upper_count + 1)

        super(StaticPage, self).save(*args, **kwargs)

    def parents(self):
        if not self.parent:
            return ()
        paths = [self.materialized_path[:i] for i in xrange(0, len(self.materialized_path), 3)]
        return StaticPage.objects.filter(materialized_path__in=paths).order_by('materialized_path')

    def ticket_parents(self):
        if not self.parent:
            return ()
        paths = [self.materialized_path[:i] for i in xrange(0, len(self.materialized_path), 3)]
        return StaticPage.ticket_objects.filter(materialized_path__in=paths).order_by('materialized_path')

    def children(self):
        return StaticPage.objects.filter(parent=self, is_published=True)

    def ticket_children(self):
        return StaticPage.ticket_objects.filter(parent_id=self.id, is_published=True)

    class Meta:
        app_label = 'staticpages'
        ordering = ('materialized_path',)
        verbose_name = _(u'статическая страница')
        verbose_name_plural = _(u'статические страницы')

    def get_absolute_url(self):
        return reverse('info', args=[self.id])

    @classmethod
    def recalc_materialized_paths(cls, parent=None):
        if parent:
            pages = cls.admin_manager.filter(parent=parent)
        else:
            pages = cls.admin_manager.filter(parent__isnull=True)

        pages = pages.order_by('lang', 'order')

        for p in pages:
            p.save()

            cls.recalc_materialized_paths(p)


class StaticLink(models.Model):
    page = models.ForeignKey(StaticPage, verbose_name=_(u"страница"))
    title = models.CharField(verbose_name=_(u"текст ссылки"), max_length=100)
    url = models.CharField(verbose_name=_(u"урл"), max_length=100)
    order = models.PositiveIntegerField(verbose_name=_(u"порядок"))

    def __unicode__(self):
        return self.title

    class Meta:
        app_label = 'staticpages'
        ordering = ('order',)
        verbose_name = _(u'ссылка стат. страницы')
        verbose_name_plural = _(u'ссылки')


class Partner(models.Model):
    u""" Описания партнеров проекта """
    services = models.CharField(verbose_name=_(u"сервисы"), max_length=10, default='ticket', editable=False)
    L_title = L_field(verbose_name=_(u"название"), max_length=50, add_local_field=True)
    url = models.URLField(verbose_name=_(u"uRL"), blank=True, null=True)
    L_content = L_field(verbose_name=_(u"описание"), add_local_field=True,
                        field_cls=models.TextField, default=None, blank=True)
    logo = models.FileField(verbose_name=_(u'логотип'), upload_to='data/partners',
                            blank=True, null=True, default=None, editable=False)
    order = models.PositiveIntegerField(verbose_name=_(u"порядок"), default=100)
    is_published = models.BooleanField(verbose_name=_(u"опубликовано"), default=False)

    t_type = models.ForeignKey('www.TransportType', blank=True, null=True,
                               verbose_name=_(u'тип транспорта'))

    NATIONAL_VERSIONS = [(v, v) for v in ['ru', 'ua', 'kz', 'tr', 'com']]
    national_version = models.CharField(choices=NATIONAL_VERSIONS,
                                        max_length=25,
                                        default='ru',
                                        verbose_name=_(u'Национальная версия'))

    order_partner = models.ForeignKey('order.Partner', verbose_name=_(u'Партнёр в демоне'),
                                      null=True, blank=True,
                                      related_name='staticpages_partners')
    dohop_vendor = models.ForeignKey(
        'order.DohopVendor', verbose_name=_(u'Партнёр в демоне (Вендор дохопа)'),
        null=True, blank=True, related_name='staticpages_partners'
    )

    admin_manager = models.Manager()
    objects = FrontendManager()

    def __unicode__(self):
        return self.title

    def get_local_language(self):
        lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(self.national_version, 'ru')

        return lang

    class Meta:
        app_label = 'staticpages'
        verbose_name = _(u'партнер')
        verbose_name_plural = _(u'партнеры')
        ordering = ('order', 'title')

    @classmethod
    def empty_instance(cls):
        return Partner(title="", content="", url="")

    def __json__(self):
        return {
            'title': self.title,
            'url': self.url or '',
            'content': self.content,
            'logo': self.logo and self.logo.url or '',
        }
