# -*- coding: utf-8 -*-

import logging

from django.core import validators
from django.db import models
from django.utils.encoding import smart_unicode
from django.utils.html import escape
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Station, Settlement
from travel.avia.library.python.common.models_abstract.tariffs import BaseThreadTariff
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.caching import CachingManager
from travel.avia.library.python.common.utils.fields import CodeCharField, RegExpField, TrimmedCharField
from travel.avia.library.python.common.utils.models import SwapModel
from travel.avia.library.python.common.utils.regexp import RegExpSet
from travel.avia.library.python.common.xgettext.i18n import mark_gettext


log = logging.getLogger(__name__)


# Имена классов для отображения, в порядке отображения
# Закомментированные отображать не нужно
CLS_MAPPING = [
    # классы тарифов для ж/д билетов
    (mark_gettext(u'СВ'), 'suite'),
    (mark_gettext(u'купе'), 'compartment'),
    (mark_gettext(u'плацкарт'), 'platzkarte'),
    (mark_gettext(u'сидячие'), 'sitting'),
    (mark_gettext(u'общий'), 'common'),
    # (mark_gettext(u'М'), 'soft'),

    # классы тарифов для авиа билетов
    (mark_gettext(u'бизнес'), 'business'),
    (mark_gettext(u'эконом'), 'economy'),

    # фейковые классы тарифов без отображаемых названий
    (u'', 'bus'),
    (u'', 'blablacar'),
]


ORDER_CLS_MAPPING = [
    ('common', mark_gettext(u'общий')),
    ('sitting', mark_gettext(u'сидячие')),
    ('platzkarte', mark_gettext(u'плацкарт')),
    ('compartment', mark_gettext(u'купе')),
    ('compartment_min', mark_gettext(u'купе')),
    ('compartment_max', mark_gettext(u'купе эконом-класса')),
    ('suite', mark_gettext(u'люкс (СВ)')),
    ('soft', mark_gettext(u'мягкие')),

    # Самолеты
    ('economy', mark_gettext(u'эконом')),
    ('business', mark_gettext(u'бизнес')),
    ('first', mark_gettext(u'первый')),
]

CLS_NAMES = dict(ORDER_CLS_MAPPING)


class TCHDirection(models.Model):

    iata_from = CodeCharField(verbose_name=_(u"иАТА код отправления"),
                              max_length=10, null=False)

    iata_to = CodeCharField(verbose_name=_(u"иАТА код прибытия"),
                            max_length=10, null=False)

    def __unicode__(self):
        title_from = get_point_title_by_iata(self.iata_from)
        title_to = get_point_title_by_iata(self.iata_to)
        return u"TCH направление %s %s, '%s' '%s'" % (
            title_from, title_to, self.iata_from, self.iata_to)

    class Meta:
        app_label = 'tariffs'
        verbose_name = _(u"направление обслуживаемое TCH")
        verbose_name_plural = _(u"направления для TCH")
        unique_together = (('iata_from', 'iata_to'),)


def get_point_title_by_iata(iata):
    try:
        point = Station.get_by_code('iata', iata)
    except Station.DoesNotExist:
        point = None
    if point is None:
        try:
            point = Settlement.objects.get(iata=iata)
        except Settlement.DoesNotExist:
            point = None

    return (point.title if point else iata)


class UFSDepthOfSales(models.Model):
    number = RegExpField(verbose_name=_(u'номер рейса'),
                         help_text=_(u'номер рейса можно задавать регулярым выражением'),
                         max_length=255, unique=True, null=False, blank=False)

    days_ahead = models.IntegerField(verbose_name=_(u'глубина продажи'),
                                     help_text=_(u'количество дней вперед на которые разрешена продажа'),
                                     default=60, null=False, blank=False)

    class Meta:
        app_label = 'tariffs'
        verbose_name = _(u"Глубина продажи билетов для рейса UFS")
        verbose_name_plural = _(u"Глубина продажи билетов для рейсов UFS")

    def __unicode__(self):
        return u"'{number}' - {days_ahead} дней".format(number=self.number, days_ahead=self.days_ahead)


class UFSDirectionDaysAhead(models.Model):
    """
    Направления, для которых глубина продажи отличается от стандартной.
    Заполняется скриптом, исходя из данных UFSDepthOfSales
    """

    from_express_code = CodeCharField(max_length=50, null=False, blank=False)
    to_express_code = CodeCharField(max_length=50, null=False, blank=False)
    days_ahead = models.IntegerField(null=False, blank=False)

    class Meta:
        app_label = 'tariffs'
        unique_together = (('from_express_code', 'to_express_code'),)


class SuburbanTariff(models.Model):
    code = TrimmedCharField(verbose_name=_(u'код'), max_length=10, primary_key=True)
    tariff = models.FloatField(verbose_name=_(u'цена'))

    objects = CachingManager(keys=['pk', 'code'])

    @classmethod
    def get_tariff(cls, code):
        try:
            return cls.objects.get(code=code).tariff
        except SuburbanTariff.DoesNotExist:

            log.error("Tariff with code %s not found" % code)

            return None

    class Meta:
        verbose_name = _(u'тариф пригородного транспорта')
        verbose_name_plural = _(u'тарифы пригородного транспорта')
        app_label = 'www'

    def __unicode__(self):
        return self.code


class AeroexTariff(models.Model):
    u"""Тарифы аэрокспрессов и электричек станция-станция"""

    station_from = models.ForeignKey('Station', verbose_name=_(u'первая станция'), related_name='aeroex_tariffs_from')
    station_to = models.ForeignKey('Station', verbose_name=_(u'последняя станция'), related_name='aeroex_tariffs_to')
    tariff = models.FloatField(u'Цена')
    suburban_search = models.BooleanField(u'В поиске электричек', default=True)
    reverse = models.BooleanField(verbose_name=_(u"туда-обратно"), blank=True, default=True)
    type = models.ForeignKey('TariffType', null=False, default=1, blank=False)
    precalc = models.BooleanField(verbose_name=_(u'Зонный'), blank=True, default=False)
    currency = models.CharField(_(u'Валюта'), max_length=10, null=True, default=None, blank=True)

    @classmethod
    def zone_tariff(cls, tariff):
        t = cls()

        t.tariff = tariff
        t.type = TariffType.objects.get(code='etrain')

        return t

    def __unicode__(self):
        return u'<%s %s: %s>' % (self.__class__.__name__, self.type.code, self.tariff)

    class Meta:
        unique_together = ('station_from', 'station_to', 'type', 'precalc')
        verbose_name = _(u'тариф пригородного транспорта точка-точка')
        verbose_name_plural = _(u'тарифы пригородного транспорта точка-точка')
        app_label = 'www'


class ThreadTariff(BaseThreadTariff):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.tariffs'

    class Meta(BaseThreadTariff.Meta):
        app_label = 'www'


class NotTariffTrain(models.Model):
    u"""Поезда, которые не тарифицирует Экспресс

    Номер это регулярное выражение
    """
    number = RegExpField(verbose_name=_(u'номер рейса'), max_length=255, unique=True,
                         blank=False, help_text=_(u'номер рейса можно задавать регулярым выражением'))

    class Meta:
        verbose_name = _(u"поезд, который не тарифицирует Экспресс")
        verbose_name_plural = _(u"поезда, которые не тарифицирует Экспресс")
        app_label = 'www'

    _number_set = []

    @classmethod
    def precache_numbers(cls):
        cls._number_set = RegExpSet([ntt.number for ntt in cls.objects.all()])

    @classmethod
    def number_set(cls, not_cache=False):
        if not_cache:
            return RegExpSet([ntt.number for ntt in cls.objects.all()])

        return cls._number_set


class TariffType(models.Model):
    """
    Типы тарифов
    """

    DEFAULT_ID = 1
    EXPRESS_ID = 2

    USUAL_CATEGORY = 'usual'
    SEASON_TICKET_CATEGORY = 'season_ticket'
    SPECIAL_CATEGORY = 'special'

    L_title = L_field(_(u"название типа"), add_local_field=True, max_length=100)
    link = TrimmedCharField(_(u'Ссылка'), max_length=255, blank=True)
    category = models.CharField(
        verbose_name=_(u"категория"),
        choices=(
            (SEASON_TICKET_CATEGORY, _(u'Абонемент')),
            (USUAL_CATEGORY, _(u'Обычный')),
            (SPECIAL_CATEGORY, _(u'Специальный')),
        ),
        max_length=20, null=False, blank=False,
        default=USUAL_CATEGORY,
    )
    code = CodeCharField(verbose_name=_(u"код"), max_length=20,
                         null=False, blank=False, unique=True)
    order = models.IntegerField(verbose_name=_(u"порядок показа"),
                                default=0)

    tariff_groups = models.ManyToManyField('www.TariffGroup', verbose_name=_(u'Группы тарифов'))

    objects = CachingManager(keys=['pk', 'code'])

    TANKER_L_FIELDS = ['title']

    def __unicode__(self):
        return self.L_title()

    @property
    def title_with_link(self):
        if self.link:
            return '<a href="%s">%s</a>' % (escape(self.link), self.L_title())

        return self.L_title()

    class Meta:
        ordering = ('order',)
        verbose_name = _(u"Тип тарифа")
        verbose_name_plural = _(u"Типы тарифов")
        app_label = 'www'


class TariffGroup(models.Model):
    title = models.CharField(_(u'Название группы тарифа'), max_length=100, blank=False)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u'Группа тарифов')
        verbose_name_plural = _(u'Группы тарифов')
        app_label = 'www'


SETTINGS_TYPES = {
    'str': (_(u"Строка"), unicode),
    'int': (_(u"Целое"), int),
    'float': (_(u"Число с плавающей точкой"), float),
}


class Setting(models.Model):
    name = TrimmedCharField(_(u"Название"), max_length=255, blank=False, null=False)
    code = CodeCharField(
        _(u"Код"), max_length=255, blank=False, null=False,
        validators=[
            validators.RegexValidator(
                u'(?u)^[A-Z][A-Z_0-9]*$',
                _(u"Код должен начинаться с буквы, допустимые символы [A-Z_0-9]")
            )
        ]
    )
    type = models.CharField(
        _(u"Тип"), max_length=20,
        choices=list(  # list(...) не засоряет неймспейс в отличие от [...]
            (type_code, type_text)
            for type_code, (type_text, _type) in SETTINGS_TYPES.items()
        ),
        blank=False, null=False, default='str'
    )
    value = models.CharField(_(u'Значение'), max_length=1024,
                             blank=True, null=False, default="")
    help_text = models.TextField(_(u'Подсказка'), max_length=1024,
                                 blank=True, null=True, default="")

    def get_value(self):
        return SETTINGS_TYPES[self.type][1](self.value)

    def clean(self):
        try:
            self.get_value()
        except Exception, e:
            raise validators.ValidationError(
                _(u"Не удалось преобразовать %s к типу %s: %s") %
                (self.value, self.get_type_display(), smart_unicode(e)))

    objects = CachingManager(keys=['code'])

    @classmethod
    def get(cls, code):
        return cls.objects.get(code=code).get_value()

    @classmethod
    def get_settings_as_class(cls):
        class S:
            pass

        s = S()

        for code, value in cls.get_settings_as_dict().items():
            setattr(s, code, value)

        return s

    @classmethod
    def get_settings_as_dict(cls):
        d = {}

        for s in cls.objects.all():
            d[s.code] = s.get_value()

        return d

    class Meta:
        app_label = 'www'
        verbose_name = _(u"Настройка")
        verbose_name_plural = _(u"Настройки")
