# -*- coding: utf-8 -*-

import warnings

from django.conf import settings
from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _, get_language

from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarningLite
from travel.avia.library.python.common.utils.environment import get_request


class FrontendManager(models.Manager):
    def get_queryset(self):
        lang = get_language() or 'ru'
        request = get_request()
        national_version = getattr(request, 'NATIONAL_VERSION', None) or 'ru'

        warnings.warn('Use Teaser.get_objects_for_request, was marked at 2015-02-24', RaspDeprecationWarningLite)

        return (
            super(FrontendManager, self)
            .get_queryset()
            .filter(lang=lang)
            .filter(Q(national_version__isnull=True) | Q(national_version=national_version))
        )


class Page(models.Model):
    """ Логическая страница (RASP-2838) """
    code = models.CharField(verbose_name=_(u"код страницы"), max_length=100, unique=True)
    title = models.CharField(verbose_name=_(u"наименование страницы"), max_length=250)

    class Meta:
        verbose_name = _(u'логическая страница')
        verbose_name_plural = _(u'логические страницы')
        app_label = 'www'

    def __unicode__(self):
        return self.title


class Teaser(models.Model):
    """ Тизер (RASP-2838) """
    title = models.CharField(verbose_name=_(u"наименование"), max_length=255)
    content = models.TextField(verbose_name=_(u"текст"))
    mobile_content = models.TextField(verbose_name=_(u"мобильный"), default="", blank=True)

    url = models.URLField(verbose_name=_(u"url"), null=True, blank=True)
    image = models.ImageField(verbose_name=_(u"картинка"), null=True, blank=True, upload_to="data/teasers")
    template = models.CharField(verbose_name=_(u"шаблон"), max_length=20, null=True, blank=True)
    is_active_rasp = models.BooleanField(verbose_name=_(u"активен в ПМ"), default=False)
    is_active_ticket = models.BooleanField(verbose_name=_(u"активен в КБ"), default=False)
    is_active_export = models.BooleanField(verbose_name=_(u'Активен в приложениях (export)'), default=False)
    MODES = (('normal', _(u'Обычный')),
             ('ahtung', _(u'Внимание')),
             ('special', _(u'Спецпредложение')),
             ('banner', _(u'Баннер')),
             )
    mode = models.CharField(verbose_name=_(u"тип"), choices=MODES, max_length=20, default='normal')
    importance = models.PositiveIntegerField(verbose_name=_(u"важность"), default=0,
                                             help_text=_(u"чем больше цифра, тем важнее"))
    date_start = models.DateField(verbose_name=_(u"дата начала кампании"))
    date_finish = models.DateField(verbose_name=_(u"дата окончания кампании"))

    companies = models.ManyToManyField('www.Company', verbose_name=_(u'перевозчики'), blank=True, default=None)
    pages = models.ManyToManyField(Page, verbose_name=_(u'страницы'), blank=True, default=None)
    countries = models.ManyToManyField('www.Country', verbose_name=_(u'страны'), blank=True, default=None)
    settlements = models.ManyToManyField('www.Settlement', verbose_name=_(u'города'), blank=True, default=None, related_name="settlements")
    stations = models.ManyToManyField('www.Station', verbose_name=_(u'станции'), blank=True, default=None, related_name="stations")
    directions = models.ManyToManyField('www.ExternalDirection', verbose_name=_(u"направления"), blank=True, default=None, related_name="directions")
    threads = models.ManyToManyField('www.RThread', verbose_name=_(u"нитки"), blank=True, default=None, related_name="threads")

    lang = models.CharField(choices=settings.LANGUAGES, max_length=25, default='ru', verbose_name=_(u'Язык'))

    national_version = models.CharField(
        choices=[('ru', 'ru'), ('ua', 'ua'), ('tr', 'tr')],
        max_length=25,
        null=True,
        blank=False,
        default='ru',
        verbose_name=_(u'Национальная версия')
    )

    # FIXME: смотри RaspDeprecationWarning для FrontendManager
    base_objects = models.Manager()
    objects = FrontendManager()

    @classmethod
    def get_objects(cls, national_version=None, language=None):
        language = language or get_language() or 'ru'
        if not national_version:
            request = get_request()
            national_version = (getattr(request, 'NATIONAL_VERSION', None) if request else None) or 'ru'

        return (
            cls.base_objects
            .filter(lang=language)
            .filter(Q(national_version__isnull=True) | Q(national_version=national_version))
        )

    @classmethod
    def get_objects_for_request(cls, request=None, lang=None):
        request = request or get_request()
        national_version = getattr(request, 'NATIONAL_VERSION', None) or 'ru'

        return cls.get_objects(national_version, lang)

    class Meta:
        verbose_name = _(u'тизер')
        verbose_name_plural = _(u'тизеры')
        app_label = 'www'

    def __json__(self):
        return {
            'title': self.title,
            'content': self.content,
            'url': self.url,
            'image': self.image and self.image.url or '',
            'template': self.template,
        }

    def __unicode__(self):
        return self.title

    def __repr__(self):
        return (u"""Teaser:
        mode: %s
        title: %s
        is_active_rasp: %s
        is_active_ticket: %s
        """ % (self.mode, self.title, self.is_active_rasp, self.is_active_ticket)).encode('utf8')


Teaser.companies.through._meta.verbose_name_plural = _(u'перевозчики тизера')
Teaser.pages.through._meta.verbose_name_plural = _(u'страницы тизера')
Teaser.countries.through._meta.verbose_name_plural = _(u'страны тизера')
Teaser.settlements.through._meta.verbose_name_plural = _(u'города тизера')
Teaser.stations.through._meta.verbose_name_plural = _(u'станции тизера')
Teaser.directions.through._meta.verbose_name_plural = _(u'направления тизера')
Teaser.threads.through._meta.verbose_name_plural = _(u'нитки тизера')
