# -*- coding: utf-8 -*-
from django.db import models
from django.utils.translation import ugettext_lazy as _


class TranslatedTitle(models.Model):
    class Meta:
        app_label = 'www'
        db_table = 'www_translated_title'
        verbose_name = _(u'название для перевода')
        verbose_name_plural = _(u'название для перевода')

    ru_nominative = models.CharField(max_length=255, null=True)
    ru_genitive = models.CharField(max_length=255, null=True)
    ru_dative = models.CharField(max_length=255, null=True)
    ru_accusative = models.CharField(max_length=255, null=True)
    ru_locative = models.CharField(max_length=255, null=True)

    en_nominative = models.CharField(max_length=255, null=True)
    de_nominative = models.CharField(max_length=255, null=True)

    tr_nominative = models.CharField(max_length=255, null=True)

    uk_nominative = models.CharField(max_length=255, null=True)
    uk_accusative = models.CharField(max_length=255, null=True)

    def translation(self, lang='ru', case='nominative'):
        return getattr(self, '{}_{}'.format(lang, case), None)


class TranslatedText(models.Model):
    class Meta:
        app_label = 'www'
        db_table = 'www_translated_text'
        verbose_name = _(u'текст для перевода')
        verbose_name_plural = _(u'текст для перевода')

    ru = models.TextField(null=True)
    en = models.TextField(null=True)
    de = models.TextField(null=True)
    tr = models.TextField(null=True)
    uk = models.TextField(null=True)

    def translation(self, lang='ru'):
        return getattr(self, lang)


class TranslationMixin(object):
    OLD_LANG_CASES = {
        'ru': ['nominative', 'genitive', 'dative', 'accusative', 'locative'],
        'en': ['nominative'],
        'tr': ['nominative'],
        'uk': ['nominative', 'accusative'],
    }
    TRANSLATION_MODELS = (TranslatedTitle, TranslatedText)

    def save(self, **kwargs):
        for field in self._meta.fields:
            if self._is_translation_field(field):
                old_translations = self._get_old_translation_fields(field)
                translations = {}

                if issubclass(field.related_model, TranslatedTitle):
                    for lang, cases in self.OLD_LANG_CASES.iteritems():
                        for case in cases:
                            field_name = '%s_%s' % (lang, case)
                            translations[field_name] = old_translations.get(field_name, old_translations.get(lang))
                else:
                    translations = old_translations

                try:
                    object_pk = getattr(self, self._meta.pk.name)
                    object_in_db = self.__class__.objects.get(pk=object_pk)
                    translation_id = getattr(object_in_db, field.attname)
                    field.related_model.objects.filter(pk=translation_id).update(**translations)
                    setattr(self, field.attname, translation_id)
                except (self.__class__.DoesNotExist, field.related_model.DoesNotExist):
                    translation = field.related_model.objects.create(**translations)
                    setattr(self, field.attname, translation.id)

        return super(TranslationMixin, self).save(**kwargs)

    def _is_translation_field(self, field):
        return bool(
            field.related_model and
            issubclass(field.related_model, self.TRANSLATION_MODELS) and
            not getattr(self, field.attname)
        )

    def _gen_possible_suffixes(self, field_suffix):
        all_suffixes = []

        for lang, cases in self.OLD_LANG_CASES.iteritems():
            lang_suffix = '%s_%s' % (field_suffix, lang)
            all_suffixes.append(lang_suffix)
            for case in cases:
                all_suffixes.append('%s_%s' % (lang_suffix, case))

        return set(all_suffixes)

    def _get_old_translation_fields(self, new_L_field):
        assert new_L_field.attname.startswith('new_L_'), 'New Schema translation fields must be started with new_L_'
        field_suffix = new_L_field.name[6:]
        possible_suffixes = self._gen_possible_suffixes(field_suffix)
        old_translations = {}

        for field in self._meta.fields:
            if field.attname in possible_suffixes:
                old_translations[field.attname[len(field_suffix)+1:]] = getattr(self, field.attname)

        return old_translations
