# -*- coding: utf-8 -*-

from __future__ import absolute_import

from django.core.validators import RegexValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _, get_language

from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.caching import CachingManager, cache_until_switch
from travel.avia.library.python.common.utils.fields import CodeCharField, TrimmedTextField, TrimmedCharField
from travel.avia.library.python.common.utils.files.validators import FileSizeValidator, SvgValidator
from travel.avia.library.python.common.xgettext.i18n import mark_gettext


TRANSPORT_TYPES = {
    'express': mark_gettext(u'экспресс'),
    'aeroexpress': mark_gettext(u'аэроэкспресс'),
    'plane': mark_gettext(u'самолёт'),
    'train': mark_gettext(u'поезд'),
    'bus': mark_gettext(u'автобус'),
    'suburban': mark_gettext(u'электричка'),
    'sea': mark_gettext(u'морской транспорт'),
    'river': mark_gettext(u'речной транспорт'),
    'water': mark_gettext(u'водный транспорт')
}


class TransportType(models.Model):
    """ Тип транспорта: train, plane, .. """

    TANKER_L_FIELDS = ['title', 'title_plural', 'station_name', 'station_name_plural', 'title_partner_page']

    code_validator = RegexValidator(
        ur'^[0-9a-zA-Z-]+$',
        _(u'В коде типа транспорта могут учавствовать только английские буквы цифры и знак минус')
    )

    code = models.CharField(verbose_name=_(u'код'), max_length=50, unique=True, validators=[code_validator])
    L_title = L_field(_(u'наименование'), base_field_critical=True, base_lang='ru')
    L_title_plural = L_field(_(u'наименование множественное'), base_lang='ru')
    L_station_name = L_field(_(u'название станции'), base_lang='ru',
                             help_text=_(u"например, для поездов - 'вокзал'."))
    L_station_name_plural = L_field(_(u'название станций'), base_lang='ru',
                                    help_text=_(u"например, для поездов - 'вокзалы'."))
    L_title_partner_page = L_field(_(u'название для страницы партнеров'), base_lang='ru')

    objects = CachingManager(keys=['pk', 'code'])

    # Константы, чтобы не хардкодить в коде числа
    TRAIN_ID = 1
    PLANE_ID = 2
    BUS_ID = 3
    RIVER_ID = 4
    SEA_ID = 5
    SUBURBAN_ID = 6
    HELICOPTER_ID = 7
    URBAN_ID = 8
    PSEUDO_GORTRANS = 9
    BLABLACAR = 10
    WATER_ID = 11

    WATER_TTYPE_IDS = [RIVER_ID, SEA_ID, WATER_ID]
    WATER_TTYPE_CODES = ['river', 'sea', 'water']

    def __repr__(self):
        return '{}: {}'.format(self.__class__.__name__, self.code)

    def __unicode__(self):
        return self.L_title()

    @property
    def name(self):
        return self.L_title()

    @classmethod
    @cache_until_switch
    def get_train_type(cls):
        return cls.objects.get(id=cls.TRAIN_ID)

    @classmethod
    @cache_until_switch
    def get_plane_type(cls):
        return cls.objects.get(id=cls.PLANE_ID)

    type = 'transporttype'

    class Meta:
        verbose_name = _(u'тип транспорта')
        verbose_name_plural = _(u'типы транспорта')
        app_label = 'www'
        db_table = 'www_transporttype'


class TransportModel(models.Model):
    """ Модель транспортного средства: Як-40, Ту-154, .. """

    title = models.CharField(verbose_name=_(u'наименование'), max_length=100)
    title_en = models.CharField(verbose_name=_(u'англ. наименование'),
                                default=u'', blank=True, max_length=100)
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'тип'))
    t_subtype = models.ForeignKey('www.TransportSubtype', verbose_name=_(u'подтип'), default=None, blank=True, null=True)
    TYPES = (
        (None, _(u'')),
        ('standard', _(u'Стандартный')),
        ('wire', _(u'Широкофюзеляжный')),
        ('tight', _(u'Узкий салон')),
    )
    plane_body_type = models.CharField(
        verbose_name=_(u'тип фюзеляжности самолёта'),
        max_length=10, null=True, blank=True, choices=TYPES)
    is_propeller_flight = models.BooleanField(
        verbose_name=_(u'винтовой самолёт'), default=False)

    template_page = models.CharField(verbose_name=_(u'имя шаблона'),
                                     max_length=100, null=True, blank=True)
    code = CodeCharField(verbose_name=_(u'код'), max_length=10,
                         null=True, blank=True, default=None, unique=True)
    code_en = CodeCharField(verbose_name=_(u'код Английский'), max_length=10,
                            null=True, blank=True, default=None, unique=True)
    producer = models.ForeignKey('www.VehicleProducer', verbose_name=_(u'производитель'),
                                 null=True, blank=True, default=None)
    img = models.ImageField(verbose_name=_(u'фото ТС'), upload_to='data/transportmodel/img',
                            null=True, blank=True, default=None)
    background = models.ImageField(verbose_name=_(u'картинка для фона'),
                                   upload_to='data/transportmodel/background',
                                   null=True, blank=True, default=None)
    ttx = models.TextField(verbose_name=_(u'тТХ'),
                           null=True, blank=True, default=None)
    descr = models.TextField(verbose_name=_(u'описание'),
                             null=True, blank=True, default=None)
    is_cargo = models.BooleanField(verbose_name=_(u"грузовой"), blank=True,
                                   default=False)

    objects = CachingManager()

    def L_title(self, lang=None):
        lang = lang or get_language()

        return self.title if lang == 'ru' else self.title_en

    def __unicode__(self):
        return self.L_title()

    type = "transportmodel"

    class Meta:
        verbose_name = _(u'модель транспорта')
        verbose_name_plural = _(u'модели транспорта')
        app_label = 'www'
        db_table = 'www_transportmodel'


def transportsubtypeiconfield_generate_filename(t_subtype, filename):
    return 'data/transportsubtype/icon/{}.svg'.format(t_subtype.code)


class TransportSubtypeIconField(models.FileField):
    def __init__(self, *args, **kwargs):
        kwargs.setdefault('upload_to', transportsubtypeiconfield_generate_filename)
        super(TransportSubtypeIconField, self).__init__(*args, **kwargs)


class TransportSubtype(models.Model):
    """Подтип транспорта"""
    MAX_ICON_SIZE_IN_BYTES = 1024 * 1024
    TANKER_L_FIELDS = ['title', 'title_plural', 'station_name', 'station_name_plural', 'title_partner_page']

    file_size_validator = FileSizeValidator(size_limit_in_bytes=MAX_ICON_SIZE_IN_BYTES)
    svg_validator = SvgValidator()
    code_validator = RegexValidator(
        ur'^[0-9a-zA-Z-]+$',
        _(u'В коде подтипа транспорта могут учавствовать только английские буквы цифры и знак минус')
    )

    code = models.CharField(verbose_name=_(u'код'), max_length=50, unique=True, validators=[code_validator])
    L_title = L_field(_(u'наименование'), base_field_critical=True, base_lang='ru')
    L_title_plural = L_field(_(u'наименование множественное'), base_lang='ru')
    L_station_name = L_field(_(u'название станции'), base_lang='ru',
                             help_text=_(u"например, для поездов - 'вокзал'."))
    L_station_name_plural = L_field(_(u'название станций'),
                                    base_lang='ru',
                                    help_text=_(u"например, для поездов - 'вокзалы'."))
    L_title_partner_page = L_field(_(u'название для станицы партнеров'), base_lang='ru')
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'),
                               null=False, blank=False, related_name='subtype')
    color = models.ForeignKey('www.TransportSubtypeColor', verbose_name=_(u'Цвет'), null=True, blank=True, default=None)
    icon = TransportSubtypeIconField(
        _(u'Иконка (svg)'), validators=[file_size_validator, svg_validator],
        null=True, blank=True, default=None
    )
    comment = TrimmedTextField(_(u'Комментарий'), null=False, blank=True, default='')

    objects = CachingManager(keys=['pk', 'code'])

    TRAIN_ID = 1
    PLANE_ID = 2
    BUS_ID = 3
    RIVER_ID = 4
    SEA_ID = 5
    SUBURBAN_ID = 6
    HELICOPTER_ID = 7

    def __unicode__(self):
        return self.L_title()

    class Meta:
        verbose_name = _(u'подтип транспорта')
        verbose_name_plural = _(u'подтипы транспорта')
        app_label = 'www'
        db_table = 'www_transportsubtype'


class TransportSubtypeColor(models.Model):
    code_validator = RegexValidator(
        ur'^[0-9a-zA-Z-]+$',
        _(u'В коде цвета могут учавствовать только английские буквы цифры и знак минус')
    )
    color_validator = RegexValidator(
        ur'^#[0-9a-fA-F]{6}$',
        _(u'Цвет должен быть задан в формате #123456')
    )

    code = models.CharField(verbose_name=_(u'Код'), max_length=50, null=False, blank=False, unique=True,
                            validators=[code_validator])
    color = TrimmedCharField(_(u'Цвет'), max_length=10, null=False, blank=False, validators=[color_validator])

    def __unicode__(self):
        return u'{} {}'.format(self.code, self.color)

    class Meta:
        verbose_name = _(u'Цвет подтипа транспорта')
        verbose_name_plural = _(u'Цвета подтипов транспорта')
        app_label = 'www'


class TrainPseudoStationMap(models.Model):
    number = TrimmedCharField(verbose_name=_(u'номер'), max_length=50)
    pseudo_station = models.ForeignKey('www.Station', verbose_name=_(u'псевдостанция'), related_name='pseudo_pseudo_station_set')
    station = models.ForeignKey('www.Station', verbose_name=_(u'станция'), related_name='pseudo_station_set')

    class Meta:
        verbose_name = _(u'замена псевдостанции')
        verbose_name_plural = _(u'замены псевдостанций')
        app_label = 'www'
        unique_together = (('number', 'station'),)
